#!/bin/zsh
# Format: UTF8, Linux EOL

# Script, das eine Bibliothek von Scripts aus dem Internet lädt, 
# wenn sie lokal veraltet sind

# 001, 250610, Tom
# 002, 250610, Tom
#	Setzt +x für Scripts
#	Neu mit --force
# 003, 250611, Tom
#	Etwas bereinigt
# 004, 250627, Tom
#	neu zsh

# Ex
# /Library/Scripts/SSF/NinjaOne/Update-NinjaOne-Scripts.sh -force


### Root forcieren
if [[ "$EUID" -ne 0 ]]; then
  echo "Dieses Script benötigt root-Rechte. Es wird erneut mit sudo ausgeführt..."

  # Das Script mit sudo neu starten.
  # "$0" ist der Pfad zum aktuellen Script.
  # "$@" sind alle Argumente, die dem Script ursprünglich übergeben wurden.
  # So wird sichergestellt, dass z.B. './script.sh --arg1 value1' zu
  # 'sudo ./script.sh --arg1 value1' wird.
  sudo zsh "$0" "$@"

  # Den ursprünglichen (nicht-root) Prozess beenden.
  # $? enthält den Exit-Code des sudo-Befehls.
  exit $?
fi

# Ab hier Root


### Init

## Optionen verarbeiten
FORCE_UPDATE=false

for arg in "$@"; do
	case "$arg" in
		-force|--force)
			FORCE_UPDATE=true
			;;
	esac
done



### Config

# Das LogFile dieses Scripts 
# /Users/Shared    > r+w, für alle User
LOG_FILE="/Users/Shared/NinjaOne/Update-NinjaOne-Scripts.sh.log"
LOG_DIR=$(dirname "$LOG_FILE")

# /Library                      > Ressourcen für alle User
# /Library/Scripts/SSF/NinjaOne > Nur Admin-User
SCRIPT_DIR="/Library/Scripts/SSF/NinjaOne"
VERSION_FILE="$SCRIPT_DIR/Version.txt"
TEMP_DIR_PREFIX="/tmp/NinjaOne-"
BASE_URL="https://www.akros.ch/it/MacOS"
# Zsh braucht kein \{6\}, sondern {6}
ZIP_PATTERN="Admin-Scripts-[0-9]{6}\\.zip"  



### Funcs

# Logging-Funktion mit Einrückung
# Ex
# log 4 "This is an indented log message."
log() {
	local indent=""
	local spaces=$1
	shift

	if [[ "$spaces" -eq 0 ]]; then
		echo ""
	fi

	for ((i = 0; i < spaces; i++)); do
		indent+="  "
	done

	# echo "$(date '+%Y-%m-%d %H:%M:%S') - ${indent}$1" | tee -a "$LOG_FILE"
	echo "${indent}$1"
	echo "$(date '+%Y-%m-%d %H:%M:%S') - ${indent}$1" >> "$LOG_FILE"
}


### Main-Funktion
main() {
	echo ""
	echo ""

	# Schritt 1: Sicherstellen, dass der Zielordner existiert
	mkdir -p "$SCRIPT_DIR"

	# Schritt 2: Finden der neuesten Zip-Datei
	log 0 "Suche das neuste Script Zip-File"
	FILE_LIST=$(curl -s $BASE_URL/ | grep -oE "$ZIP_PATTERN")

	if [[ -z "$FILE_LIST" ]]; then
	  log 1 "Kein zip file gefunden:"
	  log 1 "URL    : $BASE_URL"
	  log 1 "Pattern: $ZIP_PATTERN"
	  return 1
	fi

	LATEST_FILE=$(echo "$FILE_LIST" | sort | tail -n 1)
	LATEST_VERSION=$(echo "$LATEST_FILE" | grep -oE '[0-9]{6}')

	log 1 "Neustes Zip-File gefunden   : $LATEST_FILE"
	log 1 "Neuste Version              : $LATEST_VERSION"

	# Schritt 2: Überprüfen der aktuellen Version?
	if [[ "$FORCE_UPDATE" == true ]]; then
		log 1 "FORCE-Option erkannt: Versionsprüfung wird übersprungen"
	else
		if [[ -f "$VERSION_FILE" ]]; then
			CURRENT_VERSION=$(head -n 1 "$VERSION_FILE")
			log 1 "Aktuell installierte Version: $CURRENT_VERSION"
			if [[ "$LATEST_VERSION" -le "$CURRENT_VERSION" ]]; then
				log 2 "> Kein Update nötig"
				log 2 "> Forcieren mit:"
				log 3 "-force"
				return 0
			else
				log 2 "> Update nötig!"
			fi
		else
			log 2 "Version-File existiert nicht: $VERSION_FILE"
			log 3 "> Installiere Scripts"
		fi
	fi	

	# Schritt 3: Erstellen des temporären Verzeichnisses
	log 1 "Erstelle Temp-Dir:"
	TIMESTAMP=$(date '+%Y%m%d%H%M%S')
	TEMP_DIR="${TEMP_DIR_PREFIX}${TIMESTAMP}"
	mkdir -p "$TEMP_DIR"
	log 2 "$TEMP_DIR"

	# Schritt 4: Herunterladen der neuesten Zip-Datei
	LATEST_URL="${BASE_URL}/${LATEST_FILE}"
	log 1 "Downloading der neusten Zip-Datei:"
	log 2 "$LATEST_URL"
	curl -o "$TEMP_DIR/$LATEST_FILE" "$LATEST_URL"

	# Schritt 5: Extrahieren und Ersetzen der Skripte
	log 1 "Extrahiere die Scripts"
	unzip -o "$TEMP_DIR/$LATEST_FILE" -d "$SCRIPT_DIR"

	# Die Versions-Info aktualisieren
	echo "$LATEST_VERSION" > "$VERSION_FILE"
	log 1 "Scripts aktualisiert auf: $LATEST_VERSION"

	# Schritt 6: Löschen des temporären Verzeichnisses
	log 1 "Lösche Temp Dir"
	rm -rf "$TEMP_DIR"
	log 1 "Update installiert"

	# Schritt 7: Setze Ausführbarkeit für .sh und .ps1 Dateien
	log 1 "Setze chmod +x für *.sh und *.ps1 Dateien im Zielverzeichnis:"
	find "$SCRIPT_DIR" \( -name "*.sh" -o -name "*.ps1" \) -exec chmod +x {} \;
	log 2 "Berechtigungen gesetzt"
}



### Prepare

# Allenfalls Logdir erstellen
mkdir -p "$LOG_DIR"



### Main
# Start des Skripts
main


