#!/bin/zsh
# Format: UTF8, Linux EOL

# Script bei Fehler abbrechen
set -e


# Erzeugt oder löscht einen MacOS Launch Agent Eintrag,
# um beim login ein Script zu starten

# !Ex
#	# Eintrag erzeugen, Script wird bei jedem Login ausgeführt
#	/Library/Scripts/SSF/NinjaOne/Set-Login-Script.sh add localadmin /Library/Scripts/SSF/NinjaOne/Get-Hostname.ps1
#
#	# Eintrag erzeugen, RunOnce
#	/Library/Scripts/SSF/NinjaOne/Set-Login-Script.sh add localadmin runonce /Library/Scripts/SSF/NinjaOne/Get-Hostname.ps1
#
#	/Library/Scripts/SSF/NinjaOne/Set-Login-Script.sh add localadmin /Library/Scripts/SSF/NinjaOne/Set-Local-MacUser-Password.ps1
#
#	# Eintrag löschen
#	/Library/Scripts/SSF/NinjaOne/Set-Login-Script.sh remove localadmin /Library/Scripts/SSF/NinjaOne/Get-Hostname.ps1 
#
#	/Library/Scripts/SSF/NinjaOne/Set-Login-Script.sh remove localadmin /Library/Scripts/SSF/NinjaOne/Set-Local-MacUser-Password.ps1
#   # Alle löschen
#	/Library/Scripts/SSF/NinjaOne/Set-Login-Script.sh remove localadmin


# 001, 250611, Tom
# 002, 250616, Tom
# Bereinigt:
#	check_and_fix_ps_script
# 003, 250627, Tom
#	Neu zsh
# 004, 250723, Tom
#	Parameter-Reihenfolge geändert:
#     Bisher:
#       1: USERNAME
#       2: POWERSHELL_SCRIPT_PATH
#       3: ACTION
#          - add
#          - remove
#       4: EXECUTION_MODE
#          - persistent
#          - runonce
#     Neu:
#       1: ACTION
#          - add
#          - remove
#       2: USERNAME
#       3: EXECUTION_MODE
#          - persistent (=Default)
#          - runonce
#       4: POWERSHELL_SCRIPT_PATH



### Root forcieren
# --- Anfang des root-Checks ---

# Prüfen, ob die Effektive User ID (EUID) nicht 0 ist (0 ist die ID von root).
if [ "$EUID" -ne 0 ]; then
  # Meldung an den Benutzer ausgeben.
  echo "Dieses Script benötigt root-Rechte. Es wird erneut mit sudo ausgeführt..."

  # Das Script mit sudo neu starten.
  # "$0" ist der Pfad zum aktuellen Script.
  # "$@" sind alle Argumente, die dem Script ursprünglich übergeben wurden.
  # So wird sichergestellt, dass z.B. './script.sh --arg1 value1' zu
  # 'sudo ./script.sh --arg1 value1' wird.
  sudo zsh "$0" "$@"

  # Den ursprünglichen (nicht-root) Prozess beenden.
  # $? enthält den Exit-Code des sudo-Befehls.
  exit $?
fi

# --- Ende des root-Checks ---

# Ab hier wird der Code garantiert mit root-Rechten ausgeführt



### Config

# --- Variablen ---
PWSH_PATH="/usr/local/bin/pwsh"

# Bezeichner für den Launch Agent (sollte eindeutig sein)
LAUNCH_AGENT_LABEL="com.noser.ssf.pwsh.autostart"

# Basispfad für das Wrapper-Skript
WRAPPER_SCRIPT_DIR="/Library/Scripts/SSF/NinjaOne"
WRAPPER_SCRIPT_NAME="run-ps-script-wrapper.sh"
WRAPPER_SCRIPT_PATH="${WRAPPER_SCRIPT_DIR}/${WRAPPER_SCRIPT_NAME}"



### Funcs


# Funktion zur Anzeige der Nutzungsinformationen
usage() {
    echo "Nutzung: $0 remove <username>"
    echo "  Löscht für den User alle Autostart-Einträge"
    echo ""
    echo "Nutzung: $0 remove <username> <powershell_script_path>"
    echo "  Löscht für den User den Autostart-Eintrag mit dem angegebenen PowerShell-Skript"
    echo ""
    echo "Nutzung: $0 add <username> [persistent|runonce] <powershell_script_path>"
    echo "  Erstellt für den User einen Autostart-Eintrag"
    echo "  <username>               : Der Benutzername, für den der Autostart-Eintrag erstellt/gelöscht wird."
    echo "  <powershell_script_path> : Der vollständige Pfad zum PowerShell-Skript, das ausgeführt werden soll."
    echo "  <add|remove>             : 'add' um den Eintrag zu erstellen, 'remove' um ihn zu löschen."
    echo "  [persistent|runonce]     : Optionaler Parameter für den Ausführungsmodus (Standard: persistent)."
    echo "                             'persistent': Das Skript wird bei jedem Login ausgeführt."
    echo "                             'runonce'   : Das Skript wird einmal ausgeführt und dann gelöscht."
    echo ""
    echo "Beispiel (einmalige Ausführung): $0 add localadmin runonce /Library/Scripts/SSF/NinjaOne/Set-Local-MacUser-Password.ps1"
    echo "Beispiel (Ausführung bei jedem Login): $0 add localadmin persistent /Library/Scripts/SSF/NinjaOne/Set-Local-MacUser-Password.ps1 add persistent"
    echo "Beispiel (Ausführung bei jedem Login, Standard): $0 add localadmin /Library/Scripts/SSF/NinjaOne/Set-Local-MacUser-Password.ps1"
    echo "Beispiel (Explizites Script entfernen): $0 localadmin remove /Library/Scripts/SSF/NinjaOne/Set-Local-MacUser-Password.ps1"
    echo "Beispiel (Alle entfernen): $0 localadmin remove /Library/Scripts/SSF/NinjaOne/Set-Local-MacUser-Password.ps1"
    exit 1
}


# Prüft, ob das PowerShell-Skript existiert und ausführbar ist
check_and_fix_ps_script() {
    local ps_script="$1"

    if [ ! -f "$ps_script" ]; then
        echo "Fehler: PowerShell-Skript '$ps_script' wurde nicht gefunden."
        return 1
    fi

    # Prüfen, ob es eine reguläre Datei ist (nicht z.B. ein Verzeichnis)
    if [ ! -f "$ps_script" ]; then
        echo "Fehler: '$ps_script' ist keine reguläre Datei."
        return 1
    fi

    if [ ! -x "$ps_script" ]; then
        echo "Warnung: PowerShell-Skript '$ps_script' ist nicht ausführbar. Versuche, die Rechte zu korrigieren..."
        sudo chmod +x "$ps_script"
        if [ $? -eq 0 ]; then
            echo "Berechtigungen für '$ps_script' erfolgreich auf ausführbar gesetzt."
        else
            echo "Fehler: Konnte Berechtigungen für '$ps_script' nicht auf ausführbar setzen. Bitte manuell prüfen."
            return 1
        fi
    fi
    return 0
}



# Funktion zum Erstellen des PowerShell-Wrapper-Skripts
Create_PowerShell_Script_Wrapper() {
    local ps_script="$1"
    local mode="$2" # Neuer Parameter für den Ausführungsmodus

    # Sicherstellen, dass das Verzeichnis existiert
    sudo mkdir -p "${WRAPPER_SCRIPT_DIR}"
    
    echo "Erstelle Wrapper-Skript: ${WRAPPER_SCRIPT_PATH} (Modus: ${mode})"
    sudo tee "${WRAPPER_SCRIPT_PATH}" > /dev/null << EOF
#!/bin/zsh
# Dies ist ein Wrapper-Skript, das von einem Launch Agent ausgeführt wird.

# Pfad zum PowerShell-Skript (wird von aussen übergeben)
POWERSHELL_SCRIPT="${ps_script}"

# Pfad zum pwsh-Executable
PWSH_PATH="${PWSH_PATH}"

echo ""
if [ "${mode}" = "runonce" ]; then
	echo "Starte Login-Script, RunOnce:"
else
	echo "Starte Login-Script, bei jedem Login:"
fi

echo "\${POWERSHELL_SCRIPT}"
echo ""


# Führe das PowerShell-Skript aus
# Standard Out und Error des PowerShell-Skripts werden in die launchd Logs umgeleitet (.plist paths)
"\${PWSH_PATH}" -File "\${POWERSHELL_SCRIPT}"
SCRIPT_EXIT_CODE=\$? # Capture exit code of PS script

echo ""
echo "Das PowerShell-Skript wurde ausgeführt (Exit-Code: \$SCRIPT_EXIT_CODE)."
echo ""

if [ "${mode}" = "runonce" ]; then
    # In 'runonce' Modus wird das Agent nach der Ausführung entladen und die .plist Datei gelöscht
    CURRENT_USER="\$(logname)" # Get current logged in user to find plist path
    PLIST_FILE="/Users/\$CURRENT_USER/Library/LaunchAgents/${LAUNCH_AGENT_LABEL}.plist"
    
    # echo "Bitte drücke Enter, um das Terminal zu schließen und die einmalige Ausführung abzuschließen."
	# echo ""
    # read -r

    # Aufräumen: Launch Agent entladen und .plist-Datei löschen
    echo "Entlade Launch Agent '\${LAUNCH_AGENT_LABEL}' und lösche die .plist Datei..."
    # launchctl unload needs to be run by the user who loaded it, which this script is.
    launchctl unload "\${PLIST_FILE}" 2>/dev/null
    rm -f "\${PLIST_FILE}"
    echo "Aufräumen abgeschlossen."
else
    echo ""
    # Im 'persistent' Modus bleibt das Terminal offen, bis es manuell geschlossen wird.
    # Wenn das PowerShell-Skript keine weitere Interaktion erfordert, 
	# schließt sich das Terminal nach 'exit'.
    # echo "Das Skript wurde im 'persistent' Modus ausgeführt."
    # echo "Das Terminal wird sich nun schließen."
fi

exit \$SCRIPT_EXIT_CODE
EOF

    # Ausführungsrechte für den Wrapper setzen
    sudo chmod +x "${WRAPPER_SCRIPT_PATH}"
}


# Funktion zum Erstellen des Launch Agent .plist Eintrags
add_launch_agent() {
    local username="$1"
    local ps_script="$2"
    local mode="$3" # Neuer Parameter: Ausführungsmodus
    local user_home="/Users/${username}"
    local plist_path="${user_home}/Library/LaunchAgents/${LAUNCH_AGENT_LABEL}.plist"

    # Stellen Sie sicher, dass das Home-Verzeichnis des Benutzers existiert
    if [ ! -d "$user_home" ]; then
        echo "Fehler: Benutzer-Home-Verzeichnis '$user_home' existiert nicht."
        exit 1
    fi

    # Prüfen und Korrigieren des PowerShell-Skripts
    if ! check_and_fix_ps_script "$ps_script"; then
        echo "Abbruch: Das PowerShell-Skript ist nicht bereit zur Ausführung."
        exit 1
    fi

    # Stellen Sie sicher, dass der Launch Agent Verzeichnis existiert
    sudo mkdir -p "${user_home}/Library/LaunchAgents"

    # Erstelle das Wrapper-Skript, das das PowerShell-Skript aufruft
    Create_PowerShell_Script_Wrapper "$ps_script" "$mode" # Den Modus an das Wrapper-Skript übergeben

    echo "Erstelle Launch Agent .plist Datei für Benutzer '${username}' (Modus: ${mode}): ${plist_path}"
    sudo tee "${plist_path}" > /dev/null << EOF
<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE plist PUBLIC "-//Apple//DTD PLIST 1.0//EN" "http://www.apple.com/DTDs/PropertyList-1.0.dtd">
<plist version="1.0">
<dict>
    <key>Label</key>
    <string>${LAUNCH_AGENT_LABEL}</string>
    <key>ProgramArguments</key>
    <array>
        <string>/usr/bin/open</string>
        <string>-a</string>
        <string>Terminal</string>
        <string>${WRAPPER_SCRIPT_PATH}</string>
    </array>
    <key>RunAtLoad</key>
    <true/>
    <key>RemainAfterExit</key>
    <false/> <key>StandardOutPath</key>
    <string>/tmp/${LAUNCH_AGENT_LABEL}.stdout.log</string>
    <key>StandardErrorPath</key>
    <string>/tmp/${LAUNCH_AGENT_LABEL}.stderr.log</string>
    <!-- TomTom: Login-Script periodisch alle n Sekunden wieder starten -->
    <!-- <key>StartInterval</key> -->
    <!-- Zeit in Sekunden: -->	
    <!-- <integer>10</integer> -->
</dict>
</plist>
EOF

    # Setze korrekte Berechtigungen und Besitzer für die .plist Datei
    sudo chown "${username}" "${plist_path}"
    sudo chmod 644 "${plist_path}"

    echo "Launch Agent erfolgreich für Benutzer '${username}' im Modus '${mode}' erstellt."
    echo "Das PowerShell-Skript wird beim nächsten Login dieses Benutzers ausgeführt."
}

# Funktion zum Entfernen des Launch Agent Eintrags
remove_launch_agent() {
    local username="$1"
    local user_home="/Users/${username}"
    local plist_path="${user_home}/Library/LaunchAgents/${LAUNCH_AGENT_LABEL}.plist"

    echo "Versuche, Launch Agent für Benutzer '${username}' zu entfernen..."

    if [ -f "${plist_path}" ]; then
        # Entlade den Launch Agent, wenn er aktiv ist (muss als der Benutzer erfolgen)
        local user_uid=$(id -u "${username}")
        if [ -n "${user_uid}" ]; then
            echo "Entlade Launch Agent mit 'launchctl bootout gui/${user_uid}/${LAUNCH_AGENT_LABEL}'..."
            # bootout ist robuster für GUI-Agenten als unload
            sudo launchctl bootout gui/${user_uid}/"${LAUNCH_AGENT_LABEL}" 2>/dev/null
            sleep 1 # Kurze Pause, um dem System Zeit zu geben
        else
            echo "Warnung: Benutzer-UID für '${username}' konnte nicht ermittelt werden. Kann Agent nicht entladen."
        fi

        # Lösche die .plist-Datei
        sudo rm -f "${plist_path}"
        echo "Launch Agent .plist Datei '${plist_path}' erfolgreich entfernt."
    else
        echo "Launch Agent .plist Datei '${plist_path}' nicht gefunden. Keine Aktion erforderlich."
    fi

    # Entferne auch das Wrapper-Skript, wenn es nicht mehr benötigt wird
    # Nur entfernen, wenn keine anderen Launch Agents dieses Wrapper-Skript nutzen könnten.
    # Bei dieser Implementierung wird ein gemeinsames Wrapper-Skript verwendet.
    # Es ist sicherer, das Wrapper-Skript nur dann zu löschen, wenn KEIN aktiver
    # Launch Agent mehr existiert, der es verwendet. Da das nicht trivial zu prüfen ist,
    # belassen wir es standardmäßig, es sei denn, man möchte es immer löschen, wenn
    # die letzte bekannte Verwendung entfernt wird. Für diese Aufgabe lassen wir es.
    if [ -f "${WRAPPER_SCRIPT_PATH}" ]; then
        echo "Entferne Wrapper-Skript: ${WRAPPER_SCRIPT_PATH}"
        sudo rm -f "${WRAPPER_SCRIPT_PATH}"
    fi

    echo "Entfernungsoperation abgeschlossen."
}

# --- Hauptlogik ---

# Überprüfen der Argumente
if [ "$#" -lt 2 ] || [ "$#" -gt 4 ]; then
    usage
    exit 1
fi

ACTION="$1"
if [ "$ACTION" != "add" ] && [ "$ACTION" != "remove" ]; then
    echo "Fehler: Ungültige Action '$ACTION'. Erlaubt sind 'add' oder 'remove'."
    usage
    exit 1
fi

USERNAME="$2"

# Wenn 3 Argumente vorhanden sind
if [ "$#" -eq 3 ]; then

    # Wenn nur 3 Argumente vorhanden sind, dann ist das dritte Argument immer der Pfad zum PowerShell-Skript
    POWERSHELL_SCRIPT_PATH="$3"
    if [ ! -f "$POWERSHELL_SCRIPT_PATH" ]; then
        echo "Fehler, existiert nicht: '$POWERSHELL_SCRIPT_PATH'"
        usage
        exit 1
    fi

    if [ "$ACTION" = "add" ]; then
        # Bei Add und nur 3 Argumenten
        # Standardmodus ist 'persistent'
        EXECUTION_MODE="persistent"
    fi

else
    # Wenn 4 Argumente vorhanden sind, dann ist das dritte Argument der Ausführungsmodus
    EXECUTION_MODE="$3"
    if [ "$EXECUTION_MODE" != "persistent" ] && [ "$EXECUTION_MODE" != "runonce" ]; then
        echo "Fehler: Ungültiger Ausführungsmodus '$EXECUTION_MODE'. Erlaubt sind 'persistent' oder 'runonce'."
        usage
		exit 1
    fi

    # Das vierte Argument ist der Pfad zum PowerShell-Skript
    POWERSHELL_SCRIPT_PATH="$4"
    if [ ! -f "$POWERSHELL_SCRIPT_PATH" ]; then
        echo "Fehler, existiert nicht: '$POWERSHELL_SCRIPT_PATH'"
        usage
        exit 1
    fi
fi


# Überprüfen, ob das Skript als Root ausgeführt wird
if [ "$(id -u)" -ne 0 ]; then
    echo "Fehler: Dieses Skript muss mit Root-Rechten (sudo) ausgeführt werden."
    exit 1
fi

case "$ACTION" in
    add)
        add_launch_agent "${USERNAME}" "${POWERSHELL_SCRIPT_PATH}" "${EXECUTION_MODE}"
        ;;
    remove)
        remove_launch_agent "${USERNAME}"
        ;;
    *)
        echo "Ungültige Aktion: '$ACTION'. Erlaubt sind 'add' oder 'remove'."
        usage
        ;;
esac

exit 0
