#!/usr/local/bin/pwsh
# !9^9 Unix (LF), nur UTF8 für MacOS, KEIN BOM

#Requires -Version 7

## Suppress PSScriptAnalyzer Warning
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidUsingCmdletAliases', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseApprovedVerbs', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidDefaultValueSwitchParameter', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidDefaultValueForMandatoryParameter', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidUsingPlainTextForPassword', '')]
# [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSShouldProcess', '')]
# [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseSingularNouns', '')]


# Lib für MacOS

# 001, 250701, Tom
# 002, 250704, Tom
# 003, 250704, Tom
# 004, 250722, Tom
#	Bereinigt


# !Kh9^9 Debuggen auf dem MacOS
#
# # PC
# # Scripts zippen und auf akros.ch/it übertragen
# \\anb030-4.akros.ch\c$\Scripts\MacOS\Für-NinjaOne\Create-Admin-Scripts-zip.ps1
#
# # MacOS
# cd /Library/Scripts/SSF/NinjaOne
# # 1. Scripts aktualisieren
# ./Update-NinjaOne-Scripts.sh -force 
# # 2. Lib laden
# . ./TomLib-MacOS.ps1 
# # 3. Lib-Funktionen nützen
# Get-FileVault-Users


# !Ex
#
### Init
# $ScriptDir = [IO.Path]::GetDirectoryName($MyInvocation.MyCommand.Path)
#
### Config
# $TomLibMacOS_ps1 = 'TomLib-MacOS.ps1'
#
### Prepare
# Lib laden
# $TomLibMacOS_ps1 = Join-Path $ScriptDir $TomLibMacOS_ps1
# . $TomLibMacOS_ps1 $PSBoundParameters


[CmdletBinding(SupportsShouldProcess, ConfirmImpact = 'Low')]
Param (
	# Optional $PSBoundParameters, 
	# um zu erkennen: IsVerbose, IsWhatIf 
	[HashTable]$PSBoundParameters_,
	# Soll GetHelp, GetEx, … parsed werden?
	[Switch]$ParseGetHelp
)



### Init

# Haben wir Verbose / WhatIf?
If ($PSBoundParameters_) {
	$IsVerbose = $PSBoundParameters_.ContainsKey('Verbose') -and $PSBoundParameters_['Verbose']
	$IsWhatIf = $PSBoundParameters_.ContainsKey('WhatIf') -and $PSBoundParameters_['WhatIf'] -or $Force -eq $False
}



### Config



### Funcs


#Region String Funcs

# Macht, was man erwartet
# 220127
Function Is-NullOrEmpty([String]$Str) {
	[String]::IsNullOrWhiteSpace($Str)
}

# Function Has-Value($Data) {
# 	If ($null -eq $Data) { Return $False }
# 	Switch ($Data.GetType().Name) {
# 		'String' {
# 			If ([String]::IsNullOrWhiteSpace($Data)) { Return $False } 
# 			Else { Return $True }
# 		}
# 		Default {
# 			Return $True
# 		}
# 	}
# }

# 250711
# 	Unterstützt neu auch die Pipeline:
#	| ? { Has-Value $_ }
Function Has-Value {
    [CmdletBinding()]
    Param (
        [Parameter(ValueFromPipeline, Position = 0)]
        [AllowNull()]
        $Data
    )
    
    Process {
        If ($null -eq $Data) { Return $False }
        Switch ($Data.GetType().Name) {
            'String' {
                If ([String]::IsNullOrWhiteSpace($Data)) { Return $False } 
                Else { Return $True }
            }
            Default {
                Return $True
            }
        }
    }
}


Function Is-Empty($Data) {
	Return !(Has-Value $Data)
}

Function Is-Verbose() {
	$VerbosePreference -eq 'Continue'
}

Function Is-WhatIf() {
	$WhatIfPreference
}

#Endregion String Funcs


#Region Log

# 250701
# 	Portiert der Windows-PowerShell Version:
# 		220123 222224
# 		Fixed: -ReplaceLine handling
$Script:LogColors = @('DarkMagenta', 'DarkGreen', 'DarkCyan', 'DarkBlue', 'Red')

# Mapping von ConsoleColor zu ANSI-Farbcodes
$Script:AnsiColorMap = @{
    Black      = 30; DarkBlue = 34; DarkGreen   = 32; 
	DarkCyan   = 36; DarkRed  = 31; DarkMagenta = 35; 
	DarkYellow   = 33; Gray     = 37; DarkGray    = 90; 
	Blue       = 94; Green    = 92; Cyan        = 96;
    Red        = 91; Magenta    = 95; Yellow      = 93; 
	White      = 97;
}

Function Convert-ToAnsiColor {
    Param (
        [ConsoleColor]$Color,
        [Switch]$Background
    )
    $ansi = $Script:AnsiColorMap[$Color.ToString()]
    If ($Background) {
        # Hintergrundfarben: 10er-Offset
        $ansi += 10
    }
    return "`e[${ansi}m"
}

# 250704 2114
#	[Switch]$IfVerbose
Function Log {
    Param (
        [Parameter(Position = 0)]
        [Int]$Indent = 0,

        [Parameter(Position = 1, ValueFromPipeline, ValueFromPipelineByPropertyName)]
		[Alias('Msg')]
		[String]$Message = '',

        [Parameter(Position = 2)]
		[Alias('Fore')]
        [ConsoleColor]$ForegroundColor,

        [Parameter(Position = 3)]
        [Switch]$NewLineBefore,

        [Parameter(Position = 4)]
        [Switch]$ReplaceLine = $false,

        [Parameter(Position = 5)]
        [Switch]$NoNewline = $false,

        [Parameter(Position = 6)]
        [Switch]$Append = $false,

        [Parameter(Position = 7)]
        [Switch]$ClrToEol = $false,

		# Ausgabe erfolgt nur, wenn Verbose aktiv ist
		[Switch]$IfVerbose,

        [Parameter(Position = 8)]
		[Alias('Back')]
        [ConsoleColor]$BackgroundColor
    )

    Begin {
        If ($Script:LogDisabled -eq $true) { return }

        If (-not $ForegroundColor) {
            If ($Indent -lt $Script:LogColors.Count) {
                $ForegroundColor = [ConsoleColor]::Parse([ConsoleColor], $Script:LogColors[$Indent])
            } else {
                $ForegroundColor = [ConsoleColor]::White
            }
        }

        $AnsiStart = Convert-ToAnsiColor -Color $ForegroundColor
        If ($BackgroundColor) {
            $AnsiStart += Convert-ToAnsiColor -Color $BackgroundColor -Background
        }
        $AnsiEnd = "`e[0m"

		# $Script:LogMaxWindowSizeWidth
		If ($null -eq $Script:LogMaxWindowSizeWidth) {
			Try {
				$ThisWidth = $Host.UI.RawUI.WindowSize.Width
				If ($Null -eq $ThisWidth) {
					$Script:LogMaxWindowSizeWidth = 132
				} Else {
					$Script:LogMaxWindowSizeWidth = $ThisWidth
				}
			} Catch {
				$Script:LogMaxWindowSizeWidth = 132
			}
		}

    }

    Process {
		If ($Script:LogDisabled -eq $True) { Return }

		# Wenn Verbose gewünscht aber nicht aktiv, dann sind wir fertig
		If ($IfVerbose -and (Is-Verbose) -eq $False) { Return }

		# Wenn -Indent < 0, sind wir fertig
		If ($Indent -lt 0) { Return }

		If ([String]::IsNullOrEmpty($Message)) { $Message = '' }

        If ($NewLineBefore) { Write-Host '' }

		If ($Append) {
			$Msg = $Message
			If ($ClrToEol) {
				If ($Msg.Length -lt $Script:LogMaxWindowSizeWidth) {
					$Msg += ' ' * ($Script:LogMaxWindowSizeWidth - $Msg.Length)
				}
			}
		}
		Else {
			Switch ($Indent) {
				0 {
					$Msg = "* $Message"
					If ($NoNewline -and $ClrToEol) {
						If ($Msg.Length -lt $Script:LogMaxWindowSizeWidth) {
							$Msg += ' ' * ($Script:LogMaxWindowSizeWidth - $Msg.Length)
						}
					}
					If (!($ReplaceLine)) {
						$Msg = "`n$Msg"
					}
				}
				Default {
					$Msg = $(' ' * ($Indent * 2) + $Message)
					If ($NoNewline -and $ClrToEol) {
						# Rest der Zeile mit Leerzeichen überschreiben
						$Width = $Script:LogMaxWindowSizeWidth
						If ($Msg.Length -lt $Script:LogMaxWindowSizeWidth) {
							$Msg += ' ' * ($Script:LogMaxWindowSizeWidth - $Msg.Length)
						}
					}
				}
			}
		}

        If ($ReplaceLine) { $Msg = "`r`e[0K$Msg" }

        If ($NoNewline) {
            Write-Host "$AnsiStart$Msg$AnsiEnd" -NoNewline
        } else {
            Write-Host "$AnsiStart$Msg$AnsiEnd"
        }
    }
}


# Gibt die Props eines PSObject aus
Function Log-Obj-Props() {
    [CmdletBinding()]
    Param (
        [Parameter(Position = 0)]
        [Int]$Indent = 5,
        [Parameter(Mandatory, Position = 1)]
        [Object]$Obj,
        [Parameter(Position = 2)]
		[String]$Comment,
        [Parameter(Position = 3)]
		[Alias('Fore')]
        [ConsoleColor]$ForegroundColor = 'Magenta'
    )

	If (Has-Value $Comment) {
		Log 4 $Comment -Fore Red
	}

	($Obj | Out-String) -Split "`n" | ? { Has-Value $_ } | % {
		Log -Indent $Indent -Message $_ -Fore $ForegroundColor
	}
}


# Gibt die Zeilen aus
Function Log-String-Arr() {
    [CmdletBinding()]
    Param (
        [Parameter(Position = 0)]
        [Int]$Indent = 1,

		[Parameter(Mandatory, Position = 1)]
		[AllowNull()]
		[AllowEmptyString()]
		[AllowEmptyCollection()]
	    [String[]]$Strings,

		[Parameter(Position = 2)]
		[String]$Comment,
        [Parameter(Position = 3)]
		[Alias('Fore')]
        [ConsoleColor]$ForegroundColor
    )

	If (Has-Value $Comment) {
		Log 4 $Comment -Fore Red
	}

	ForEach($Str in $Strings) {
		If ($Str.GetType().Basetype.Name -eq 'Array') {
			ForEach($Item in $Str) {
				# Allfällige `n splitten
				$Item -Split "`n" | ? { Has-Value $_ } | % {
					Log -Indent $Indent -Message $_ -Fore $ForegroundColor
				}
			}
		} ElseIf (Has-Value $Str) {
			# Allfällige `n splitten
			$Str -Split "`n" | ? { Has-Value $_ } | % {
				Log -Indent $Indent -Message $_ -Fore $ForegroundColor
			}
		}
	}
}

#Endregion Log


#Region Shell Farben

# Shell Styles
Enum ShellColorStyle {
    Text; H1; H2; H3; H4; Err1; Err2
}

# Die Shell-Farben mit Enum-Werten als Keys
$Script:ShellColorsHell = @{
    [ShellColorStyle]::Text = @{ ForegroundColor = 'DarkGray'; BackgroundColor = 'White' }
    
    [ShellColorStyle]::H1 = @{ ForegroundColor = 'Blue'; BackgroundColor = 'White' }
    [ShellColorStyle]::H2 = @{ ForegroundColor = 'Magenta'; BackgroundColor = 'White' }
    [ShellColorStyle]::H3 = @{ ForegroundColor = 'Green'; BackgroundColor = 'White' }
    [ShellColorStyle]::H4 = @{ ForegroundColor = 'Black'; BackgroundColor = 'White' }
    
    [ShellColorStyle]::Err1 = @{ ForegroundColor = 'Red'; BackgroundColor = 'White' }
    [ShellColorStyle]::Err2 = @{ ForegroundColor = 'DarkRed'; BackgroundColor = 'White' }
}


$Script:ShellColorsDunkel = @{
    [ShellColorStyle]::Text = @{ ForegroundColor = 'White'; BackgroundColor = 'Black' }
    
    [ShellColorStyle]::H1 = @{ ForegroundColor = 'Yellow'; BackgroundColor = 'Black' }
    [ShellColorStyle]::H2 = @{ ForegroundColor = 'Cyan'; BackgroundColor = 'Black' }
    [ShellColorStyle]::H3 = @{ ForegroundColor = 'Green'; BackgroundColor = 'Black' }
    [ShellColorStyle]::H4 = @{ ForegroundColor = 'DarkCyan'; BackgroundColor = 'Black' }
    
    [ShellColorStyle]::Err1 = @{ ForegroundColor = 'Red'; BackgroundColor = 'Black' }
    [ShellColorStyle]::Err2 = @{ ForegroundColor = 'Magenta'; BackgroundColor = 'Black' }
}


### Aktives Farbschema
$Script:ShellColors = $Script:ShellColorsDunkel


# Schreibt eine voll eingefärbte Zeile auf die Shell,
# um die Lesbarkeit zu perfektionieren
# Untersttützt Styles
# !Ex
# 	Print-Line -Msg 'Überschrift 1' -Style H1
Function Print-Line {
    [CmdletBinding(DefaultParameterSetName = 'StyleSet')]
    Param (
        [Parameter(Position = 0)]
        [Int]$Indent = 0,

        [Parameter(Position = 1)]
        [Alias('Message')]
        [String]$Msg = '',

		[Parameter(Position = 2, ParameterSetName = 'StyleSet')]
        [ShellColorStyle]$Style = [ShellColorStyle]::Text,

        [Parameter(Position = 2, ParameterSetName = 'DirectColorSet')]
        [Alias('ForegroundColor')]
        [Nullable[ConsoleColor]]$Fore = $null,

        [Parameter(Position = 2, ParameterSetName = 'DirectColorSet')]
        [Alias('BackgroundColor')]
        [Nullable[ConsoleColor]]$Back = $null
    )

	# Write-Host $PsCmdlet.ParameterSetName

    # Farben basierend auf dem verwendeten ParameterSet festlegen
    if ($PSCmdlet.ParameterSetName -eq 'StyleSet') {
        $Fore = $Script:ShellColors[$Style].ForegroundColor
        $Back = $Script:ShellColors[$Style].BackgroundColor
    }

    # $width = (tput cols)
    $width = $Host.UI.RawUI.WindowSize.Width

    $AnsiStart = ''
    $AnsiEnd = ''
    If ($null -ne $Fore) {
        $AnsiStart = Convert-ToAnsiColor -Color $Fore
    }
    If ($null -ne $Back) {
        $AnsiStart += Convert-ToAnsiColor -Color $Back -Background
    }
    If ($AnsiStart -ne '') {
        # Farbe nur beenden, wenn auch definiert
        $AnsiEnd = "`e[0m"
    }

    $OutStr = $(' ' * ($Indent * 2) + $Msg).PadRight($width)
    $OutStr = "$($AnsiStart)$($OutStr)$($AnsiEnd)"

    Write-Host $OutStr -NoNewline
    # Zeile mit den Standard-Farben abschliessen
    Write-Host ''
}


# Gibt alle Styles aus
Function Print-Styles() {

	Print-Line -Msg 'Überschrift 1' -Style H1
	Print-Line -Msg 'Lorem ipsum dolor sit Amet, consetetur sadipscing elitr, Sed diam nonumy'
	
	Print-Line -Msg 'Überschrift 2' -Style H2
	Print-Line -Msg 'Lorem ipsum dolor sit Amet, consetetur sadipscing elitr, Sed diam nonumy' -Style Text

	Print-Line -Msg 'Überschrift 3' -Style H3
	Print-Line -Msg 'Lorem ipsum dolor sit Amet, consetetur sadipscing elitr, Sed diam nonumy' 

	Print-Line -Msg 'Überschrift 4' -Style H4
	Print-Line -Msg 'Lorem ipsum dolor sit Amet, consetetur sadipscing elitr, Sed diam nonumy' 


	Print-Line -Msg 'Fehler 1' -Style Err1
	Print-Line -Msg 'Lorem ipsum dolor sit Amet, consetetur sadipscing elitr, Sed diam nonumy' 
	
	Print-Line -Msg 'Fehler 2' -Style Err2
	Print-Line -Msg 'Lorem ipsum dolor sit Amet, consetetur sadipscing elitr, Sed diam nonumy' 
	
}

#Endregion Shell Farben


#Region Shell Hilfsfunktionen

# !Ex
#	$LastExitOK, $LastExitNOK = Is-LastExitCode-AllOK
Function Is-LastExitCode-AllOK() {
    Param ([Parameter(Mandatory)][Int32]$LastExitCode)
	$IsNOK = $LastExitCode -ne $null -and $LastExitCode -ne 0
	Return (-not $IsNOK), $IsNOK
}


# Startet einen Shell-Prozess, 
# bricht ihn nach einem Tiemout ab,
# Liefert StdOut, StdErr, ExitCode, etc.
Function Start-Process-WithTimeout {
    [CmdletBinding()]
    Param (
        [Parameter(Mandatory)]
        [String]$ExeFilePath,
        
        [Parameter()]
        [String]$Arguments = '',
        
        [Parameter(ValueFromPipeline)]
		# Daten, die via Pipeline an die Shell übergeben werden
        [String]$PipeInputData,

        [Parameter()]
        [Int]$MaxWaitTimeSec = 30        
    )
    
    # Initialisiere das Ergebnisobjekt
    $result = [PSCustomObject][Ordered]@{
        HasTimeout = $false
        HasExitCodeError = $false
        HasException = $false
        HasStdErr = $false
        ExceptionMsg = ''
        ExitCode = $null
        StdOut = ''
        StdErr = ''
        ParamExe = $ExeFilePath
        ParamArgs = $Arguments
        ParamMaxTimeoutS = $MaxWaitTimeSec
		PipeInputData = $PipeInputData
    }

    # Prüfen, ob die ausführbare Datei existiert
    if (-not (Test-Path $ExeFilePath)) {
        # Finden wir das exe mit which?
        $WhichRes = (which $ExeFilePath)
        If([string]::IsNullOrEmpty($WhichRes)) {
            $result.HasException = $true
            $result.ExceptionMsg = "Fehler: $ExeFilePath existiert nicht."
            return $result
        }
        # Wenn which erfolgreich war, verwenden wir den gefundenen Pfad
        $ExeFilePath = $WhichRes
    }

	## Init
	$HasPipeInputData = (Has-Value $PipeInputData)

    
    $psi = New-Object System.Diagnostics.ProcessStartInfo
    $psi.FileName = $ExeFilePath
    $psi.Arguments = $Arguments
    $psi.RedirectStandardError = $true
    $psi.RedirectStandardOutput = $true
    # Allenfalls aktivieren
	$psi.RedirectStandardInput = $HasPipeInputData
    $psi.UseShellExecute = $false
    $psi.CreateNoWindow = $true
    
    $proc = New-Object System.Diagnostics.Process
    $proc.StartInfo = $psi
    
    try {
        $proc.Start() | Out-Null
        
        # Wenn Eingabedaten vorhanden sind, schreibe sie in den Prozess
        if ($HasPipeInputData) {
            $proc.StandardInput.WriteLine($PipeInputData)
            $proc.StandardInput.Close()  # Wichtig: Schließe den Input-Stream
        }
        
        # Ausgabe lesen
        $stdout = $proc.StandardOutput.ReadToEnd()
        $stderr = $proc.StandardError.ReadToEnd()
        
        # Auf Prozessende warten
        if (-not $proc.WaitForExit(1000 * $MaxWaitTimeSec)) {
            $result.HasTimeout = $true
            try {
                $proc.Kill()
            } catch {
                # Ignoriere Fehler beim Beenden des Prozesses
            }
            return $result
        }
        
        # Setze Ergebniswerte
        $result.ExitCode = $proc.ExitCode
        $result.StdOut = $stdout
        $result.StdErr = $stderr
		$result.HasStdErr = (Has-Value $ResProc.StdErr)
        
        If ($proc.ExitCode -ne 0) {
            $result.HasExitCodeError = $true
        } Else {
			# Exit-Code ist 0, aber haben wir StdErr?
			If ($result.HasStdErr) {
				# ExitCode forcieren
				$result.ExitCode = 99
				$result.HasExitCodeError = $true
			}
		}
    }
    catch {
        $result.HasException = $true
        $result.ExceptionMsg = $_.ToString()
    }
    
    return $result
}


# Behandelt Fehler der Funktion Start-Process-WithTimeout
# Returns: $True: Fehler
Function Handle-Start-Process-Errors() {
    [CmdletBinding()]
    Param (
        [Parameter(Mandatory)]
        [PSCustomObject]$ResProc,
        [Parameter(Mandatory)]
        [String]$ErrID = '',
		[Switch]$PrintDetails
    )

	# $ErrId für die Ausgabe vorbereiten
	If (Is-Empty $ErrID) { 
		$ErrID = '' 
	} Else {
		$ErrID = ('{0} ' -f $ErrID.Trim())
	}

	If ($ResProc.HasTimeout) {
		# Wir haben einen Fehler
		Log 4 ('{0}Prozess: Wegen Timeout abgebrochen' -f $ErrID) -Fore Red
		If ($PrintDetails) { Log-Obj-Props -Obj $ResProc}
		Return $True
	}
	ElseIf ($ResProc.HasException) {
		# Wir haben einen Fehler
		Log 4 ('{0}Prozess-Exception: {1}' -f $ErrID, $ResProc.ExceptionMsg) -Fore Red
		If ($PrintDetails) { Log-Obj-Props -Obj $ResProc}
		Return $True
	}
	ElseIf ($ResProc.HasExitCodeError) {
		# Wir haben einen Fehler
		Log 4 ('{0}Prozess mit Fehler, Exit-Code: {1}' -f $ErrID, $ResProc.ExitCode) -Fore Red
		Log 5 ('StdErr:') -Fore Magenta
		# Ausgeben: StdErr
		Log-String-Arr 6 -Strings $ResProc.StdErr -Fore Red
		# Ausgeben: $ResProc
		If ($PrintDetails) { Log-Obj-Props -Obj $ResProc}
		Return $True
	} ElseIf ($ResProc.HasStdErr) {
		# Wir haben einen Fehler, jedoch kein Exit-Code
		Log 4 ('{0}Prozess mit Fehler, Exit-Code: {1}' -f $ErrID, $ResProc.ExitCode) -Fore Red
		Log 5 ('StdErr:') -Fore Magenta
		# Ausgeben: StdErr
		Log-String-Arr 6 -Strings $ResProc.StdErr -Fore Red
		# Ausgeben: $ResProc
		If ($PrintDetails) { Log-Obj-Props -Obj $ResProc}
		Return $True
	} Else {
		# Fahler? Nein:
		Return $False
	}
}


# Liefert von einem Verzeichnis den Besitzer und die Besitzer-Gruppe
Function Get-Dir-Owner-Info {
    Param (
        [String]$Path
    )

	# todo
    # Verwende ls, um die Langversion (-ld) zu erhalten, und awk, um die korrekten Spalten auszuwählen
    # $result = bash -c "ls -ld '$Path' | awk '{print $3, $4}'"
    # $result = & "ls -ld `"$Path`" | awk '{print $3, $4}'"

	# $CmdArgs = "ls -ld '/Users/test' | awk '{print $3, $4}'"
	$CmdArgs = "-c `"ls -ld `"$Path`" | awk '{print `$3, `$4}'`""
	$ResProc = Start-Process-WithTimeout -ExeFilePath '/bin/zsh' `
										-Arguments $CmdArgs `
										-MaxWaitTimeSec 15
	
	If (Handle-Start-Process-Errors $ResProc '2jfsMi') { 
        Log 4 "Pfad nicht gefunden oder keine Berechtigung zum Zugriff: $Path" -Fore Red
		Return $Null
	} Else {
		# Teile Ergebnis in Besitzer und Gruppe auf
		$parts = $ResProc.StdOut -split ' '
	
		# Konstruiere das PSCustomObject mit den Informationen
		$ownerInfo = [PSCustomObject][Ordered]@{
			Owner       = $parts[0]
			OwnerGroup  = $parts[1]
		}
		
		Return $ownerInfo
	}
}


#Endregion Shell Hilfsfunktionen


#Region String

# Konverstiert einen SecureString
Function ConvertFrom-SecureStringToPlainText {
    Param (
        [Parameter(Mandatory)]
        [System.Security.SecureString]$SecureString
    )

    $bstr = [System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($SecureString)
    try {
        [System.Runtime.InteropServices.Marshal]::PtrToStringBSTR($bstr)
    }
    finally {
        [System.Runtime.InteropServices.Marshal]::ZeroFreeBSTR($bstr)
    }
}


# Erzeugt aus einem '' einen SecureString
Function Get-Empty-SecureString {
    Return New-Object System.Security.SecureString
}

#Endregion String


#Region User Input

# Liefert $True, wenn Ja gewählt wurde
Function Ask-JaNein($Frage) {
	$Frage = $Frage.TrimEnd('?')
	
	do {
		$response = Read-Host "$($Frage)? (j/n)"
	} while ($response -notmatch '^(j|n)$')

	Return $response -eq 'j'
}


# Fragt den Benutzer nach dem Benutzernamen
# !Ex
# $UserName = Ask-Username -Prompt 'Bitte Domain-Join Benutzernamen eingeben' -MustBeDomainUser
Function Ask-UserName() {
	Param (
		[String]$Prompt = 'Bitte Benutzernamen eingeben',
		[Switch]$MustBeDomainUser
	)

	# Write-Host $Prompt -ForegroundColor Cyan
	Do {
		# Write-Host $Prompt -ForegroundColor Cyan
		$UserName = Read-Host -Prompt $Prompt

		# Prüfen, ob es ein Domain-Benutzer sein muss
		If ($MustBeDomainUser) {
			# Ein Domain-Benutzername muss ein @ enthalten
			If ($UserName -notmatch '@' `
				-and $UserName -notmatch '\\') {
				Log 1 'Bitte Domain-Usernamen (Domain\User oder User@Domain)' -ForegroundColor Red
				$UserName = ''
			}
		}

	} While ([String]::IsNullOrWhiteSpace($UserName))
	
	Return $UserName.Trim()
}


# Fragt den Benutzer nach einem Passwort
# Optional wird ein leeres Passwort erlaubt
# !Ex
# $ScurePW = Ask-secure-Password -Prompt 'Bitte Domain-Join PW eingeben'
Function Ask-secure-Password() {
	Param (
		[String]$Prompt = 'Bitte das Passwort eingeben',
		[Switch]$AllowEmptyPW
	)

	# Write-Host $Prompt -ForegroundColor Cyan

	# So lange wiederholen, bis ein Passwort eingegeben wurde
	If ($AllowEmptyPW) {
		$SecurePW = Read-Host -AsSecureString -Prompt $Prompt
	} Else {
		# Solange wiederholen, bis ein Passwort eingegeben wurde
		Do {
			# Write-Host $Prompt -ForegroundColor Cyan
			$SecurePW = Read-Host -AsSecureString -Prompt $Prompt
		} While ($SecurePW.Length -eq 0)
	}

	Return $SecurePW
}


# Der Funktion können UserName, PW oder SecurePW übergeben werden
# und sie fragt ab, was fehlt
Function Resolve-Credentials {
    Param (
		[Parameter(Mandatory)]
        [AllowNull()][AllowEmptyString()]
        [String]$UserName,

		[Parameter(Mandatory)]
        [AllowNull()][AllowEmptyString()]
        [Object]$PWTextOrSecureStr,

		[Switch]$AllowEmptyPW,
        [Switch]$MustBeDomainUser,
        [String]$UsernamePrompt = 'Benutzername',
        [String]$PWPrompt = 'Passwort'
    )

	# Den UserName validieren
    If ([String]::IsNullOrWhiteSpace($UserName)) {
        $UserName = Ask-Username -Prompt $UsernamePrompt -MustBeDomainUser:$MustBeDomainUser
    }

	# Analyse von $PWTextOrSecureStr
	$SecurePW = $Null
	$PW = $Null
	If ($Null -ne $PWTextOrSecureStr -and $PWTextOrSecureStr.GetType().Name -eq 'String') {
		$SecurePW = $Null
		$PW = $PWTextOrSecureStr
	} ElseIf ($Null -ne $PWTextOrSecureStr -and $PWTextOrSecureStr.GetType().Name -eq 'SecureString') {
		$SecurePW = $PWTextOrSecureStr
		$PW = $Null
	}

    If ($null -eq $SecurePW) {
        # Ein leeres PW wäre OK, $Null aber nicht
		If ($null -eq $PW -or
			(-not $AllowEmptyPW -and [String]::IsNullOrEmpty($PW)) ) {
            $SecurePW = Ask-secure-Password -Prompt $PWPrompt -AllowEmptyPW:$AllowEmptyPW
        } Else {
			# Wenn $PW leer ist, muss der SecureString speziell erzeugt werden
			If ([String]::IsNullOrEmpty($PW)) {
				$SecurePW = Get-Empty-SecureString
			} Else {
				$SecurePW = ConvertTo-SecureString $PW -AsPlainText -Force
			}
        }
	}

    Return [pscustomobject]@{
        UserName = $UserName
        SecurePW = $SecurePW
    }
}

#Endregion User Input


#Region Root User

# True, wenn wir Root-Rechte haben
Function Is-Root() {
    Param ([Switch]$Dbg)
	$userID = $(id -u)
	If( $Dbg ) {
		Log 4 "Is-Root(): UserID = $userID" -Fore DarkGray
	}
    Return $userID -eq 0
}


Function Is-not-Root() {
    Param ([Switch]$Dbg)
	Return -not (Is-Root -Dbg:$Dbg)
}


Function Assert-Root([Switch]$BreakScript) {
	If (Is-not-Root) {
		Log 4 'Domain-Join benötigt Root' -Fore Red
		If ($BreakScript) {
			Log 5 'Abbruch' -Fore Magenta
			Break Script
		} Else {
			Return $False
		}
	}
	Return $True
}

#Endregion Root User


#Region MacOS User handling

#  Meldet den aktuellen macOS User ab
Function Logoff-MacOS-User($LoggedInUser) {
	try {
		# Diese Befehle erfordern sudo, um andere Benutzer abzumelden.
		# Für den angemeldeten Benutzer selbst reicht oft der direkte Aufruf ohne sudo aus,
		# aber da das Skript als sudo läuft, ist es gut, den launchctl-Ansatz zu verwenden.
		# Alternativ: & osascript -e 'tell app "System Events" to log out'
		# Die launchctl bootout Methode ist robuster für Skripte, die als root laufen.
		
		# UID des angemeldeten Benutzers ermitteln
		$LoggedInUserUID = $(id -u $LoggedInUser)
		# Logoff
		& launchctl bootout user/$LoggedInUserUID
		Log 0 'Benutzerabmeldung erfolgreich eingeleitet.'
	} catch {
		Log 4 "Fehler beim Einleiten der Benutzerabmeldung: $($_.Exception.Message)" -Fore red
		Log 5 "Bitte melden Sie sich manuell ab und wieder an, um FileVault zu aktivieren" -Fore Magenta
	}	
}


# Setzt rekursiv den Owner von /Users/$NewUserLoginName/"
# um Config-Files, die als root erstellt wurden,
# dem User zuzuordnen
# > Sonst ignoriert sie der Installations-Wizard!
Function Set-Owner-NewUser-Home() {
    Param(
        [Parameter(Mandatory)]
		# Der Login-Name des neuen Users
        [String]$NewUserLoginName
    )
	
	Log 3 "Setze den Owner von: $NewUserHome"

	# $NewUserPrefDir = "/Users/$NewUserLoginName/Library/Preferences"
	$NewUserHome = "/Users/$NewUserLoginName"

	If (-not (Test-Path -LiteralPath $NewUserHome -PathType Container)) {
		New-Item -Path $NewUserHome -ItemType Directory
	}

	# Den aktuellen Owner des Home-Verzeichnisses bestimmen
	$CurrOwner = Get-Dir-Owner-Info -Path $NewUserHome

	# Wenn's schon nicht mehr der root-User ist, dann sind wir fertig
	If ($CurrOwner -and $CurrOwner.Owner -ne 'root') {
		Log 4 "Owner von $NewUserHome ist bereits: $($CurrOwner.Owner)" -Fore White
		Log 4 'Fertig' -Fore Green
		Return
	}

	# sudo chown $NewUserLoginName $NewUserPrefDir
	$CmdArgs = "-R $($NewUserLoginName):staff $($NewUserHome)"
	$ResProc = Start-Process-WithTimeout -ExeFilePath 'chown' `
										-Arguments $CmdArgs `
										-MaxWaitTimeSec 15
	
	If (Handle-Start-Process-Errors $ResProc '2jfsMb') { 
		Log 4 '(Ignoriert)' -Fore Red 
	} Else {
		Log 3 "Owner von $NewUserHome auf $NewUserLoginName gesetzt" -Fore Green
	}
}


# Deaktiviert Elemente, die der MacOS Setup Wizard nicht abfragen soll
# !9 Der Wizard kann auch nach einem OS-Upgrade gestartet werden!
# Setzt für den Wizard SkipSetupItems
Function Set-SetupAssistant-SkipSetupItems() {
    Param(
        [Parameter(Mandatory)]
		# Der Login-Name des neuen Users
        [String]$NewUserLoginName
    )

	## Config
	$DefaultsCommand = '/usr/bin/defaults'


	# Die Elemente, die übersprungen werden sollen
	# !M https://developer.apple.com/documentation/devicemanagement/skipkeys

	#Region ❯ iOS & MacOS
	# 
	# Accessibility				skip the Accessibility pane, when creating additional users. 
	#                           Not available in macOS.
	# AdditionalPrivacySettings	skip the Additional Privacy Settings pane. 
	#                           Available in macOS 26 and later.
	# Appearance				skip the Choose Your Look screen. 
	#                           Available in iOS 13+ and macOS 10.14 and later.
	# AppleID					skip Apple Account setup.
	#                           Available in iOS 7.0+, tvOS 10.2 and later, and macOS 10.9 and later.
	# AppStore					skip the App Store pane.
	#                           Available in iOS 14.3 and later, and macOS 11.1 and later.
	# Biometric					skip biometric setup.
	#                           Available in iOS 8.1 and later, and macOS 10.12.4 and later.
	# Diagnostics				skip the App Analytics pane. 
	#                           Available in iOS 7 and later, tvOS 10.2 and later, and macOS 10.9 and later.
	# EnableLockdownMode		skip the Lockdown Mode pane if an Apple Account is set up. 
	#                           Available in iOS 17.1 and later, and macOS 14 and later.
	# FileVault					disable the FileVault Setup Assistant screen in macOS. 
	#                           Available in macOS 10.10 and later.
	# iCloudDiagnostics			skip the iCloud Analytics screen
	#                           Available in macOS 10.12.4 and later.
	# iCloudStorage				skip the iCloud Documents and Desktop screen in macOS. 
	#                           Available in macOS 10.13.4 and later.
	# Intelligence				skip the Intelligence pane. 
	#                           Available in iOS 18 and later and macOS 15 and later.
	# Location					disable Location Services. 
	#                           Available in iOS 7 and later, and macOS 10.11 and later.
	# Payment					skip Apple Pay setup. 
	#                           Available in iOS 8.1 and later, and macOS 10.12.4 and later.
	# Privacy					skip the privacy pane. 
	#                           Available in iOS 11.13 and later, tvOS 11.13 and later, and macOS 10.13.4 and later.
	# Restore					disable restoring from backup. 
	#                           Available in iOS 7 and later, and macOS 10.9 and later.
	# ScreenTime				skip the Screen Time pane. 
	#                           Available in iOS 12 and later, and macOS 10.15 and later.
	# Welcome					skip the Get Started pane. 
	#                           Available in iOS 13 and later, and macOS 15 and later.
	# RestoreCompleted			skip the Restore Completed pane. 
	# Siri						disable Siri. 
	#                           Available in iOS 7 and later, tvOS 10.2 and later, and macOS 10.12 and later.
	# TermsOfAddress			skip the Terms of Address pane. This key isn’t always skippable because this pane appears before the device retrieves the Cloud Configuration from the server. 
	#                           Available in iOS 16 and later, and macOS 13 and later.
	# Tips						skip the Tips pane. Available in macOS.
	# TOS						skip Terms and Conditions. 
	#                           Available in iOS 7 and later, tvOS 10.2 and later, and macOS 10.9 and later.
	#Endregion ❯ iOS & MacOS


	#Region ❯ Andere

	#Region ❯ iOS
	# ❯ iOS
	#
	# ActionButton				skip the Action Button configuration pane. 
	#                           Available in iOS 17 and later.
	# Android					If the Restore pane isn’t skipped, this is the key to remove the Move from Android option in the Restore pane on iOS. 
	#                           Available in iOS 9 and later.
	# CameraButton				skip the Camera Button pane. 
	#                           Available in iOS 18 and later.
	# DeviceToDeviceMigration	skip Device to Device Migration pane. 
	#                           Available in iOS 13 and later.
	# iMessageAndFaceTime		skip the iMessage and FaceTime screen in iOS. 
	#                           Available in iOS 12 and later.
	# Keyboard					skip the Keyboard pane. This pane isn’t always skippable because it appears before the device retrieves the Cloud Configuration from the server. 
	#                           Available in iOS 13 and later.
	# MessagingActivationUsingPhoneNumber
	# The key to skip the iMessage pane. Available in iOS 10 and later.
	#
	# Multitasking				skip the Multitasking pane. 
	#                           Available in iOS 26 and later.
	# OSShowcase				skip the OS Showcase pane. 
	#                           Available in iOS 26 and later.
	# Passcode					hide and disable the passcode pane. 
	#                           Available in iOS 7 and later.
	#                           Available in iOS 14 and later.
	# Safety					skip the Safety pane. 
	#                           Available in iOS 16 and later.
	# SafetyAndHandling			skip the Safety and Handling pane. 
	#                           Available in iOS 18.4 and later.
	# SIMSetup					skip the add cellular plan pane. Skipping this pane prevents automatic eSIM setup during Setup Assistant. 
	#                           Available in iOS 12 and later.
	# SoftwareUpdate			skip the mandatory software update screen in iOS. 
	#                           Available in iOS 12 and later.
	# SpokenLanguage			skip the Dictation pane. This pane isn’t always skippable because it appears before the device retrieves the Cloud Configuration from the server. 
	#                           Available in iOS 13 and later.
	# UpdateCompleted			skip the Software Update Complete pane. 
	#                           Available in iOS 14 and later.
	# WatchMigration			skip the screen for watch migration. 
	#                           Available in iOS 11 and later.
	# WebContentFiltering		skip the Web Content Filtering pane. 
	#                           Available in iOS 18.2 and later.
	#Endregion ❯ iOS

	#Region ❯ tvOS
	# ScreenSaver				skip the tvOS screen about using aerial screensavers in ATV. 
	#                           Available in tvOS 10.2 and later.
	# TVHomeScreenSync			skip TV home screen layout sync screen. 
	#                           Available in tvOS 11 and later.
	# TVProviderSignIn			skip the TV provider sign in screen. 
	#                           Available in tvOS 11 and later.
	# TVRoom					skip the “Where is this Apple TV?” screen in tvOS. 
	#                           Available in tvOS 11.4 and later.
	# TapToSetup				skip the Tap To Set Up option in Apple TV related to using an iOS device to set up your Apple TV. 
	#                           Available in tvOS 10.2 and later.
	#Endregion ❯ tvOS

	#Region ❯ Deprecated
	# ❯ Deprecated
	# DisplayTone				skip DisplayTone setup
	# HomeButtonSensitivity		skip the Meet the New Home Button screen on iPhone 7, iPhone 7 Plus, iPhone 8, iPhone 8 Plus, and iPhone SE. Available in iOS 10 and later, and deprecated in iOS 15.
	# OnBoarding				skip the on-boarding informational screens for user education (Go Home, Cover Sheet, Multitasking & Control Center, for example) in iOS. Available in iOS 11 and later, and deprecated in iOS 14.
	#Endregion ❯ Deprecated

	#Endregion ❯ Andere
	


	# Systemweit die Elemente, die der User Setup Wizard überspringen soll
	$SysSkipItems = @(
		'Accessibility',
		'AdditionalPrivacySettings',
		'Appearance',
		# skip Apple Account setup
		'AppleID',
		# skip the Lockdown Mode pane if an Apple Account is set up. 
		'EnableLockdownMode',
		'AppStore',
		'Biometric',
		'Diagnostics',
		'FileVault',
		'iCloudDiagnostics',
		'iCloudStorage',
		'Intelligence',
		'Location',
		'Payment',
		'Privacy',
		'Restore',
		'ScreenTime',
		'Welcome',
		'Siri',
		'TermsOfAddress',
		'Tips',
		'TOS'
		# 'RestoreCompleted'
	)


	# Die Elemente für diesen User, die der User Setup Wizard überspringen soll
	# Skip Items vom System übernehmen
	$UserSkipItems = $SysSkipItems


	# Skip Items für MacOS konvertieren
	$SysSkipItems_Str = $SysSkipItems -join ' '
	$UserSkipItems_Str = $UserSkipItems -join ' '

	Log 1 'Setzte default *User* Setup Assistant Settings'
	Try {
		# & "$DefaultsCommand" "write" "/Users/$NewUserLoginName/Library/Preferences/com.apple.SetupAssistant" "SkipSetupItems" -array $UserSkipItems_Str
		$CmdArgs = "write `"/Users/$NewUserLoginName/Library/Preferences/com.apple.SetupAssistant`" `"SkipSetupItems`" -array $UserSkipItems_Str"
		$ResProc = Start-Process-WithTimeout -ExeFilePath $DefaultsCommand `
												-Arguments $CmdArgs `
												-MaxWaitTimeSec 15

		If (Handle-Start-Process-Errors $ResProc '2jfsJa') {
			Log 4 '(Ignoriert)' -Fore Red
		} Else {
			Log 2 "SkipSetupItems erfolgreich gesetzt für: '$NewUserLoginName'"
			Log 2 '✅ OK' -Fore Green
		}
	
	} Catch {
		$Ex = $_
		Log 4 'Fehler!' -Fore Red
		Log 5 $Ex.Exception.Message -Fore Magenta
	}

	
	Log 1 'Setzte default *System* Setup Assistant Settings'

	Try {
		$CmdArgs = "write /Library/Preferences/com.apple.SetupAssistant.managed SkipSetupItems -array $SysSkipItems_Str"
		$ResProc = Start-Process-WithTimeout -ExeFilePath $DefaultsCommand `
												-Arguments $CmdArgs `
												-MaxWaitTimeSec 15

		If (Handle-Start-Process-Errors $ResProc '2jfsJb') {
			Log 4 '(Ignoriert)' -Fore Red
		} Else {
			Log 2 "SkipSetupItems erfolgreich gesetzt für das System"
			Log 2 '✅ OK' -Fore Green
		}

	
	} Catch {
		$Ex = $_
		Log 4 'Fehler!' -Fore Red
		Log 5 $Ex.Exception.Message -Fore Magenta
	}


	# ToDo
	Return

	# Neu in:
	# Set-SetupAssistant-Default-Values

	Log 1 'Deaktiviere default *System* Setup Assistant Settings'
	# !Dbg Testen der Konfig:
	# defaults read /Library/Preferences/com.apple.SetupAssistant  

	$SysLibPrefDir = '/Library/Preferences'
	$DefaultsFile = Join-Path $SysLibPrefDir 'com.apple.SetupAssistant'

	$PropsTrue = @(
		'DidSeeAppleIntelligenceSetup'
		'DidSeeApplePaySetup'
		'DidSeeCloudDiagnostics'
		'DidSeeCloudLogin'
		'DidSeeCloudSetup'
		'DidSeeDiagnostics'
		'DidSeePrivacy'
		'DidSeeScreenTime'
		'DidSeeSiriSetup'
    	'DidSeeTouchIDSetup'
		'DidSeeAppearanceSetup'
		'DidSeeTrueToneSetup'
		'DidSeeTrueTone'
		'SkipAnalyticsSetup'
		'SkipAppleIDSetup'
		'SkipAppleIntelligenceSetup'
		'SkipSiriSetup'
	)

	$PropsFalse = @(
		'SubmitDiagInfo'
	)

	Log 2 $DefaultsFile
	ForEach($Prop in $PropsTrue) {
		Log 3 $Prop
		sudo defaults write $DefaultsFile $Prop -bool TRUE
	}
	ForEach($Prop in $PropsFalse) {
		Log 3 $Prop
		sudo defaults write $DefaultsFile $Prop -bool FALSE
	}

}


# Setzt die Default-Werte für den MacOS SetupAssistant, der für einen neuen User gestartet wird
# 
# !9 Test der Config in der Shell!:
# defaults read /Users/test61/Library/Preferences/com.apple.SetupAssistant.plist
# 
# Er fragt diese Einstellungen ab:
#
# 1	Anredeform									✓, Config per Script
#	Geschlecht
# 2	Bedienungshilfe				✓, Disable 
#	Später
# 3	Datenschutz					n/a
#	Akzeptieren / Fortfahren
# 4	Apple Accounut				✓, Disable
#	Später
# 5	Bildschirmzeit				✓, Disable
#	Später
# 6	Apple Intelligence			✓, Disable
#	Später
# 7	Siri						✓, Disable
#	Nein
# 8	TouchID						✓, Disable
#	Später
# 9	Theme / Aussegen / Design	✓, Auto
#	Automatisch
Function Set-SetupAssistant-Default-Values() {
    Param (
        [Parameter(Mandatory)]
        [String]$NewUserLoginName,
        [Int]$Indent = 3
    )

	## Prepare
	Assert-Root -BreakScript

	## Config
	$NewUserPrefDir = "/Users/$NewUserLoginName/Library/Preferences"

	Log $Indent 'SetupAssistant: Setze Default Values'

	$DefaultSettings = @{
		## PreferencesDir: System
		'/Library/Preferences' = @{
			# Section
			'com.apple.SetupAssistant' = @{
				PropsTrue = @(
					'DidSeeAppleIntelligenceSetup'
					'DidSeeApplePaySetup'
					'DidSeeCloudDiagnostics'
					'DidSeeCloudLogin'
					'DidSeeCloudSetup'
					'DidSeeDiagnostics'
					'DidSeePrivacy'
					'DidSeeScreenTime'
					'DidSeeSiriSetup'
					'DidSeeTouchIDSetup'
					'DidSeeAppearanceSetup'
					'DidSeeTrueToneSetup'
					'DidSeeTrueTone'
					'SkipAnalyticsSetup'
					'SkipAppleIDSetup'
					'SkipAppleIntelligenceSetup'
					'SkipSiriSetup'
				)
				PropsFalse = @(
					'SubmitDiagInfo'
				)
			}
		}

		## PreferencesDir: User
		$NewUserPrefDir = @{
			# Section
			'.GlobalPreferences' = @{
				PropsTrue = @(
					'AppleInterfaceStyleSwitchesAutomatically'
				)
			}
			# Section
			'com.apple.assistant.support' = @{
				PropsFalse = @(
					'Assistant Enabled'
				)
			}
			# Section
			'com.apple.SetupAssistant' = @{
				PropsTrue = @(
					'DidSeeAccessibility'
					'DidSeeAppearanceSetup'
					'DidSeePrivacy'
					'DidSeeScreenTime'
					'DidSeeSiriSetup'
					'DidSeeTermsOfAddress'
					# 6. Touch ID Setup überspringen (nur Wizard, Hardware nicht per Script abschaltbar)
					'DidSeeTouchIDSetup'
					'InitialAccountOnMac'
					# Deaktiviert Apple ID-Abfrage
					'SkipAppleIDSetup'
					# Deaktiviert Diagnosedaten-Freigabe
					'SkipAnalyticsSetup'
					# Deaktiviert Apple Intelligence (macOS 15+)
					'SkipAppleIntelligenceSetup'
					# Deaktiviert Siri (zusätzlich)
					'SkipSiriSetup'
				)
			}
		}
	}

	## Properties setzen
	ForEach($PreferencesDir in $DefaultSettings.Keys) {
		Log ($Indent+1) "PreferencesDir: $PreferencesDir" -Fore Cyan
		$Sections = $DefaultSettings[$PreferencesDir]
		ForEach($SectionKey in $Sections.Keys) {
			Log ($Indent+2) "Section: $SectionKey"
			$Section = $Sections[$SectionKey]
			$SectionPath = Join-Path $PreferencesDir $SectionKey

			$SectionPropsTrue = $Section.PropsTrue
			If($SectionPropsTrue) {
				# Log ($Indent+3) 'Aktiviere:'
				ForEach($Prop in $SectionPropsTrue) {
					# Log ($Indent+4) "$Prop"
					sudo defaults write $SectionPath $Prop -bool TRUE
				}
			}

			$SectionPropsFalse = $Section.PropsFalse
			If($SectionPropsFalse) {
				# Log ($Indent+3) 'Deaktiviere:'
				ForEach($Prop in $SectionPropsFalse) {
					# Log ($Indent+4) "$Prop"
					sudo defaults write $SectionPath $Prop -bool FALSE
				}
			}

		}
	}

	# Version setzen
	Log ($Indent+1) 'Setze Default Version'
	$DefaultsFile = Join-Path $NewUserPrefDir 'com.apple.SetupAssistant'
	Log ($Indent+2) $DefaultsFile -Fore Cyan
	sudo defaults write $DefaultsFile LastSeenCloudProductVersion $(sw_vers -productVersion)

}


# Listet alle User, die FileVault-Zugriff haben
Function Get-FileVault-Users {
	# $Res = & fdesetup list
	$CmdArgs = "list"
	$ResProc = Start-Process-WithTimeout -ExeFilePath 'fdesetup' `
	-Arguments $CmdArgs `
	-MaxWaitTimeSec 15

	Log 1 'Liste der User mit FileVault Zugriff'
	If (Handle-Start-Process-Errors $ResProc '2jfsba') { 
		Log 4 '(Ignoriert)' -Fore Red 
	} Else {
		$ResProc.StdOut -Split "`n" | ? { Has-Value $_ } | % {
			Log 2 $_
		}
	}
}


# Nimmt dem User den FileVault-Zugriff
Function Remove-FileVault-User {
    [CmdletBinding()]
    Param (
        [Parameter(Mandatory)]
        [String]$UserLoginName
    )

	Log 1 "Entferne FileVault-Zugriff für: $UserLoginName"

	# $Res = & fdesetup list
	$CmdArgs = "remove -user $UserLoginName"
	$ResProc = Start-Process-WithTimeout -ExeFilePath 'fdesetup' `
	-Arguments $CmdArgs `
	-MaxWaitTimeSec 15

	If (Handle-Start-Process-Errors $ResProc '2jfsbb') { 
		Log 4 '(Ignoriert)' -Fore Red 
	} Else {
		$ResProc.StdOut -Split "`n" | ? { Has-Value $_ } | % {
			Log 2 $_
		}
		Log 2 '✅ OK' -Fore Green
	}
}



# Gibt einem MacOS User FileVault-Zugriff
# !TT: MacOS FileVault Konfig prüfen
# # User mit FileVault-Zugriff prüfen
# fdesetup list
# # User aus FileVault-Zugriff löschen
# fdesetup remove -user test40
# 
# 
# !Ex
# Add-FileVaultUser -AdminUserLoginName localadmin -AdminUserPW xxx -UserToAddLoginName Test40 -UserToAddPW xxx
Function Add-FileVault-User {
    [CmdletBinding()]
    Param (
        [Parameter(Mandatory)]
        [String] $MacosAdminUserLoginName,

        [Parameter(Mandatory)]
        # [SecureString] $AuthPassword
        [String] $MacosAdminUserPW,

        [Parameter(Mandatory)]
        [String] $UserToAddLoginName,
        [Parameter(Mandatory)]
        [String] $UserToAddPW
    )

	## Prepare
	Assert-Root -BreakScript


    ## Config

	# Config für fdesetup
	$plistContent = @"
		<plist>
		<dict>
			<key>Username</key>
			<string>$MacosAdminUserLoginName</string>
			<key>Password</key>
			<string>$MacosAdminUserPW</string>
			<key>AdditionalUsers</key>
			<array>
				<dict>
					<key>Username</key>
					<string>$UserToAddLoginName</string>
					<key>Password</key>
					<string>$UserToAddPW</string>
				</dict>
			</array>
		</dict>
		</plist>
"@


    Try {
        Log 1 "Aktiviere für '$UserToAddLoginName' Zugriff auf FileVault"

        # User dem FileVault zufügen

		# Prüfen
		# sudo fdesetup list
		
		# $plistContent | fdesetup add -inputplist
		$CmdArgs = "add -inputplist"
		$ResProc = $plistContent | Start-Process-WithTimeout -ExeFilePath 'fdesetup' `
												-Arguments $CmdArgs `
												-MaxWaitTimeSec 15

		If (Handle-Start-Process-Errors $ResProc '2jfsba') {
			# Log 4 '(Ignoriert)' -Fore Red
		} Else {
			Log 2 '✅ OK' -Fore Green
		}

		# Debug
		# Ausgeben: $ResProc
		# Log-Obj-Props -Obj $ResProc

    }
    Catch {
        Log 4 "❌ 2jfsbb Exception: $_"
    }
}



# Initialisiert einen MacOS User, 
# so dass der Einrichtungs-Wizard nicht durchgeklickt werden muss
# Der Wizard fragt diese Props ab: (250716)
#								Disable Sys		Disable User
Function Init-MacOS-User-Archiv() {
    Param(
        [Parameter(Mandatory)]
		# Muss das MacOS SecureToken haben
        [String]$MacosAdminUserLoginName,
        [Parameter(Mandatory)]
        [String]$MacosAdminUserPW,

        [Parameter(Mandatory)]
		# Der Login-Name des neuen Users
        [String]$NewUserLoginName,
        [Parameter(Mandatory)]
        [String]$NewUserPW,

        [Parameter(Mandatory)]
        [String]$NewUserFullName
	)

	
	### Config	
	
	### Init
	Assert-Root -BreakScript


	Log 0 "Erzeuge neuen User: $NewUserFullName"
	# ❯ MacOS Shell meldet:
	# 	sysadminctl[68194:1017387] No clear text password or interactive option was specified (adduser, change/reset password will not allow user to use FDE) !
	# > FDE steht für FileVault Disk Encryption
	# > Meint: 
	#   Wenn beim Erstellen des Benutzerkontos das Passwort nicht interaktiv angegeben wird,
	#   kann es sein, dass dieser Benutzer nicht sofort für die Entsperrung des Startvolumes mit FileVault berechtigt ist.
	#   Normalerweise muss ein Benutzer einmalig das Passwort interaktiv eingeben 
	#   (z.B. bei der ersten Anmeldung oder wenn er zu den FileVault-Benutzern hinzugefügt wird), 
	#   damit sein Passwort für die Entschlüsselung des Startvolumes registriert wird.
	
	# sudo sysadminctl -addUser $NewUserLoginName -fullName $NewUserFullName -password $NewUserPW
	$CmdArgs = "-addUser `"$NewUserLoginName`" -fullName `"$NewUserFullName`" -password `"$NewUserPW`""
	$ResProc = Start-Process-WithTimeout -ExeFilePath 'sysadminctl' `
											-Arguments $CmdArgs `
											-MaxWaitTimeSec 15

	If (Handle-Start-Process-Errors $ResProc '2jfsJc') {
		Log 4 '(Ignoriert)' -Fore Red
	}

}



# Prüft, ob der locale MacOS-Benutzer existiert
# Liefert:
#	UserExists		Ob der User existiert
#	IsDomainUser	Ob es sich um einen Domain-User handelt
Function Does-MacOS-User-Exists {
    Param(
        [Parameter(Mandatory)]
        [String]$UserLoginName
    )

	### Init
	$UserLoginName = $UserLoginName.Trim()


    # Prüft, ob der Benutzer existiert (nutzt dscl)
	# $Null = dscl . -read "/Users/$UserLoginName" 2>$null
	$CmdArgs = ". -read `"/Users/$UserLoginName`""
	$ResProc = Start-Process-WithTimeout -ExeFilePath 'dscl' `
											-Arguments $CmdArgs `
											-MaxWaitTimeSec 15

	# Dbg
	# If (Handle-Start-Process-Errors $ResProc '2jfsJc') {
	# 	Log 4 '(Ignoriert)' -Fore Red
	# }

	# Existiert der User?
	$UserExists = $ResProc.ExitCode -eq 0

	# Ist der User ein Domain-User?
	$IsDomainUser = $False
	If ($UserExists) {
		$IsDomainUser = @($ResProc.StdOut -Split "`n" | ? { $_ -like 'PrimaryNTDomain:*' }).Count -ge 1
	}

	Return [PSCustomObject][Ordered]@{
		UserExists = $UserExists
		IsDomainUser = $IsDomainUser
	}
}


# Sucht eine freie UniqueID
Function New-UniqueID {
    $AllUsers = dscl . -list /Users UniqueID
    $UsedIDs = $AllUsers | % { ($_ -Split('\s+'))[1] }
    $UniqueID = 501
    While ($UsedIDs -contains "$UniqueID") {
        $UniqueID++
    }
    Return $UniqueID
}



# Nachdem ein MacOS User erzeiugt wurde, wird er vorbereitet:
# - Setzt die Shell auf zsh
# - Gibt dem User Admin-Rechte
# - Setzt das Secure Token
# - Aktiviert FileVault für den User
# - Setzt den Owner des Home-Verzeichnisses
# - Konfiguriert den User für den Setup Wizard
#	 Fehlt noch:
#	 - Apple Intelligence
#	 - Bei Apple Account anmelden
#	 - Analyse (Mac-Analysedaten mit Apple teilen)
Function Prepare-MacOS-User() {
    Param (
        [Parameter(Mandatory)]
        [String]$MacosAdminUserLoginName,
        [Parameter(Mandatory)]
        [String]$MacosAdminUserPW,
        [Parameter(Mandatory)]
        [String]$NewUserLoginName,
        [Parameter(Mandatory)]
        [String]$NewUserPW
    )


	### Init
	Assert-Root -BreakScript

	
	### Main

    Log 2 'Setze die Shell zsh'
    # dscl . -read /Users/nypadmin UserShell
    # /bin/zsh
    # dscl . -create /Users/$NewUserLoginName UserShell /bin/zsh
	$CmdArgs = ". -create /Users/$NewUserLoginName UserShell /bin/zsh"
	$ResProc = Start-Process-WithTimeout -ExeFilePath 'dscl' `
										-Arguments $CmdArgs `
										-MaxWaitTimeSec 15
	
	If (Handle-Start-Process-Errors $ResProc '2jfsMd') { Log 4 '(Ignoriert)' -Fore Red }
	
	## .zshrc
	## .zshrc anlegen, um zsh-Initialisierungs-Wizard zu unterdrücken
	## ! Nicht nötig, MacOS erzeugt keine userspezifische .zshrc Datei, sondern:
	# ls -al /etc/zshrc
	# -r--r--r--  1 root  wheel  3094  4 Mai 07:39 /etc/zshrc


	Log 2 'Weise Admin-Rechte zu'
    # Die PrimaryGroupID 20 steht auf macOS für die Gruppe staff.
    # Standardmäßig werden alle normalen Benutzer (einschließlich Administratoren) auf macOS
    # der Gruppe staff zugeordnet, deren numerische Gruppen-ID (GID) 20 ist.
    # Diese Gruppe ist die primäre Gruppe für fast alle lokalen Benutzerkonten auf macOS

	Log 3 'Setzte die PrimaryGroupID'
	# dscl . -create /Users/$NewUserLoginName PrimaryGroupID 20
	$CmdArgs = ". -create /Users/$NewUserLoginName PrimaryGroupID 20"
	$ResProc = Start-Process-WithTimeout -ExeFilePath 'dscl' `
										-Arguments $CmdArgs `
										-MaxWaitTimeSec 15
	
	If (Handle-Start-Process-Errors $ResProc '2jfsMg') { Log 4 '(Ignoriert)' -Fore Red }

	Log 3 'Setzte das NFSHomeDirectory'
    # dscl . -create /Users/$NewUserLoginName NFSHomeDirectory /Users/$NewUserLoginName
	$CmdArgs = ". -create /Users/$NewUserLoginName NFSHomeDirectory /Users/$NewUserLoginName"
	$ResProc = Start-Process-WithTimeout -ExeFilePath 'dscl' `
										-Arguments $CmdArgs `
										-MaxWaitTimeSec 15
	
	If (Handle-Start-Process-Errors $ResProc '2jfsMh') { Log 4 '(Ignoriert)' -Fore Red }

	Log 3 'Setzte die Admin Group Membership'
    # dscl . -append /Groups/admin GroupMembership $NewUserLoginName
	$CmdArgs = ". -append /Groups/admin GroupMembership $NewUserLoginName"
	$ResProc = Start-Process-WithTimeout -ExeFilePath 'dscl' `
										-Arguments $CmdArgs `
										-MaxWaitTimeSec 15
	
	If (Handle-Start-Process-Errors $ResProc '2jfsMj') { Log 4 '(Ignoriert)' -Fore Red }


    ## Das Secure Token vergeben
    # !KH9^9: Die Passwörte müssen in Anführungszeichen stehen, damit sie korrekt verarbeitet werden
    #           Sonst interpretiert die zsh ein Passwort mit einem führenden Minuszeichen als Stopp der Parameter
    
    Log 2 'Weise das Secure-Token zu'
    # sysadminctl -secureTokenOn $NewUserLoginName -password "`"$NewUserPW`"" -adminUser $MacosAdminUserLoginName -adminPassword  "`"$MacosAdminUserPW`""
	$CmdArgs = "-adminUser `"$MacosAdminUserLoginName`" -adminPassword `"$MacosAdminUserPW`" -secureTokenOn `"$NewUserLoginName`" -password `"$NewUserPW`""
	$ResProc = Start-Process-WithTimeout -ExeFilePath 'sysadminctl' `
										-Arguments $CmdArgs `
										-MaxWaitTimeSec 15
	
	If (Handle-Start-Process-Errors $ResProc '2jfsMk') { Log 4 '(Ignoriert)' -Fore Red }

    Log 2 'Aktiviere FileVault'
	Add-FileVault-User -MacosAdminUserLoginName $MacosAdminUserLoginName `
						-MacosAdminUserPW $MacosAdminUserPW `
						-UserToAddLoginName $NewUserLoginName `
						-UserToAddPW $NewUserPW


	Log 2 'Konfiguriere Willkommens-Assistenten'
	# Config
	$NewUserPrefDir = "/Users/$NewUserLoginName/Library/Preferences"
	
	Log 3 'SetupAssistant: Setze Basis settings'
	Set-SetupAssistant-SkipSetupItems -NewUserLoginName $NewUserLoginName

	# Log 3 'SetupAssistant: Setze Default settings'
	Set-SetupAssistant-Default-Values -NewUserLoginName $NewUserLoginName

	Log 3 "Setze rekursiv den Owner in: $NewUserPrefDir"
	# !9^9 Sonst ignoriert der MacOS user Setup Wizard die Vorgaben!
	Set-Owner-NewUser-Home -NewUserLoginName $NewUserLoginName

}



# Erzeugt einen neuen lokalen Administrator
#
# !Ex
# New-MacOS-Local-User -MacosAdminUserLoginName localadmin -MacosAdminUserPW "xxx" -NewUserLoginName test50 -NewUserPW "xxx" -NewUserFullName "Hans Muster"
Function New-MacOS-Local-User() {
    Param (
        [Parameter(Mandatory)]
		# Der MacOS Admin-User, der den neuen User anlegt
        [String]$MacosAdminUserLoginName,
        [Parameter(Mandatory)]
		# Das PW des MacOS Admin-User, der den neuen User anlegt
        [String]$MacosAdminUserPW,
        [Parameter(Mandatory)]
		# Der Login-Name des neuen Users
        [String]$NewUserLoginName,
        [Parameter(Mandatory)]
		# Das PW des neuen Users
        [String]$NewUserPW,
        [Parameter(Mandatory)]
		# Vor- und Nachname des neuen Users
        [String]$NewUserFullName,
        [String]$UniqueID = (New-UniqueID)
    )

	### Init
	Assert-Root -BreakScript
	
	
	### Main

    ## Den neuen User anlegen
	Log 2 'Erzeuge den lokalen MacOS User'

	## Erzeugen: Lokalen MacOS User
	# dscl . -create /Users/$NewUserLoginName
	$CmdArgs = ". -create /Users/$NewUserLoginName"
	$ResProc = Start-Process-WithTimeout -ExeFilePath 'dscl' `
	-Arguments $CmdArgs `
	-MaxWaitTimeSec 15
	
	If (Handle-Start-Process-Errors $ResProc '2jfsMc') { Log 4 '(Ignoriert)' -Fore Red }

	Log 2 'Setze den RealName'
    # dscl . -create /Users/$NewUserLoginName RealName "$NewUserLoginName"
	$CmdArgs = ". -create /Users/$NewUserLoginName RealName `"$NewUserFullName`""
	$ResProc = Start-Process-WithTimeout -ExeFilePath 'dscl' `
										-Arguments $CmdArgs `
										-MaxWaitTimeSec 15
	
	If (Handle-Start-Process-Errors $ResProc '2jfsMe') { Log 4 '(Ignoriert)' -Fore Red }


	Log 2 'Setze die UniqueID'
    # dscl . -create /Users/$NewUserLoginName UniqueID "$UniqueID"
	$CmdArgs = ". -create /Users/$NewUserLoginName UniqueID `"$UniqueID`""
	$ResProc = Start-Process-WithTimeout -ExeFilePath 'dscl' `
										-Arguments $CmdArgs `
										-MaxWaitTimeSec 15
	
	If (Handle-Start-Process-Errors $ResProc '2jfsMf') { Log 4 '(Ignoriert)' -Fore Red }

	
	Log 3 'Setzte das PW'
    # dscl . -passwd /Users/$NewUserLoginName "$NewUserPW"
	$CmdArgs = ". -passwd /Users/$NewUserLoginName `"$NewUserPW`""
	$ResProc = Start-Process-WithTimeout -ExeFilePath 'dscl' `
										-Arguments $CmdArgs `
										-MaxWaitTimeSec 15
	
	If (Handle-Start-Process-Errors $ResProc '2jfsMi') { Log 4 '(Ignoriert)' -Fore Red }


    # Kurz warten, damit der User angelegt ist
	Start-Sleep -Milliseconds 3500


	Log 2 'Initialisiere den neuen User'
	Prepare-MacOS-User -MacosAdminUserLoginName $MacosAdminUserLoginName `
										-MacosAdminUserPW $MacosAdminUserPW `
										-NewUserLoginName $NewUserLoginName `
										-NewUserPW $NewUserPW

	Log 2 '✅ Alles OK' -Fore Green
}



# Erzeugt einen mobilen MacOS User als Administrator
# 
# !Ex
# New-MacOS-Mobile-User -MacosAdminUserLoginName localadmin -MacosAdminUserPW "xxx" -NewADUserName test -NewUserPW "xxx"
Function New-MacOS-Mobile-User() {
    Param (
        [Parameter(Mandatory)]
        [String]$MacosAdminUserLoginName,
        [Parameter(Mandatory)]
        [String]$MacosAdminUserPW,
        [Parameter(Mandatory)]
		# Kann beides sein:
		# - Test@akros.ch
		# - Test
		# Eff. benötigt wird nur 'Test'
        [String]$NewADUserName,
        [Parameter(Mandatory)]
        [String]$NewUserPW
    )

	### Init
	Assert-Root -BreakScript
	
	### Config
	$CreateMobileAccountPath = '/System/Library/CoreServices/ManagedClient.app/Contents/Resources/createmobileaccount'
	
	### Prepare
	## Params bereinigen
	$NewADUserName = $NewADUserName.Trim()
	# Nur den Username ohne Domäne
	$NewADUserName = $NewADUserName.Split('@')[0]

	# Existiert: createmobileaccount?
	if (-not (Test-Path $CreateMobileAccountPath)) {
		Log 4  'Nicht gefunden:' -Fore Red
		Log 5  $CreateMobileAccountPath -Fore Magenta
		Exit 1
	}

	
	### Main

    ## Den neuen User anlegen
	Log 2 'Erzeuge den Mobilen MacOS User'

	## Erzeugen: Mobilen MacOS User
	$CmdArgs = "-n `"$NewADUserName`" -a `"$MacosAdminUserLoginName`" -U `"$MacosAdminUserPW`""
	$ResProc = Start-Process-WithTimeout -ExeFilePath "$CreateMobileAccountPath" `
										-Arguments $CmdArgs `
										-MaxWaitTimeSec 15
	
	If (Handle-Start-Process-Errors $ResProc '2jfsMa') { Log 4 '(Ignoriert)' -Fore Red }

		
    # ToDo
	# Log 2 'Setze die UniqueID'
    # # dscl . -create /Users/$NewADUserName UniqueID "$UniqueID"
	# $CmdArgs = ". -create /Users/$NewADUserName UniqueID `"$UniqueID`""
	# $ResProc = Start-Process-WithTimeout -ExeFilePath 'dscl' `
	# 									-Arguments $CmdArgs `
	# 									-MaxWaitTimeSec 15
	#
	# If (Handle-Start-Process-Errors $ResProc '2jfsMf') { Log 4 '(Ignoriert)' -Fore Red }
	
	# Log 3 'Setzte das PW'
    # # dscl . -passwd /Users/$NewADUserName "$NewUserPW"
	# $CmdArgs = ". -passwd /Users/$NewADUserName `"$NewUserPW`""
	# $ResProc = Start-Process-WithTimeout -ExeFilePath 'dscl' `
	# 									-Arguments $CmdArgs `
	# 									-MaxWaitTimeSec 15
	#	
	# If (Handle-Start-Process-Errors $ResProc '2jfsMi') { Log 4 '(Ignoriert)' -Fore Red }

    # Kurz warten, damit der User angelegt ist
    Start-Sleep -Milliseconds 3500


	Log 2 'Initialisiere den neuen User'
	Prepare-MacOS-User -MacosAdminUserLoginName $MacosAdminUserLoginName `
										-MacosAdminUserPW $MacosAdminUserPW `
										-NewUserLoginName $NewADUserName `
										-NewUserPW $NewUserPW

	Log 2 '✅ Alles OK' -Fore Green
}



#Endregion MacOS User handling


#Region Hostname

# Gibt auf der Konsole die Info der Hostnamen aus
# !Ex
# $oHostnamesInfo = Get-HostNames-Info
# Print-HostNamesInfo -oHostnamesInfo $oHostnamesInfo
Function Print-HostNamesInfo($oHostnamesInfo) {

	Log 0 'HostNamesInfo'

	Log 1 'Domain-Join : ' -NoNewline
	If ($oHostnamesInfo.IsDomainJoined) {
		Log -Append -Msg 'Ja' -Fore Red
	} Else {
		Log -Append -Msg 'Nein' -Fore Red
	}
	
    Log 1 "JoinedDomain: $($oHostnamesInfo.JoinedDomain)"

	Log 1 'Aktuelle Hostnamen'
    Log 2 "  MacOS ComputerName   : $($oHostnamesInfo.MacOS_ComputerName)"
	# HostName == FQDN
    Log 2 "  MacOS HostName (Fqdn): $($oHostnamesInfo.MacOS_HostName)"

    Log 2 "  MacOS LocalHostName  : $($oHostnamesInfo.MacOS_LocalHostName)"
    Log 2 "  > Effektiv           : $($oHostnamesInfo.MacOS_LocalHostNameEffektiv)"

	Log 1 'FQDN (berechnet aus HostName)'
	Log 2 "  FQDN                 : $($oHostnamesInfo.FQDN)"
	Log 2 "  FQDN HostName        : $($oHostnamesInfo.FQDN_HostName)"
	Log 2 "  FQDN Domain          : $($oHostnamesInfo.FQDN_Domain)"
}


# Berechnet alle Infos zum MacOS Hostnamen
# $oHostnamesInfo = Get-HostNames-Info
Function Get-HostNames-Info() {
	# Hostnamen bestimmen
	# HostName == FQDN
	$MacOS_HostName = "$(scutil --get HostName 2>$null)"

	$oHostnamesInfo = [PSCustomObject][Ordered]@{
		IsDomainJoined = Is-Domain-Joined
		JoinedDomain = Get-Joined-Domain
		MacOS_ComputerName = "$(scutil --get ComputerName 2>$null)"
		# HostName == FQDN
		MacOS_HostName = $MacOS_HostName
		MacOS_LocalHostName = "$(scutil --get LocalHostName 2>$null)"
		MacOS_LocalHostNameEffektiv = "$(scutil --get LocalHostName 2>$null).local"
		# FQDN == HostName
		FQDN = $MacOS_HostName
		FQDN_HostName = $MacOS_HostName.Split('.')[0]
		FQDN_Domain = ($MacOS_HostName.Split('.')[1..99] -join '.')
	}
	Return $oHostnamesInfo
}


# Ermittelt die aktuellen Hostnamen durch Analyse von $oHostnamesInfo
# $CurrentHostName = Get-Current-HostName $oHostnamesInfo
Function Get-Current-HostName() {
	Param(
		# Optional
		[PSCustomObject]$oHostnamesInfo
	)

	## Init
	If ($oHostnamesInfo -eq $null) {
		# Wenn kein oHostnamesInfo übergeben wurde, dann holen wir die Infos
		$oHostnamesInfo = Get-HostNames-Info
	}

	# Zuerst den LocalHostName verwenden, wenn er gesetzt ist
	If (Has-Value $oHostnamesInfo.MacOS_LocalHostName) {
		Return $oHostnamesInfo.MacOS_LocalHostName
	}
	# Sonst ComputerName
	If (Has-Value $oHostnamesInfo.MacOS_ComputerName) {
		Return $oHostnamesInfo.MacOS_ComputerName
	}
	# Sonst das erste Element von HostName (=FQDN)
	Return $oHostnamesInfo.MacOS_HostName.Split('.')[0]
}


# Setzt die MacOS Hostnamen
Function Set-HostNames {
	Param (
		[AllowEmptyString()][String]$NewHostName,
		[AllowEmptyString()][String]$DomainName,
		[Switch]$Force,
		[Switch]$Dbg
	)
	

	# 📌 Empfohlen bei domain-joined Macs:
	# Wenn dein Mac Mitglied einer Active Directory-Domain ist, 
	# ist es besonders wichtig, die Namen konsistent zu setzen:
	#
	# sudo scutil --set HostName "myhost.mydomain.com"
	# sudo scutil --set ComputerName "myhost"
	# sudo scutil --set LocalHostName "myhost"


	## Check
	If ($Dbg) {
		Log 4 'Set-HostNames(), vor Bereinigung:'
		Log 5 "NewHostName: '$NewHostName'"
		Log 5 "DomainName : '$DomainName'"
	}

	If ((Is-NullOrEmpty $NewHostName) -and (Is-NullOrEmpty $DomainName)) {
		Log 4 'Set-HostNames(): Nichts zu tun: Weder Hostname noch Domain angegeben' -Fore Red
		Log 5 "NewHostName: $NewHostName" -Fore Red
		Log 5 "DomainName : $DomainName" -Fore Red
		Return
	}


	## Init
	# Hostnamen lesen
	$oThisHostnamesInfo = Get-HostNames-Info

	# Params bereinigen
	If (Has-Value $NewHostName) { $NewHostName = $NewHostName.Trim() }
	If (Has-Value $DomainName) { $DomainName = $DomainName.Trim() }

	If ($Dbg) {
		Log 4 'Set-HostNames(), nach Bereinigung:'
		Log 5 "NewHostName: '$NewHostName'"
		Log 5 "DomainName : '$DomainName'"
	}


	# Ändert der Hostname?
	If (Is-NullOrEmpty $NewHostName) {
		# Kein neuer Hostname gesetzt - den alten verwenden
		$NewHostName = Get-Current-HostName $oThisHostnamesInfo

		If ($Dbg) {
			Log 4 'Set-HostNames(), Kein neuer Hostname angegeben, nütze den aktuell gesetzten:'
			Log 5 "NewHostName: '$NewHostName'"
		}
	}


	# Den DQDN bestimmen
	# Ohne Domäne ist der full-qualified Domain Name (FQDN) nur der Hostname
	$FQDN = $NewHostName
	If (Has-Value $DomainName) {
		$FQDN = ('{0}.{1}' -f $NewHostName, $DomainName.Trim())	
	}
	If ($Dbg) {
		Log 4 'Set-HostNames(), berechneter FQDN:'
		Log 5 "FQDN: '$FQDN'"
	}


	If ($Force) {
		# Write-Host "Setze Hostnamen: '$NewHostName'.."
		If ($Dbg) { Log 4 '-Force aktiv' }

		# Änderte der ComputerName?
		If ($oThisHostnamesInfo.MacOS_ComputerName -ne $NewHostName) {
			If ($Dbg) { Log 5 "Aktualisiere ComputerName: '$NewHostName'" }
			sudo scutil --set ComputerName "$NewHostName"
		} Else {
			If ($Dbg) { Log 5 'ComputerName: Schon aktuell' }
		}

		# Änderte der HostName?
		# HostName == FQDN
		If ($oThisHostnamesInfo.MacOS_HostName -ne $FQDN) {
			If ($Dbg) { Log 5 "Aktualisiere HostName: '$FQDN'" }
			sudo scutil --set HostName "$FQDN"
		} Else {
			If ($Dbg) { Log 5 'HostName: Schon aktuell' }
		}

		# Änderte der LocalHostName?
		If ($oThisHostnamesInfo.MacOS_LocalHostName -ne $NewHostName) {
			# Erzeugt automatisch $NewHostName.local
			If ($Dbg) { Log 5 "Aktualisiere LocalHostName: '$NewHostName'" }
			sudo scutil --set LocalHostName "$NewHostName"
		} Else {
			If ($Dbg) { Log 5 'LocalHostName: Schon aktuell' }
		}

		Log 2 'Hostnamen wurden gesetzt' -Fore Green
		
	} Else {
		# Write-Host "Würde Hostnamen setzen:"
		If ($Dbg) { Log 4 '-Force NICHT aktiv' }
		
		Log 2 'MacOS ComputerName : ' -NoNewLine
		If ($oThisHostnamesInfo.MacOS_ComputerName -eq $NewHostName) {
			Log -Append -Msg ('{0} (Schon OK)' -f $NewHostName) -Fore Green
		} Else {
			Log -Append -Msg $NewHostName -Fore Red
		}
		
		# HostName == FQDN
		Log 2 "MacOS HostName     : " -NoNewLine
		If ($oThisHostnamesInfo.MacOS_HostName -eq $FQDN) {
			Log -Append -Msg ('{0} (Schon OK)' -f $FQDN) -Fore Green
		} Else {
			Log -Append -Msg $FQDN -Fore Red
		}
		
		Log 2 "MacOS LocalHostName: " -NoNewLine
		If ($oThisHostnamesInfo.MacOS_LocalHostName -eq $NewHostName) {
			Log -Append -Msg ('{0} (Schon OK)' -f $NewHostName) -Fore Green
		} Else {
			Log -Append -Msg $NewHostName -Fore Red
		}
		
	}
}

#Endregion Hostname


#Region Domain-Join

# Ist das MacOS Domain-Joined?
Function Is-Domain-Joined() {

    #Region dsconfigad -show

    # Active Directory Forest          = akros.ch
    # Active Directory Domain          = akros.ch
    # Computer Account                 = anb289$

    # Advanced Options - User Experience
    #   Create mobile account at login = Enabled
    #      Require confirmation        = Enabled
    #   Force home to startup disk     = Enabled
    #      Mount home as sharepoint    = Enabled
    #   Use Windows UNC path for home  = Enabled
    #      Network protocol to be used = smb
    #   Default user Shell             = /bin/bash

    # Advanced Options - Mappings
    #   Mapping UID to attribute       = not set
    #   Mapping user GID to attribute  = not set
    #   Mapping group GID to attribute = not set
    #   Generate Kerberos authority    = Enabled

    # Advanced Options - Administrative
    #   Preferred Domain controller    = not set
    #   Allowed admin groups           = not set
    #   Authentication from any domain = Enabled
    #   Packet signing                 = allow
    #   Packet encryption              = allow
    #   Password change interval       = 14
    #   Restrict Dynamic DNS updates   = not set
    #   Namespace mode                 = domain

    #Endregion dsconfigad -show

	$adInfo = dsconfigad -show 2>$null
	$LastExitOK, $LastExitNOK = Is-LastExitCode-AllOK $LastExitCode
	$ADInfoMissing = [String]::IsNullOrWhiteSpace($adInfo)
	$IsDomainJoined = $LastExitOK -and $ADInfoMissing -eq $False

	# Write-Host "adInfo        : $adInfo"
	# Write-Host "LastExitOK    : $LastExitOK"
	# Write-Host "ADInfoMissing : $ADInfoMissing"
	# Write-Host "IsDomainJoined: $IsDomainJoined"
	# Write-Host "IsDomainJoined: $IsDomainJoined"

	Return $IsDomainJoined
}


# Liest die Domäne, zu der das MacOS joined ist
# !Ex
#	$JoinedDomain = Get-Joined-Domain
Function Get-Joined-Domain {
    $dsconfigadOutput = & dsconfigad -show 2>$null
	$LastExitOK, $LastExitNOK = Is-LastExitCode-AllOK $LastExitCode
	If ($LastExitOK -and $dsconfigadOutput) {
        $Domain = ($dsconfigadOutput | ? { $_ -match "^Active Directory Domain" }) -replace "^.*=\s*", ""
		Return $Domain.Trim()
	} Else {
        Return $Null
	}
}


# Domain-Join für MacOS
# Wenn UserName oder PW fehlt, fragt das Script danach
Function Join-MacOS-to-Domain {
	[CmdletBinding(DefaultParameterSetName = 'Secure')]
    Param (
		[Parameter(Mandatory)]
        [String]$DomainName,
		# Optional ein UserName
        [Object]$DomainAdUserName,
		# Optional ein PW
        [Object]$DomainAdUserPW
    )

	Log 0 'Starte Domain-Join'
	
	### Prepare
	$DomainName = $DomainName.Trim()

	## Init
	If (-Not (Assert-Root)) {
		Log 5 'Abbruch' -Fore Magenta
		Return $False
	}
	
	## UserName und PW prüfen
	$Cred = Resolve-Credentials -UserName $DomainAdUserName `
								-PWTextOrSecureStr $DomainAdUserPW `
								-MustBeDomainUser `
								-UsernamePrompt 'Domain-Join Benutzernamen (user@domain.com)' `
								-PWPrompt 'Domain-Join Passwort' `

	### Checks
	If (Is-Domain-Joined) {
		$JoinedDomain = Get-Joined-Domain
		If ($JoinedDomain -eq $DomainName) {
			Log 2 "Computer ist schon in der Domäne: $DomainName" -Fore Green
			Return $True
		} Else {
			Log 4 'Fehler beim Domain-Join!' -Fore Red
			Log 5 "Computer ist bereits in einer anderen Domäne: $JoinedDomain" -Fore Magenta
			Return $False
		}
	}


	# Aktuelle Hostnamen abrufen
	$oHostnamesInfo = Get-HostNames-Info
	$ComputerName = $oHostnamesInfo.MacOS_LocalHostName
	
	$ResDomainJoinOK = $False
    Try {
		# Das PW im Klartext
		$DomainAdUserPW = ConvertFrom-SecureStringToPlainText -SecureString $Cred.SecurePW

		# Domain Join ausführen
        # 2>&1 stderr nach stdout

		# Write-Host "DomainName   : $DomainName"
		# Write-Host "Cred.UserName: $($Cred.UserName)"
		# Write-Host "Cred.SecurePW: $($Cred.SecurePW)"
		# Write-Host "PW           : $DomainAdUserPW"

        # $ResCmdline = & /usr/sbin/dsconfigad -add "$DomainName" -computer "$ComputerName" -force -username "$($Cred.UserName)" -password "$DomainAdUserPW" 2>&1
		# $LastExitOK, $LastExitNOK = Is-LastExitCode-AllOK $LastExitCode		
		$CmdArgs = "-add `"$DomainName`" -computer `"$ComputerName`" -force -username `"$($Cred.UserName)`" -password `"$DomainAdUserPW`""
		$ResProc = Start-Process-WithTimeout -ExeFilePath 'dsconfigad' `
											-Arguments $CmdArgs `
											-MaxWaitTimeSec 15
		
		### Fehlerauswertung
		## dsconfigad: Meldungen und Fehler
		# 	Alles OK
		# 	> keine antwort / ausgabe!

		$ResDomainJoinOK = $False
		If (Handle-Start-Process-Errors $ResProc '2jfsMl') {
			Log 4 "Fehler beim Hinzufügen vom MacOS zur Domäne" -Fore Red
			
			Switch ($ResCmdline) {
				# ❯ Falsches PW
				# dsconfigad: Invalid credentials supplied for binding to the server
				{ ([Regex]::Matches($_, '\b(invalid|credentials?|supplied|username|password)\b').Count -ge 2) } {
					Log 5 $ResCmdline -Fore DarkGray
					Log 4 'Domain-Join failed:' -Fore Red
					Log 4 'Benutzername oder Passwort ungültig' -Fore Red
					Break
				}
				# ❯ Bereits joined
				# dsconfigad: This computer is already 'bound' to Active Directory. You must 'unbind' with '-remove' first
				{ ([Regex]::Matches($_, '\b(already|bound)\b').Count -ge 2) } {
					Log 5 $ResCmdline -Fore DarkGray
					Log 4 'Domain-Join failed:' -Fore Red
					Log 4 'Computer ist schon domain-joined' -Fore Red
					Break
				}
				# ❯ Domäne wird nicht gefunden
				# dsconfigad: Node name wasn't found. (2000)
				{ ([Regex]::Matches($_, "\b(Node|wasn't|found|not)\b").Count -ge 3) } {
					Log 5 $ResCmdline -Fore DarkGray
					Log 4 'Domain-Join failed:' -Fore Red
					Log 4 "Domäne nicht gefunden: '$DomainName'" -Fore Red
					Break
				}
				# ❯ Hostname existiert schon
				# Computer account already exists! Bind to Existing? (y/n):
				# > -force angeben!
				{ ([Regex]::Matches($_, "\b(account|already|exists)\b").Count -ge 2) } {
					Log 5 $ResCmdline -Fore DarkGray
					Log 4 'Domain-Join failed:' -Fore Red
					Log 4 "Computer existiert schon in der Domäne!" -Fore Red
					Log 5 '-force nützen!' -Fore Magenta
					Break
				}
				default {
					If ($LastExitNOK) {
						Log 4 "Unbekannter Fehler beim Hinzufügen zur Domäne" -Fore Red
						Log 5 "Details: $ResCmdline" -Fore Magenta
					}
				}
			}

		} Else {
			# Alles OK
            $ResDomainJoinOK = $True
            Log 3 "MacOS erfolgreich zur Domäne hinzugefügt" -ForegroundColor Green
		}
	
    } Catch {
		Log 4 'Exception:' -Fore Red
		Log 5 "$($_.Exception.Message)" -Fore Magenta
        $ResDomainJoinOK = $False
    
	} Finally {
		# Sicheres Löschen des Passworts aus dem Speicher
        if ($Cred -and $Cred.SecurePW) {
            [System.Runtime.InteropServices.Marshal]::ZeroFreeBSTR([System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($Cred.SecurePW))
        }
        
        # Sicheres Löschen des Klartext-Passworts
        if ($DomainAdUserPW) {
            # Überschreiben mit leerer Zeichenfolge und dann mit $null
            $DomainAdUserPW = ""
            $DomainAdUserPW = $null
        }
        
        # Garbage Collection forcieren
        [System.GC]::Collect()		
    }
	Return $ResDomainJoinOK
}


# MacOS: Löscht den Domain-Join
Function Unjoin-MacOS-from-Domain {
    Param (
		# Optional ein Username
        [Object]$DomainAdUserName,
		# Optional ein PW
        [Object]$DomainAdUserPW
    )

    Log 0 'Entferne Mac aus der Domäne'

	### Checks
	If (-not (Is-Domain-Joined)) {
		Log 2 'Computer ist in keiner Domäne' -Fore Green
		Return $True
	}

	If (-Not (Assert-Root)) {
		Log 5 'Abbruch' -Fore Magenta
		Return $False
	}

	## UserName und PW prüfen
	$Cred = Resolve-Credentials -UserName $DomainAdUserName `
								-PWTextOrSecureStr $DomainAdUserPW `
								-MustBeDomainUser `
								-UsernamePrompt 'Domain-Join Benutzernamen (user@domain.com)' `
								-PWPrompt 'Domain-Join Passwort' `

    
    try {
		# Das PW im Klartext
		$DomainAdUserPW = ConvertFrom-SecureStringToPlainText -SecureString $Cred.SecurePW

		# Domäne trennen
        # 2>&1 stderr nach stdout
        $ResCmdline = & /usr/sbin/dsconfigad -remove -force -u "$($Cred.UserName)" -p "$DomainAdUserPW" 2>&1
		$LastExitOK, $LastExitNOK = Is-LastExitCode-AllOK $LastExitCode


		## Fehlerauswertung
		# » Falsches PW:
		# 	KEINE Meldung, auf dem MacOS wird der Join entfernt
		$HasNoErrors = $True
		Switch -regex ($ResCmdline) {
			# ❯ Ist gar nicht joined:
			#	dsconfigad: This computer is not Bound to Active Directory. You must bind it first.
			{ ([Regex]::Matches($_, '\b(not|bound)\b').Count -ge 2) } {
				$HasNoErrors = $False
				Log 4 'Computer ist nicht domain-joined' -For Red
				Break
			}
			default {	
				If ($LastExitNOK) {
					$HasNoErrors = $False
					Log 4 "Unbekannter Fehler Trennen der Domäne" -Fore Red
					Log 5 "Details: $ResCmdline" -Fore Magenta
				}
			}
		}

        # Überprüfe den Exit-Code des letzten Befehls und die Ausgabe auf Fehlermuster.
        If ($LastExitOK -and $HasNoErrors) {
            Log 1 'Mac erfolgreich aus der Domäne entfernt' -Fore Green
			Start-Sleep -Milliseconds 4500
            Return $True
        } else {
			Log 4 'Fehler beim Trennen der Domäne' -Fore Red
			Log 5 '- Netzverbindung zur Domäne OK?' -Fore Magenta
			Log 5 '- Credential OK?' -Fore Magenta
			Log 5 'Abbruch' -Fore Magenta
            return $False
        }

    } Catch {
		Log 4 'Exception:' -Fore Red
		Log 5 "$($_.Exception.Message)" -Fore Magenta
        Return $False

	} Finally {
		# Sicheres Löschen des Passworts aus dem Speicher
        if ($Cred -and $Cred.SecurePW) {
            [System.Runtime.InteropServices.Marshal]::ZeroFreeBSTR([System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($Cred.SecurePW))
        }
        
        # Sicheres Löschen des Klartext-Passworts
        if ($DomainAdUserPW) {
            # Überschreiben mit leerer Zeichenfolge und dann mit $null
            $DomainAdUserPW = ""
            $DomainAdUserPW = $null
        }
        
        # Garbage Collection forcieren
        [System.GC]::Collect()		
    }
}

#Endregion Domain-Join


#Region MacOS File Vault

# Liefert $True, wenn FileVault aktiviert ist
Function Is-FileVault-Enabled {
    try {
		# fdesetup status
		# 	FileVault is Off.
		# 	Deferred enablement appears to be active for user 'root'
		# 	> kann so wieder ausgeschaltet werden:
		# 		sudo fdesetup disable -user root
        $fileVaultStatus = (& fdesetup status) -join "`n"
        If ($fileVaultStatus -match "FileVault is On") {
            return $true
        } else {
            return $false
        }
    } catch {
        Write-Error "Fehler beim Abrufen des FileVault-Status: $($_.Exception.Message)"
        return $false
    }
}


#Endregion MacOS File Vault


#Region URLs

# !Ex
# 	Join-URLs 'https://eu.ninjarmm.com/', 'oauth/token/'
# » 
#	https://eu.ninjarmm.com/oauth/token
Function Join-URLs() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[String[]]$URLs,
		[Switch]$AddSlash
	)
	
	$Res = $Null
	ForEach($URL in $URLs) {
		$URL = $URL.Trim()
		If ($Null -eq $Res) {
			$Res = $URL.TrimEnd('/')
		} Else {
			$Res = '{0}/{1}' -f $Res, $URL.TrimEnd('/')
		}
	}
	
	If ($AddSlash) {
		Return $Res + '/'
	} Else {
		Return $Res
	}
}

#Endegion URLs


#Region NinjaOne

# Authentifiziert sich am NinjaOne REST API
# Liefert den AuthHeader für folgeaufrufe zurück
# -Dbg zeigt Details zum Auth-Token an
# 
# !M
# https://app.NinjaOne.com/apidocs-beta/authorization/overview
# 250415
Function Auth-NinjaOne() {
	Param(
		[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]

		# Zugriffs-Scopes
		# Wenn nicht definiert, werden alle gewährt
		#  @('monitoring', 'management', 'control')
		# !Ex https://github.com/homotechsual/NinjaOne/blob/bfbc4832473fef510fe11b5a759f2d91f529b999/Source/Public/PSModule/Connect/Connect-NinjaOne.ps1#L142
		[ValidateSet('monitoring', 'management', 'control')]
		[String[]]$Scopes,
		[Switch]$Dbg
	)
	
	If ($Dbg) {
		Log 4 'Auth-NinjaOne()' -Fore DarkGray
	}
	
	## Init
	# Allenfalls alle Scopes definieren
	If ($Null -eq $Scopes) {
		$Scopes = @('monitoring', 'management', 'control')
	}
	
	
	# Sie müssen alle kleingeschrieben sein
	$Scopes = $Scopes | % { $_.ToLower() }
	# Write-Host $Scopes | Out-String
	
	
	## Config
	$AuthURL = Join-URLs @($Script:NinjaOneBaseURL, 'oauth/token/')

	$Body = @{
		grant_type = 'client_credentials'
		client_id = $Script:NinjaOneClientID
		client_secret = $Script:NinjaOneClientSecret
		redirect_uri = 'https://localhost'
		scope = ($Scopes -join ' ')
	}
	
	
	## Prepare

	$AuthHeaders = New-Object 'System.Collections.Generic.Dictionary[[String],[String]]'
	# Muss kleingeschrieben sein!
	$AuthHeaders.Add('Accept', 'application/json')
	$AuthHeaders.Add('Content-Type', 'application/x-www-form-urlencoded')
	
	
	## Main

	$AuthToken = Invoke-RestMethod -Uri $AuthURL -Method POST `
												-Headers $AuthHeaders `
												-Body $body
	# Return $AuthToken

	If ($Dbg) {
		Log 4 ($AuthToken | Out-String)
	}

	$AccessToken = $AuthToken | Select-Object -ExpandProperty 'access_token' -EA 0

	$AuthHeader = New-Object 'System.Collections.Generic.Dictionary[[String],[String]]'
	$AuthHeader.Add('accept', 'application/json')
	$AuthHeader.Add('Authorization', "Bearer $AccessToken")

	Return $AuthHeader
}


# Sucht das NinjaOne-Objekt für ein Gerät
Function Get-NinjaOne-Device-by-SystemName() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[Object]$AuthHeader,
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=1)]
		[String]$SystemName,
		[Switch]$Dbg
	)
	
	# Config
	$Limit = 10
	
	# https://app.NinjaOne.com/apidocs-beta/core-resources/operations/search
	$SearchDevicesURL = 'v2/devices/search'

	$ApiURL = Join-URLs @($Script:NinjaOneBaseURLAPI, $SearchDevicesURL)
	
	$QryURL = Join-URLs @($ApiURL, `
								('?q={0}&limit={1}' -f [URI]::EscapeDataString($SystemName), $Limit))
	
	If ($Dbg) { 
		log 4 'QryURL'
		log 5 $QryURL
	}
	
	
	# Die Geräte suchen
	$Devices = Invoke-RestMethod -Uri $QryURL -Method GET -Headers $AuthHeader
	# Return $Devices

	# Das Gerät filtern
	$MyDevice = $Devices.Devices | ? systemName -eq $SystemName
	# $ID = $MyDevice.id
	# Write-Host "ID: $ID"
	Return $MyDevice
}


# Sucht von einem NinjaOne Geräte-Objekt
# alle Felder, die es hat
Function Get-NinjaOne-Device-Fields() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[Object]$AuthHeader,
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=1)]
		[Int]$DeviceID,
		[Switch]$Dbg
	)
	
	## Config

	# https://app.NinjaOne.com/apidocs-beta/core-resources/operations/search
	$GetURL = "v2/device/$DeviceID/custom-fields"

	$ApiURL = Join-URLs @($Script:NinjaOneBaseURLAPI, $GetURL)
	If ($Dbg) { 
		Log 4 'ApiURL'
		Log 5 $ApiURL
	}

	# Die Fields auslesen
	$Fields = Invoke-RestMethod -Uri $ApiURL -Method GET -Headers $AuthHeader
	Return $Fields
}


# Setzt den Wert eines Custom Field
Function Set-NinjaOne-Device-Field() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[Object]$AuthHeader,
	
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=1)]
		[Int]$DeviceID,

		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName)]
		[String]$FieldName = 'ssfmahandynr',

		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName)]
		[String]$FieldValue,
		[Switch]$Dbg
	)
	
	## Config

	# https://app.NinjaOne.com/apidocs/?links.active=core#/devices/getNodeCustomFields
	$GetURL = "v2/device/$DeviceID/custom-fields"

	$ApiURL = Join-URLs @($Script:NinjaOneBaseURLAPI, $GetURL)
	If ($Dbg) { 
		Log 4 'ApiURL'
		Log 5 $ApiURL
	}
	
	$BodyProps = @{}
	$BodyProps.Add($FieldName, $FieldValue)
	# Return $BodyProps
	# Return (ConvertTo-Json -InputObject $BodyProps -Depth 100)

	# Die Fields Setzen
	$Res = Invoke-RestMethod -Uri $ApiURL -Method Patch `
									-Headers $AuthHeader `
									-ContentType 'application/json' `
									-Body (ConvertTo-Json -InputObject $BodyProps -Depth 100)

	Return $Res
}

#Endregion NinjaOne



#Region Netzwerk


# Löscht den DNS Cache
Function Clear-DNS-Cache() {
	If (-Not (Assert-Root)) {
		Log 5 'Abbruch' -Fore Magenta
		Return $False
	}

	# Leert den DNS Cache
	# https://www.macworld.com/article/671688/how-to-clear-dns-cache-macos.html
	# sudo dscacheutil -flushcache; sudo killall -HUP mDNSResponder
	& dscacheutil -flushcache
	& killall -HUP mDNSResponder
}


# Liefert die erste IPv4 Adresse für eine Domäne
# !Ex, liefert: [System.Net.IPAddress]
# $oIPv4 = Resolve-Domain-get-IPv4 -Domain 'akros.ch'
# $oIPv4.IPAddressToString
Function Resolve-Domain-get-IPv4($Domain) {
	# Punkt am Ende erzwingen
	$Domain = ('{0}.' -f $Domain.Trim().TrimEnd('.'))

	Try {
		$oHost = [System.Net.Dns]::GetHostEntry($Domain)
		# Erste IPv4
		$oFirstIp = $oHost.AddressList | ? { $_.AddressFamily -eq 'InterNetwork' } | Select-Object -First 1
		
		Return $oFirstIp
	} Catch {
		Return $Null
	}
}


# Prüft, ob die Domäne mit einer LAN IP-Adresse aufgelöst wird
Function Is-Computer-in-Domain-LAN($Domain) {
	$oIPv4 = Resolve-Domain-get-IPv4 -Domain $Domain
	If (-not $oIPv4) { Return $False }
	
	$FirstOctett = $oIPv4.IPAddressToString.Split('.')[0]
	# Sind wir im 10.x Netzwerk?
	Return ($FirstOctett -eq 10)
}

#Endregion Netzwerk


#Region NinjaOne Funcs

# Gibt die automatischen Variablen von NinjaOne aus
# NINJA_AGENT_NODE_ID ist die DeviceID
# 
# !Ex
# 	NINJA_AGENT_MACHINE_ID: V5;C69HWWJ1Q1;84:2F:57:3B:2B:41
# 	NINJA_AGENT_NODE_ID: 167
# 	NINJA_AGENT_VERSION_INSTALLED: 8.0.3261
# 	NINJA_COMPANY_NAME: Noser Engineering AG
# 	NINJA_DATA_PATH: /Applications/NinjaRMMAgent/programdata
# 	NINJA_EXECUTING_PATH: /Applications/NinjaRMMAgent/programfiles
# 	NINJA_LOCATION_ID: 20
# 	NINJA_LOCATION_NAME: (Kein Standort)
# 	NINJA_ORGANIZATION_ID: 2
# 	NINJA_ORGANIZATION_NAME: AKROS AG
# 	NINJA_PATCHER_VERSION_INSTALLED: 8.0.3261
# 	NINJA_SECURE_CHAIN_TOKEN: d61c462d-182b-4df7-9e4c-603e287c1f15
Function Print-NinjaOne-AutomaticVars {
	Log 0 'Automatische Variablen von NinjaOne'
	# Alle automatischen NinjaOne Variablen ausgeben
	Get-ChildItem Env: | Where-Object { $_.Name -like 'NINJA*' } | ForEach-Object { Log 1 "$($_.Name): $($_.Value)" }	
}


# Authentifiziert sich am NinjaOne REST API
# Liefert den AuthHeader für folgeaufrufe zurück
# -Dbg zeigt Details zum Auth-Token an
# 
# !M
# https://app.NinjaOne.com/apidocs-beta/authorization/overview
# 250415
Function Auth-NinjaOne() {
	Param(
		[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]

		# Zugriffs-Scopes
		# Wenn nicht definiert, werden alle gewährt
		#  @('monitoring', 'management', 'control')
		# !Ex https://github.com/homotechsual/NinjaOne/blob/bfbc4832473fef510fe11b5a759f2d91f529b999/Source/Public/PSModule/Connect/Connect-NinjaOne.ps1#L142
		[ValidateSet('monitoring', 'management', 'control')]
		[String[]]$Scopes,
		[Switch]$Dbg
	)
	
	
	## Init
	# Allenfalls alle Scopes definieren
	If ($Null -eq $Scopes) {
		$Scopes = @('monitoring', 'management', 'control')
	}
	
	
	# Sie müssen alle kleingeschrieben sein
	$Scopes = $Scopes | % { $_.ToLower() }
	# Write-Host $Scopes | Out-String
	
	
	## Config
	$AuthURL = Join-URLs @($Script:NinjaOneBaseURL, 'oauth/token/')

	$Body = @{
		grant_type = 'client_credentials'
		client_id = $Script:NinjaOneClientID
		client_secret = $Script:NinjaOneClientSecret
		redirect_uri = 'https://localhost'
		scope = ($Scopes -join ' ')
	}
	
	
	## Prepare

	$AuthHeaders = New-Object 'System.Collections.Generic.Dictionary[[String],[String]]'
	# Muss kleingeschrieben sein!
	$AuthHeaders.Add('Accept', 'application/json')
	$AuthHeaders.Add('Content-Type', 'application/x-www-form-urlencoded')
	
	
	## Main

	$AuthToken = Invoke-RestMethod -Uri $AuthURL -Method POST `
												-Headers $AuthHeaders `
												-Body $body
	# Return $AuthToken

	If ($Dbg) {
		Log 4 ($AuthToken | Out-String)
	}

	$AccessToken = $AuthToken | Select-Object -ExpandProperty 'access_token' -EA 0

	$AuthHeader = New-Object 'System.Collections.Generic.Dictionary[[String],[String]]'
	$AuthHeader.Add('accept', 'application/json')
	$AuthHeader.Add('Authorization', "Bearer $AccessToken")

	Return $AuthHeader
}


# Sucht das NinjaOne-Objekt für ein Gerät
Function Get-NinjaOne-Device-by-SystemName() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[Object]$AuthHeader,
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=1)]
		[String]$SystemName,
		[Switch]$Dbg
	)
	
	# Config
	$Limit = 10
	
	# https://app.NinjaOne.com/apidocs-beta/core-resources/operations/search
	$SearchDevicesURL = 'v2/devices/search'

	$ApiURL = Join-URLs @($Script:NinjaOneBaseURLAPI, $SearchDevicesURL)
	
	$QryURL = Join-URLs @($ApiURL, `
								('?q={0}&limit={1}' -f [URI]::EscapeDataString($SystemName), $Limit))
	
	If ($Dbg) { 
		Log 4 'QryURL'
		Log 5 $QryURL
	}
	
	
	# Die Geräte suchen
	$Devices = Invoke-RestMethod -Uri $QryURL -Method GET -Headers $AuthHeader
	# Return $Devices

	# Das Gerät filtern
	$MyDevice = $Devices.Devices | ? systemName -eq $SystemName
	# $ID = $MyDevice.id
	# Write-Host "ID: $ID"
	Return $MyDevice
}


# Sucht von einem NinjaOne Geräte-Objekt
# alle Felder, die es hat
Function Get-NinjaOne-Device-Fields() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[Object]$AuthHeader,
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=1)]
		[Int]$DeviceID,
		[Switch]$Dbg
	)
	
	## Config

	# https://app.NinjaOne.com/apidocs-beta/core-resources/operations/search
	$GetURL = "v2/device/$DeviceID/custom-fields"

	$ApiURL = Join-URLs @($Script:NinjaOneBaseURLAPI, $GetURL)
	If ($Dbg) { 
		Log 4 'ApiURL'
		Log 5 $ApiURL
	}

	# Die Fields auslesen
	$Fields = Invoke-RestMethod -Uri $ApiURL -Method GET -Headers $AuthHeader
	Return $Fields
}


# Setzt den Wert eines Custom Field
Function Set-NinjaOne-Device-Field() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[Object]$AuthHeader,
	
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=1)]
		[Int]$DeviceID,

		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName)]
		[String]$FieldName_ssfClientTasksData = 'ssfmahandynr',

		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName)]
		[String]$FieldValue,
		[Switch]$Dbg
	)
	
	## Config

	# https://app.NinjaOne.com/apidocs/?links.active=core#/devices/getNodeCustomFields
	$GetURL = "v2/device/$DeviceID/custom-fields"

	$ApiURL = Join-URLs @($Script:NinjaOneBaseURLAPI, $GetURL)
	If ($Dbg) { 
		Log 4 'ApiURL'
		Log 5 $ApiURL
	}
	
	$BodyProps = @{}
	$BodyProps.Add($FieldName_ssfClientTasksData, $FieldValue)
	# Return $BodyProps
	# Return (ConvertTo-Json -InputObject $BodyProps -Depth 100)

	# Die Fields Setzen
	$Res = Invoke-RestMethod -Uri $ApiURL -Method Patch `
									-Headers $AuthHeader `
									-ContentType 'application/json' `
									-Body (ConvertTo-Json -InputObject $BodyProps -Depth 100)

	Return $Res
}


#Region NinjaOne, Funcs für ssfClientTasksData


## Parser für den Inhalt von ssfClientTasksData

# Parst von *einem* Block die Zeilen und erzeugt ein PSCustomObject
Function Parse-Block-Lines() {
    Param (
        [String[]]$BlockLines,
		[Switch]$Dbg
    )

	# Sicherstellen, dass wir wirklich Zeilen haben
	$BlockLines = $BlockLines -split "`n" | ? { Has-Value $_ }

	If ($Dbg) {
		Log 4 '2jbdjK Parse-Block-Lines(), $BlockLines:' -Fore Magenta
		Log 5 ($BlockLines | Out-String)
	}

    ## Header-Zeile analysieren
    $headerLine = $BlockLines[0]
	If ($Dbg) {
		Log 4 'Parse-Block-Lines(), $headerLine:' -Fore Magenta
		Log 5 ($headerLine | Out-String)
	}

    $headerPattern = '^(?<Timestamp>\d{6}\s+\d{4,6})\s+(?<Key>\S+)\s*(?<KeyAddon>.*)$'

    If ($headerLine -notmatch $headerPattern) {
        Log 4 "Ungültige Headerzeile: $headerLine" -Fore Red
		Return $Null
    }

    $timestampRaw = $matches['Timestamp']
    $key          = $matches['Key']
    $KeyAddon      = $matches['KeyAddon']

    # Konvertierung in [datetime]
    $datetime = $null
    If ($timestampRaw -match '^(?<date>\d{6})\s+(?<time>\d{4,6})$') {
        $datePart = $matches['date']
        $timePart = $matches['time']

        $yy = [int]$datePart.Substring(0, 2)
        $mm = [int]$datePart.Substring(2, 2)
        $dd = [int]$datePart.Substring(4, 2)

        $hh = [int]$timePart.Substring(0, 2)
        $mi = [int]$timePart.Substring(2, 2)
        $ss = If ($timePart.Length -ge 6) { [int]$timePart.Substring(4, 2) } else { 0 }

        $year = If ($yy -lt 100) { 2000 + $yy } else { $yy }

        $datetime = Get-Date -Year $year -Month $mm -Day $dd -Hour $hh -Minute $mi -Second $ss
    }

    # Comment
	$Comment = ''
	# Vxxx
	$Version = $null
    # Key-Value Werte
	$KeyValueData = @{}
    $AdditionalLines = @()

    ForEach ($Line in $BlockLines[1..($BlockLines.Count - 1)]) {
		$Line = $Line.Trim()
        If ($Line -match '^V\d{3}$' -and -not $Version) {
            $Version = $Line
        }
        Elseif ($Line -match '^\s*(?<PKey>[^:]+)\s*:\s*(?<PValue>.+)$') {
            $keyName = $matches['PKey'].Trim()
            $value = $matches['PValue'].Trim()
            $KeyValueData[$keyName] = $value
        }
        Elseif ($Line -match '^#(?<Comment>.*)') {
			$Comment = $Matches['Comment'].Trim()
		}		
        Else {
            $AdditionalLines += $Line
        }
    }

    # Objekt erstellen und Root-Eigenschaften zusammenbauen
    $baseObject = [Ordered]@{
        Processed   = $False
        Key         = $key
        KeyAddon    = $KeyAddon
		Comment		= $Comment
        Version     = $Version
        Timestamp   = $datetime
    }

    # Properties hinzufügen
    ForEach ($kvp in $KeyValueData.GetEnumerator()) {
        $baseObject[$kvp.Key] = $kvp.Value
    }
	
	$baseObject['AdditionalLines'] = $AdditionalLines
	$baseObject['OriLines'] = $BlockLines

    return [PSCustomObject]$baseObject
}



# Parst alle Blöcke und erzeugt für jeden je ein PSCustomObject
# !Ex
# 	$Blöcke = Get-TextBlocks -Text BlockLines
# 	$oBlöcke = Parse-ClientTasksData -Blocks $Blöcke
Function Parse-ClientTasksData {
    Param (
        [String[]]$Text,
		[Switch]$Dbg
    )

	# String in Linien Splitten und leere Zeilen entfernen
	$Lines = $Text -split "`n" | ? { Has-Value $_ }
	
	If ($Dbg) {
		Log 4 'Parse-ClientTasksData(), $Lines:' -Fore Magenta
		Log 5 ($Lines | Out-String)
	}

	$Blöcke = @(Get-TextBlocks -Lines $Lines -Dbg:$Dbg)

	If ($Dbg) {
		Log 4 'Parse-ClientTasksData(), $Blöcke:' -Fore Magenta
		For($idx = 0; $idx -lt $Blöcke.Count; $idx++) {
			Log 5 "Block: $idx"
			Log 6 ($Blöcke[$idx] | Out-String)
		}
	}

	$Res = @()
    ForEach ($BlockLines in $Blöcke) {
		$Res += Parse-Block-Lines -BlockLines $BlockLines -Dbg:$Dbg
	}
	Return $Res
}



# Trennt die Zeilen in Blöcke, e.g.:
# 	$Text = @'
# 		250616 1246 LocalUserPWChanged lk flaksjf lkas df
# 		V002
# 		Prop1: x
# 		Prop2: y
# 		Prop3: y
# 		Testzeile 1
# 		Testzeile 2
#	'@
Function Get-TextBlocks {
    param (
        [String[]]$Lines,
		[Switch]$Dbg
    )
	
	## Init
	# Sicherstellen, dass wir wirklich Zeilen haben
	$Lines = $Lines -split "`n" | ? { Has-Value $_ }

	If ($Dbg) {
		Log 4 'Get-TextBlocks(), $Lines:' -Fore Magenta
		Log 5 ($Lines | Out-String)
	}

	## Config
    # Regex für den Start eines Blocks
    $startPattern = '^\d{6}\s+\d{4,6}\s+\S.+$'

	## Main
    $blocks = @()
    $currentBlock = @()

    foreach ($Line in $Lines -split "`n") {
        If ($Line.Trim() -eq '') {
            continue # Leerzeilen vollständig ignorieren
        }

        If ($Line -match $startPattern) {
            If ($currentBlock.Count -gt 0) {
                $blocks += ,@($currentBlock)
                $currentBlock = @()
            }
        }

        $currentBlock += $Line
    }

    # Letzten Block hinzufügen, falls vorhanden
    If ($currentBlock.Count -gt 0) {
        $blocks += ,@($currentBlock)
    }

	If ($Dbg) {
		Log 4 'Get-TextBlocks(), $blocks:' -Fore Magenta
		For($idx = 0; $idx -lt $blocks.Count; $idx++) {
			Log 5 "Block: $idx"
			Log 6 ($blocks[$idx] | Out-String)
		}
	}

    Return $blocks
}


# Konvertiert nicht verarbeitete Blöcke wieder zu Zeilen
# Um sie wieder ins ssfClientTasksData zu schreiben, wenn wir Fehler haben
# In -Comment kann der Fehler mitgegeben werden
Function Get-Block-unprocessed-AsLines {
    Param (
        [PSCustomObject[]]$Blocks,
        [String]$Comment
    )
	
	# Neue Liste erzeugen
	$Res = @()

	ForEach ($Block in $Blocks | ? Processed -eq $False) {
		$Idx = -1
		ForEach ($Line in $Block.OriLines) {
			$Idx++
			$Res += [String]$Line
			If ($Idx -eq 0) {
				$Res += ('# {0}' -f $Comment.TrimStart('#').Trim())
			}
		}
		# Leere Zeile als Trenner
		$Res += ''
	}
	Return $Res -join "`n"
}


# Trennt die Zeilen in Blöcke, e.g.:
# 	$Text = @'
# 		250616 1246 LocalUserPWChanged lk flaksjf lkas df
# 		V002
# 		Prop1: x
# 		Prop2: y
# 		Prop3: y
# 		Testzeile 1
# 		Testzeile 2
#	'@
Function Get-TextBlocks {
    param (
        [String[]]$Lines,
		[Switch]$Dbg
    )
	
	## Init
	# Sicherstellen, dass wir wirklich Zeilen haben
	$Lines = $Lines -split "`n" | ? { Has-Value $_ }

	If ($Dbg) {
		Log 4 'Get-TextBlocks(), $Lines:' -Fore Magenta
		Log 5 ($Lines | Out-String)
	}

	## Config
    # Regex für den Start eines Blocks
    $startPattern = '^\d{6}\s+\d{4,6}\s+\S.+$'

	## Main
    $blocks = @()
    $currentBlock = @()

    foreach ($Line in $Lines -split "`n") {
        If ($Line.Trim() -eq '') {
            continue # Leerzeilen vollständig ignorieren
        }

        If ($Line -match $startPattern) {
            If ($currentBlock.Count -gt 0) {
                $blocks += ,@($currentBlock)
                $currentBlock = @()
            }
        }

        $currentBlock += $Line
    }

    # Letzten Block hinzufügen, falls vorhanden
    If ($currentBlock.Count -gt 0) {
        $blocks += ,@($currentBlock)
    }

	If ($Dbg) {
		Log 4 'Get-TextBlocks(), $blocks:' -Fore Magenta
		For($idx = 0; $idx -lt $blocks.Count; $idx++) {
			Log 5 "Block: $idx"
			Log 6 ($blocks[$idx] | Out-String)
		}
	}

    Return $blocks
}


# Konvertiert nicht verarbeitete Blöcke wieder zu Zeilen
# Um sie wieder ins ssfClientTasksData zu schreiben, wenn wir Fehler haben
# In -Comment kann der Fehler mitgegeben werden
Function Get-Block-unprocessed-AsLines {
    Param (
        [PSCustomObject[]]$Blocks,
        [String]$Comment
    )
	
	# Neue Liste erzeugen
	$Res = @()

	ForEach ($Block in $Blocks | ? Processed -eq $False) {
		$Idx = -1
		ForEach ($Line in $Block.OriLines) {
			$Idx++
			$Res += [String]$Line
			If ($Idx -eq 0) {
				$Res += ('# {0}' -f $Comment.TrimStart('#').Trim())
			}
		}
		# Leere Zeile als Trenner
		$Res += ''
	}
	Return $Res -join "`n"
}

#Endregion NinjaOne, Funcs für ssfClientTasksData


#Region NinjaOne, Funcs für das Custom Field: ssfClientTasksData


## Prozessor für die einzelnen ssfClientTasksData Elemente

# Speichert den FileVault Recovery Key
# Block-Struktur:
# 	250616 094302 SaveFileVaultRecoveryKey
#	V001
#	RecoveryKey: x
# 	
Function Save-FileVaultRecoveryKey() {
	Param (
        [PSCustomObject]$oBlock,
		[Object]$AuthHeader,
		[String]$DeviceID,
		[Switch]$Dbg
    )
	
	Switch($oBlock.Version) {
		'V001' {
			If ($Dbg) { 
				Log 4 'Save-FileVaultRecoveryKey(): V001' -Fore Magenta
				Log 5 ('FieldName : {0}' -f 'ssfMacOsFileVaultKey')
				Log 5 ('FieldValue: {0}' -f $oBlock.RecoveryKey)
			}
			# Zeile ergänzen
			
			$RecoveryData = ("{0}:`n{1}" -f (Get-Date).ToString('yyMMdd HHmm'), $oBlock.RecoveryKey)
			
			& $SetNinjaOneDeviceFieldSecure_ps1 -DeviceIDs $DeviceID `
			-FieldName 'ssfMacOsFileVaultKey' `
			-FieldValue $RecoveryData `
			-Action 'AddNewLine' `
			-AuthHeader $AuthHeader
		}
		
		Default {
			Log 4 '2jbb5z Fehler in: Save-FileVaultRecoveryKey()' -NewLineBefore -Fore Red 
			Log 5 ('Version unbekannt: {0}' -f $oBlock.Version)
		}
	}
	
}


# Verarbeitet alle Elemente im Feld ssfClientTasksData
Function Transfer-ssfClientTasksData() {
	Param (
        [PSCustomObject[]]$oBlöcke,
		[Object]$AuthHeader,
		[String]$DeviceID,
		[Switch]$Dbg
    )

	ForEach($oBlock in $oBlöcke) {
		If ($Dbg) {
			Log 4 'Block:' -Fore Magenta
			Log 5 $oBlock
		}
		Switch ($oBlock.Key) {
			# Den FileFault RecoveryKey
			'SaveFileVaultRecoveryKey' {
				If ($Dbg) {
					Log 4 'Verarbeite: SaveFileVaultRecoveryKey' -Fore Magenta
				}
				Log 2 'Speichere den FileVault Recovery Key'
				Save-FileVaultRecoveryKey -oBlock $oBlock `
										  -AuthHeader $AuthHeader `
										  -DeviceID $DeviceID `
										  -Dbg:$Dbg
				$oBlock.Processed = $True				
			}
			
			Default {
				If ($Dbg) {
					Log 4 "Switch-Fehler!: $($oBlock.Key)" -Fore Red
				}
				Log 4 '2jbb5y Fehler in: Transfer-ssfClientTasksData()' -NewLineBefore -Fore Red
				Log 5 ('Unbekannter Key: {0}' -f $oBlock.Key)
			}
		}
	}
}

#Endregion NinjaOne, Funcs für das Custom Field: ssfClientTasksData


#Region NinjaOne, Funcs für das Custom Field: SaveFileVaultRecoveryKey


# Liefert den Record für das Custom Field:
#
# 250616 1707 SaveFileVaultRecoveryKey
# V001
# RecoveryKey: x1
Function New-ssfClientTasksData-Record-SaveFileVaultRecoveryKey() {
	Param (
		[Parameter(Mandatory)]
		[String]$FileVaultRecoveryKey
	)
	
	# Config
	$RecKey = 'SaveFileVaultRecoveryKey'

	$Rec = @"
$((Get-Date).ToString('yyMMdd HHmm')) $RecKey
V001
RecoveryKey: $FileVaultRecoveryKey
"@	
	
	Return $Rec
}


# Erzeugt im NinjaOne Custom Field ssfClientTasksData einen Record
Function Create-ssfClientTasksData-Rec() {
	Param(
		[Parameter(Mandatory)]
		[String]$FileVaultRecoveryKey
	)

	# Den Record erzeugen
	$RecData = New-ssfClientTasksData-Record-SaveFileVaultRecoveryKey `
				-FileVaultRecoveryKey $FileVaultRecoveryKey
	
	# Dbg
	# Log 4 "`nRecData:"
	# Log 5 ($RecData | Out-String)
	
	# Den aktuellen Wert lesen
	# !9 Liefert ein String[]
	$WertOri = & "$Script:NinjaCliPath" get "$Script:FieldName_ssfClientTasksData"
	$WertOri = $WertOri -Join "`n"
	
	
	# Den Rec ergänzen
	$WertNeu = ("{0}`n`n{1}" -f $WertOri, $RecData)
	# Dbg
	# Log 4 "`nWertNeu:"
	# Log 5 ($WertNeu | Out-String)
	

	# Pfad zum NinjaRMM-Agenten
	& "$Script:NinjaCliPath" set "$Script:FieldName_ssfClientTasksData" "$WertNeu"
	
}


# Prüft, dass das NinjaOne Custom Field den FileVaultRecoveryKey hat
# Liefert $True, wenn der Key vorhanden ist
Function Assert-ssfClientTasksData-has-FileVaultRecoveryKey() {
	Param(
		[Parameter(Mandatory)]
		# eg EDK4-JR4M-LRW2-GHQQ-47PL-39EQ
		[String]$FileVaultRecoveryKey,
		[Int]$RetryMs = 8000,
		[Int]$TimeoutSec = 120
	)

	Log 2 'Prüfe, ob NinjaOne den FileVault Recovery Key hat' -NewLineBefore

	$StartTime = Get-Date
	$FileVaultKeyFound = $False
	Do {
		
		# Den aktuellen Wert lesen
		# !9 Liefert ein String[]
		$WertOri = & "$Script:NinjaCliPath" get "$Script:FieldName_ssfClientTasksData"
		$WertOri = $WertOri -Join "`n"
		
		# Wenn wir einen Wert erhalten haben
		If (-not [string]::IsNullOrWhiteSpace($WertOri)) {
			$FileVaultKeyFound = $WertOri.IndexOf($FileVaultRecoveryKey, [StringComparison]::OrdinalIgnoreCase) -ge 0
		}

		# Wenn der VaultKey fehlt, warten
		If ($FileVaultKeyFound) {
			Log 3 '✅ OK, gefunden' -Fore Green
		} Else {
			Log 3 ('Noch nicht gefunden, warte {0}ms' -f $RetryMs) -Fore Magenta
			Start-Sleep -Milliseconds $RetryMs
		}
		
		$Elapsed = (Get-Date) - $StartTime
		
	} While (-not $FileVaultKeyFound -and $Elapsed.TotalSeconds -lt $TimeoutSec)
	
	Return $FileVaultKeyFound
}

#Endregion NinjaOne, Funcs für das Custom Field: SaveFileVaultRecoveryKey

#Endregion NinjaOne Funcs


#Region GetHelp Hilfsfunktionen

# Liefert $True, wenn $TestString vermutlich ein PowerShell Script / Kommando ist,
# sonst ist es vermutlich ein normaler Text
Function Is-PowerShell-Command($TestString) {
	$Token = $Null
	$ParseError = $Null

	$Res = [System.Management.Automation.Language.Parser]::ParseInput($TestString, [ref]$Token, [ref]$ParseError)

	# Wenn wir anderes als nur Tokens haben, haben wir vermutlich ein Script
	If (($Token | ? { ($_.GetType()).Name -ne 'Token' }).Count -gt 0) {
		# Vermutlich ein Script
		$True
	} Else {
		# Vermutlich Text
		$False
	}
}


# Analysier das Code-Beispiel
# Syntax, e.g.
# 	i3lines ! .\Get-My-ActivityStream.ps1 -GetCurrentMonth -JiraTicketsInteractive -npid xxxx
#
# i3lines	» Das Code-Beispiel hat 3 Zeilen
# !			» Das Code-Beispiel ist direkt ausführbar
Function Analyze-CodeExample($Code, $ScriptFileName, $ScriptFullName) {

	# Syntax:
	# i<n>Lines ! .\…
	$Rgx = '(?<NoOfLinesInfo>i(?<NoOfLines>\d+)lines)|(?<IsExecutable>\!\s+)'

	# 250708, MacOS
	$NoOfLines = 1
	$MyCode = $Code
	$IsDirectExecutable = $false
	$MyMatches = $MyCode | Select-String $Rgx -AllMatches
	While ($MyMatches) {
		$MyMatches | % {
			# Haben wir das Element gefunden, das die Anzahl der Zeilen angibt?
			$NoOfLinesInfo = $_.Matches[0].Groups['NoOfLinesInfo']
			If ($NoOfLinesInfo.Success) {
				# Vom Code-Beispiel den gefundenen Parameter entfernen
				$MyCode = $MyCode.Remove($NoOfLinesInfo.Index, $NoOfLinesInfo.Length).Trim()
				$NoOfLines = $_.Matches[0].Groups['NoOfLines'].Value
				# Pipeline stoppen: Sicherstellen, dass wir neu den Regex ausführen
				Return
			}

			# Haben wir das Element gefunden, das angibt, dass das Beispiel direkt ausführbar ist?
			$IsExecutable = $_.Matches[0].Groups['IsExecutable']
			If ($IsExecutable.Success) {
				# Vom Code-Beispiel den gefundenen Parameter entfernen
				$MyCode = $MyCode.Remove($IsExecutable.Index, $IsExecutable.Length).Trim()
				$IsDirectExecutable = $true
				# Pipeline stoppen: Sicherstellen, dass wir neu den Regex ausführen
				Return
			}

		}
		# Den Rest wieder Parsen
		$MyMatches = $MyCode | Select-String $Rgx -AllMatches
	}

  	# 250708 Code-Zeilen
	$MyCodeLines = $MyCode -Split "`n"
	# Von den Code-Zeilen nur die ersten $NoOfLines Zeilen behalten
	$OnlyCodeLines = ($MyCodeLines[0..($NoOfLines - 1)]) -join "`n"
	$CodeCommentLines = ($MyCodeLines[$NoOfLines..($MyCodeLines.Count - 1)]) -join "`n"

	[PSCustomObject][Ordered]@{
		FullCommandLineOri = $OnlyCodeLines
		# Den Script-Namen mit dem vollen Pfad ersetzen
		FullCommandLine	 = ($OnlyCodeLines.Replace(".\$ScriptFileName", $ScriptFullName))
		# Nur die Kommandozeilen-Parameter
		Parameters		    = ($OnlyCodeLines.Replace(".\$ScriptFileName", '')).Trim()
		IsDirectExecutable = $IsDirectExecutable
		NoOfLines			= $NoOfLines
		CodeCommentLines  		= $CodeCommentLines
	}
}


# Zeigt eine nützliche Formatierung von Get-Help -Examples
# 250711
#	Wechsel 
#		von: Write-Host 
#		auf: Print-Line
Function Show-Help($ScriptFullName, $HelpText, $RecognizeCode = $false) {
	$ScriptFileName = [IO.Path]::GetFileName($ScriptFullName)

	# Die Synopsis (den Header) anzeigen
	Print-Line -Msg 'Script-Funktion' -Style H1
	Print-Line -Msg $HelpText.Synopsis
	Print-Line 

	$ExNo = 1
	ForEach ($Example In $HelpText.Examples.Example) {
		# Jedes Beispiel anzeigen
		# Den Titel
		Print-Line -Msg ("Example #{0}" -f ($ExNo++)) -Style H2

		## Das Code-Beispiel analysieren
		$Code = $Example.Code

		$CodeInfo = Analyze-CodeExample $Code $ScriptFileName $ScriptFullName
		If ($CodeInfo.IsDirectExecutable) {
			Print-Line -Msg "! $($CodeInfo.FullCommandLine)" -Style H3
		} Else {
			Print-Line -Msg $CodeInfo.FullCommandLine -Style H3
		}

		### Die Kpommentare anzeigen
		## Die Kommentare, die der PS-Parser selber fand, scheint in PS5 und PS7 unterschieldich zu sein
		# Die Leeren Zeilen am Ende löschen
		$Comments = $Example.Remarks.Text -replace "(?s)`n\s*$"
		# Die ersten Zeilen im Kommentar allenfalls als Code anzeigen
		$CommentStarted = $False
		$CodeLines = $CodeInfo.NoOfLines -1 # Die erste Zeile wurde bereits angezeigt
		ForEach ($Comment In $Comments) {
			# Die Comments in Zeilen aufteilen und leere Zeilen ignorieren
			$Lines = ($Comment -split "`r`n|`r|`n") | ? { -not [String]::IsNullOrEmpty($_) }
			ForEach ($Line In $Lines) {
				# Wir haben noch eine Code-Zeile
				If (($CodeLines--) -gt 0) {
					# Der Kommentar ist vermutlich ein PowerShell Befehl
					Print-Line -Msg $Line.Replace(".\$ScriptFileName", $ScriptFullName) -Style H3
				} Else {
					# Der Kommentar ist vermutlich Text
					If ($CommentStarted -eq $False) {
						$CommentStarted = $True
					}
					Print-Line -Msg $Line
				}
			}
		}


		## Die Kommentare ausgeben, die mein eigener Parser fand
		$CommentStarted = $False
		# Leere Zeilen ignorieren
		ForEach ($Line In $CodeInfo.CodeCommentLines | ? { -not [String]::IsNullOrEmpty($_) }) {
			If ($CommentStarted -eq $False) {
				$CommentStarted = $True
			}
			Print-Line -Msg $Line
		}
		Print-Line
	}
}


# Schreibt eines der Examples ins ClipBoard
# oder liefert @($ExampleIsExecutable, $ScriptFullName, $Parameters)
Function Get-Ex($GetEx, $ScriptFullName, $HelpText, [Switch]$ToClipboard = $True) {
	# 250708, MacOS
	# $HelpText ist PSCustomObject
	# Return $HelpText

	$ScriptFileName = [IO.path]::GetFileName($ScriptFullName)
	# $Example ist PSCustomObject
	$Example = $HelpText.Examples.Example[($GetEx - 1)]
	# Return $Example
	$Comments = $Example.Remarks.Text -replace "(?s)`n\s*$"
	# Return $Comments

	## Das Code-Beispiel analysieren
	$Code = $Example.Code
	# Return $Code
	$CodeInfo = Analyze-CodeExample $Code $ScriptFileName $ScriptFullName

	# Haben wir mehrere Zeilen im Code-Beispiel?
	$CodeExample = @()
	$CodeExample += $CodeInfo.FullCommandLine
	If ($CodeInfo.NoOfLines -gt 1) {
		$CodeLines = $CodeInfo.NoOfLines - 1 # Die erste Zeile wurde bereits angezeigt
		ForEach ($Comment In $Comments) {
			# Die Comments in Zeilen aufteilen und leere Zeilen ignorieren
			$Lines = ($Comment -split "`r`n|`r|`n") | ? { -not [String]::IsNullOrEmpty($_) }
			ForEach ($Line In $Lines) {
				# Wir haben noch eine Code-Zeile
				If (($CodeLines--) -gt 0) {
					$CodeExample += $Line
				}
			}
		}
	}

	If ($ToClipboard) {
		Set-Clipboard -Value ($CodeExample -join "`n")
	} Else {
		Return @($CodeInfo.IsDirectExecutable, $CodeExample)
	}
}

#Endregion GetHelp Hilfsfunktionen


#Region GetHelp Logik

# Sollen wir die -GetHelp, -GetEx, … Parameter parsen?
If ($ParseGetHelp) {

	# Wurde TomLib interaktiv in der Shell eingebunden?
	If (-Not $MyInvocation.PSCommandPath) {
		Write-Host '-ParseGetHelp funktoniert nicht, wenn TomLib interaktiv in der Shell eingebunden wird' -Fore Red
		Write-Host 'Lade ganz normal die TomLib, aber ihne Parsng von -GetHelp' -Fore DarkGreen
		Return
	}
	
	### Init
	## Parameter prüfen
	$IsGetHelp = $PSBoundParameters_.ContainsKey('GetHelp') -and $PSBoundParameters_['GetHelp']
	$IsGetHelpCls = $PSBoundParameters_.ContainsKey('GetHelpCls') -and $PSBoundParameters_['GetHelpCls']
	# $GetEx = $PSBoundParameters_.ContainsKey('GetEx') -and $PSBoundParameters_['GetEx']
	# $GetEx = If ($PSBoundParameters.ContainsKey('GetEx')) { $PSBoundParameters['GetEx'] } Else { $Null }
	$GetEx = $PSBoundParameters_['GetEx'] ?? $null
	# $IsRunEx = $PSBoundParameters_.ContainsKey('RunEx') -and $PSBoundParameters_['RunEx']
	$RunEx = $PSBoundParameters_['RunEx'] ?? $null

	## Das aufrufende Script
	$CallerScriptPath = $MyInvocation.PSCommandPath

	### Dbg
	# Write-Host "IsGetHelp   : $IsGetHelp"
	# Write-Host "IsGetHelpCls: $IsGetHelpCls"
	# Write-Host "GetEx       : $GetEx"
	# Write-Host "RunEx       : $RunEx"
	# Write-Host "CallerScript: $CallerScript"
	# Break Script

	### Logik
	If ($GetHelp -or $GetHelpCls) {
		If ($GetHelpCls) { CLS }
		Show-Help $CallerScriptPath (Get-Help -Examples $CallerScriptPath)
		Break Script
	}

	# Kopiert eines der Examples ins Clipboard
	If ($GetEx) {
		$ResGetEx = Get-Ex $GetEx $CallerScriptPath (Get-Help -Examples $CallerScriptPath)
		# Dbg
		# Return $ResGetEx
		Break Script
	}
	
	# Führt eines der Examples direkt aus
	If ($RunEx) {
		$ExampleIsExecutable, $CodeExample = Get-Ex $RunEx $CallerScriptPath (Get-Help -Examples $CallerScriptPath) -ToClipboard:$False
		If ($ExampleIsExecutable) {
			# Bei mehr als zwei Zeilen: Notepad öffnen und nur die erste Zeile ausführen
			Invoke-Expression -Command $CodeExample[0]
		} Else {
			Write-Host "Example $RunEx kann nicht direkt ausgeführt werden - es wurde in die Zwischenablage kopiert"
			Set-Clipboard -Value ("{0} {1}" -f $ScriptFullName, $Parameters)
		}
		Break Script
	}

}

#Endregion GetHelp Logik

