#!/usr/local/bin/pwsh
# !9^9 Unix (LF), nur UTF8 für MacOS, KEIN BOM

#Requires -Version 7


<#
	.SYNOPSIS
		Setzt für den lokalen MacOS User ein neues PW
		Und aktualisiert bei NinjaOne die Custom Fields

	.PARAMETER TargetUserName
		Für welchen User wird das Passwort gesetzt?

	.PARAMETER NewPassword
		Das neue PW des Users

	.PARAMETER AdminUserName
		Der Name des lokalen MMacOS Admin Users

	.PARAMETER AdminPassword
		Das PW des lokalen MMacOS Admin Users


	.EXAMPLE
		./Set-Local-MacUser-Password.ps1 -TargetUserName aaa -NewPassword bbb -AdminUserName ccc -AdminPassword ddd
		Setzt für den User das neue PW

	.NOTES
		001, 250611, Tom
#>


## Suppress PSScriptAnalyzer Warning
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidUsingCmdletAliases', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseApprovedVerbs', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidDefaultValueSwitchParameter', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidDefaultValueForMandatoryParameter', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidUsingPlainTextForPassword', '')]
# [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSShouldProcess', '')]
# [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseSingularNouns', '')]

[CmdletBinding(SupportsShouldProcess, ConfirmImpact='High', DefaultParameterSetName = 'GetHelp')]
Param(
	[Parameter(Mandatory, ParameterSetName='Script')]
	[String]$TargetUserName,

	[Parameter(Mandatory, ParameterSetName='Script')]
	[String]$NewPassword,

	[Parameter(Mandatory, ParameterSetName='Script')]
	[String]$AdminUserName,

	[Parameter(Mandatory, ParameterSetName='Script')]
	[String]$AdminPassword,

	# Get-Help Parameter
	# 005, 200314
	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Switch]$GetHelp,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Switch]$GetHelpCls,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Nullable[Int]]$GetEx,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Nullable[Int]]$RunEx
)

# Write-Host $PsCmdlet.ParameterSetName



### Init
$ScriptDir = [IO.Path]::GetDirectoryName($MyInvocation.MyCommand.Path)

# Ist irgend ein GetHelp aktiv?
$IsGetHelpActive = (($GetHelp) -or ($GetHelpCls) -or ($GetEx -ne $Null) -or ($RunEx -ne $Null))

# Wenn kein GetHelp Parameter aktiv ist, aber das ParameterSet GetHelb gewählt ist,
# dann wird -GetHelp aktiviert
If(($PsCmdlet.ParameterSetName -eq 'GetHelp') -and (-not $IsGetHelpActive)) {
    $GetHelp = $True
    $IsGetHelpActive = $True
}


### Config
$TomLibMacOS_ps1 = 'TomLib-MacOS.ps1'

$dsclPath = "/usr/bin/dscl"
$sysadminctlPath = "/usr/sbin/sysadminctl"


### Funcs


### Prepare
# Lib laden
$TomLibMacOS_ps1 = Join-Path $ScriptDir $TomLibMacOS_ps1
. $TomLibMacOS_ps1 $PSBoundParameters -ParseGetHelp


# Prüfen, ob der Befehl mit root-Rechten ausgeführt wird
if (Is-Root -eq $False) {
	Log 4 'Diese Funktion muss mit Administratorrechten (sudo pwsh) ausgeführt werden' -Fore Red
	return
}

if (-not (Test-Path $sysadminctlPath)) {
	Log 4 "Der 'sysadminctl'-Befehl wurde unter '$sysadminctlPath' nicht gefunden" -Fore Red
	return
}
	
	
# Prüfen, ob der dscl-Befehl existiert
if (-not (Test-Path $dsclPath)) {
	Log 4 "Der 'dscl'-Befehl wurde unter '$dsclPath' nicht gefunden" -Fore Red
	return
}

# Überprüfen, ob der Benutzer existiert (optional, aber empfohlen)
Write-Verbose "Überprüfe Existenz des Benutzers '$UserName'.."
try {
	$userCheck = & "$dsclPath" . -read "/Users/$UserName" 2>&1
	$LastExitOK, $LastExitNOK = Is-LastExitCode-AllOK $LastExitCode
	if ($LastExitOK -or $userCheck -match "No such key: Users") {
		Log 4 "Benutzer '$UserName' nicht gefunden. Bitte überprüfen Sie den Benutzernamen" -Fore Red
		return
	}
} catch {
	Log 4 "Fehler beim Überprüfen der Benutzer-Existenz: $($_.Exception.Message)" -Fore Red
	return
}



### Main

if ($PSCmdlet.ShouldProcess("Benutzer '$TargetUserName'", "Passwort setzen")) {
	Write-Verbose "Versuche Passwort für Benutzer '$TargetUserName' mit Admin-Konto '$AdminUserName' zu setzen.."
	try {
		# Argumentliste für sysadminctl
		$arguments = @(
			"-resetPasswordFor", $TargetUserName,
			"-newPassword", "-", # - als Platzhalter für stdin
			"-adminUser", $AdminUserName,
			"-adminPassword", "-" # - als Platzhalter für stdin
		)

		Write-Verbose "Starte Prozess: '$sysadminctlPath' mit Argumenten: '$($arguments -join ' ')'"

		$processInfo = New-Object System.Diagnostics.ProcessStartInfo
		$processInfo.FileName = $sysadminctlPath
		$processInfo.Arguments = $arguments -join ' '
		$processInfo.UseShellExecute = $false
		$processInfo.RedirectStandardInput = $true
		$processInfo.RedirectStandardOutput = $true
		$processInfo.RedirectStandardError = $true
		$processInfo.CreateNoWindow = $true

		$process = New-Object System.Diagnostics.Process
		$process.StartInfo = $processInfo

		$process.Start() | Out-Null # Startet den Prozess im Hintergrund

		# Sende Passwörter an stdin
		$process.StandardInput.WriteLine($NewPassword) # Direkte Übergabe des Klartext-Passworts
		$process.StandardInput.WriteLine($AdminPassword) # Direkte Übergabe des Klartext-Passworts
		$process.StandardInput.Close() # Wichtig: stdin schließen, damit der Prozess fortfährt

		$process.WaitForExit()

		$exitCode = $process.ExitCode
		$output = $process.StandardOutput.ReadToEnd()
		$errorOutput = $process.StandardError.ReadToEnd()

		Write-Verbose "sysadminctl Exit Code: $exitCode"
		if (![string]::IsNullOrWhiteSpace($output)) {
			Write-Verbose "sysadminctl Standard Output:`n$output"
		}
		if (![string]::IsNullOrWhiteSpace($errorOutput)) {
			Log 4 "sysadminctl Standard Error:`n$errorOutput" -Fore Magenta
		}

		if ($exitCode -ne 0) {
			Log 4 "Fehler beim Setzen des Passworts für Benutzer '$TargetUserName'. Exit Code: $exitCode. Details: $errorOutput" -Fore Red
		} else {
			Log 2 "Passwort für Benutzer '$TargetUserName' erfolgreich gesetzt"
		}

	} catch {
		Log 4 "Ein unerwarteter Fehler ist aufgetreten: $($_.Exception.Message)" -Fore Red
	} finally {
		# In dieser Version gibt es keine SecureStrings zu nullen,
		# aber die Passwörter sind nach der Ausführung im Speicher der Prozess-Klasse
		# und können in der Historie des aufrufenden Scripts landen.
	}
}

