#!/usr/local/bin/pwsh
# !9^9 Unix (LF), nur UTF8 für MacOS, KEIN BOM

#Requires -Version 7


<#
	.SYNOPSIS
		Setzt den MacOS Hostnamen 
		Behandelt auch das Problem, wenn das MacOS Domain-Joined ist

	.PARAMETER HostName
		Der neue Hostname

	.PARAMETER DomainName
		Die neue Domäne für den Host, e.g. 'akros.ch'

	.PARAMETER KeepDomainName
		Wenn nur der Hostname definiert ist und der Hostname einen Domänennamen hat,
		wird der Domänen-Name beibehalten

	.PARAMETER RemoveDomainName
		Wenn nur der Hostname definiert ist und der Hostname einen Domänennamen hat,
		wird der Domänen-Name entfernt

	.PARAMETER Force
		Erst mit -Force wird die System-Config verändert


	.EXAMPLE
		./Set-Hostname.ps1 -HostName AkMb011 -Force
		Einen neuen Hostnamen ohne Domäne setzen

	.EXAMPLE
		./Set-Hostname.ps1 -HostName AkMb011 -RemoveDomainName
		Einen Hostnamen setzen und die Domäne löschen
		Stoppt, wenn der Host domain-joined ist

	.EXAMPLE
		./Set-Hostname.ps1 AkMb011 akros.ch
		Einen Hostnamen und als Vorbereitung die Domäne setzen
		Stoppt, wenn der Host domain-joined ist


	.NOTES
		001, 250610, Tom
		002, 250611, Tom
			Shebang angepasst
		003, 250627, Tom
			Wenn das MacOS Domain-Joined ist,
			dann wird gefragt, ob der Domain-join 
			getrennt und wieder neu hergestellt werden soll
			damit das AD den neuen Hostnamen kennt
		004, 250704, Tom
			Wenn Domain-Joined, wird das setzen des Hostnamens nicht erlaubt
#>

## Suppress PSScriptAnalyzer Warning
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidUsingCmdletAliases', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseApprovedVerbs', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidDefaultValueSwitchParameter', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidDefaultValueForMandatoryParameter', '')]
# [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSShouldProcess', '')]
# [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseSingularNouns', '')]

[CmdletBinding(DefaultParameterSetName = 'HostNameOnly')]
Param (
	[Parameter(Mandatory, Position = 0, ParameterSetName = 'HostNameOnly')]
	[Parameter(Mandatory, Position = 0, ParameterSetName = 'HostAndDomain')]
	# HostName ohne Domäne
	# Wenn nicht definiert, dann wird allenfalls nur die Domäne geändert
    [String]$HostName,

	[Parameter(Mandatory, Position = 0, ParameterSetName = 'DomainOlny')]
	[Parameter(Mandatory, Position = 1, ParameterSetName = 'HostAndDomain')]
    [String]$DomainName,

	[Parameter(ParameterSetName = 'HostNameOnly')]
	# Der Domänenname soll beibehalten werden
	[Switch]$KeepDomainName,
	[Parameter(ParameterSetName = 'HostNameOnly')]
	# Der Domänenname soll entfernt werden
    [Switch]$RemoveDomainName,
    [Switch]$Force,

	# Get-Help Parameter
	# 005, 200314
	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Switch]$GetHelp,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Switch]$GetHelpCls,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Nullable[Int]]$GetEx,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Nullable[Int]]$RunEx
)

# Write-Host $PsCmdlet.ParameterSetName



### Init
$ScriptDir = [IO.Path]::GetDirectoryName($MyInvocation.MyCommand.Path)

# Ist irgend ein GetHelp aktiv?
$IsGetHelpActive = (($GetHelp) -or ($GetHelpCls) -or ($GetEx -ne $Null) -or ($RunEx -ne $Null))

# Wenn kein GetHelp Parameter aktiv ist, aber das ParameterSet GetHelb gewählt ist,
# dann wird -GetHelp aktiviert
If(($PsCmdlet.ParameterSetName -eq 'GetHelp') -and (-not $IsGetHelpActive)) {
    $GetHelp = $True
    $IsGetHelpActive = $True
}



If (($IsGetHelpActive -eq $False) -and (-not $Force)) {
	Write-Host "Kein -Force angegeben!" -Fore Magenta
	Write-Host "Es wird NICHTS verändert!" -Fore Magenta
}

### Config
$TomLibMacOS_ps1 = 'TomLib-MacOS.ps1'
$UnjoinDomain_ps1 = 'Unjoin-Domain.ps1'


### Funcs


### Prepare

## Script-Pfade auflösen
$TomLibMacOS_ps1 = Join-Path $ScriptDir $TomLibMacOS_ps1
$UnjoinDomain_ps1 = Join-Path $ScriptDir $UnjoinDomain_ps1

# Lib laden
. $TomLibMacOS_ps1 $PSBoundParameters -ParseGetHelp


## Infos zum Host abrufen
$oHostnamesInfo = Get-HostNames-Info
# Return $oHostnamesInfo

## Wenn wir Domain-Joined sind, können wir nichts ändern
If ($oHostnamesInfo.IsDomainJoined) {
	Log 4 'Der Host ist Domain-joined!' -Fore Red
	Log 4 'Weder Host noch Domain können geändert werden' -Fore Magenta
	Return $False
}

## Parameter validieren
If (Has-Value $HostName) { $HostName = $HostName.Trim() }
If (Has-Value $DomainName) { $DomainName = $DomainName.Trim() }

Switch($PsCmdlet.ParameterSetName) {
	'HostNameOnly' {

		## Wenn der HostName eine Domnäne hat, dann muss ein Switch gesetzt sein
		If (Has-Value $oHostnamesInfo.FQDN_Domain) {
			If (-not $KeepDomainName -and -not $RemoveDomainName) {
				Log 4 "Der MacOS Hostname hat eine Domäne definiert: $($oHostnamesInfo.FQDN)" -Fore Red
				Log 4 'Ein Switch muss definiert sein: -KeepDomainName oder -RemoveDomainName' -Fore Magenta
				Return $False
			}
		}


		# Hostname setzen, Domäne bleibt unverändert
		If ($KeepDomainName) {
			$DomainName = $oHostnamesInfo.FQDN_Domain
		} ElseIf ($RemoveDomainName) {
			$DomainName = $Null
		}
	}

	'HostAndDomain' {
		# Hostname und Domäne setzen
		# Alles OK
	}
	'DomainOnly' {
		# Nur die Domäne neu setzen, den Hostnamen nicht ändern
		$HostName = Get-Current-HostName -oHostnamesInfo $oHostnamesInfo
	}
	Default {
		Log 4 "Unbekanntes ParameterSet: $($PsCmdlet.ParameterSetName)" -Fore Red
		Return $False
	}
}




### Main

## Aktuelle Host infos ausgeben
Log 0 'Aktuelle Hostnamen' -Fore Magenta
($oHostnamesInfo | Out-String) -Split "`n" | `
	% { Log 1 "  $_" }

## Ist der Host Domain-joined?
If ($oHostnamesInfo.IsDomainJoined) {
	Log 4 'Computer ist domain-joined und kann deshalb nicht umbenannt werden' -Fore Red
	Log 5 'Trennen der Domäne:' -Fore Cyan
	Log 5 $UnjoinDomain_ps1 -Fore Black
	Log 5 'Abbruch' -Fore Magenta
	Return $False
}


If ($Force) {
	Log 0 'Setze die Hostnamen'
} Else {
	Log 0 'WÜRDE die Hostnamen setzen' -Fore Magenta
}

# Hostnamen setzen
Set-HostNames -NewHostName $HostName `
			-DomainName $DomainName `
			-Force:$Force


If ($Force) {
	Log 0 'Neue Hostnamen'
	# Infos zum Host abrufen
	$oHostnamesInfoNeu = Get-HostNames-Info
	($oHostnamesInfoNeu | Out-String) -Split "`n" | `
		% { Log 1 "  $_" }
	Return $True
} Else {
	Return $False
}

