#!/usr/local/bin/pwsh
# !9^9 Unix (LF), nur UTF8 für MacOS, KEIN BOM

#Requires -Version 7



<#
	.SYNOPSIS
		!9 Exakt wie:
		   Set-NinjaOne-Device-Field.ps1
		jedoch müssen dem Script die REST API Hashs 
		als Parameter übergeben werden,
		damit ein Hacker dieses Script nicht missbrauchen kann
		
		Nützt das NinjaOne REST API
		  Sucht bei NinjaOne das Objekt mit dem -ComputerName
		  Setzt im NinjaOne Computer-Objekt das Property -FieldName 
		  auf den Wert -FieldValue
		
		Unterstützte Aktionen
		-Action Set
		 Wert = Wert aus -FieldValue
		-Action Add
		 Wert = Alter Wert + Wert aus -FieldValue
		-Action AddNewLine
		 Wert = Alter Wert + \n + Wert aus -FieldValue
		
		!Q
		https://ninjarmm.zendesk.com/hc/en-us/community/posts/30319558075405-Script-Share-Retrieving-and-Updating-values-in-Custom-Fields-via-API-Windows-PowerShell


	.PARAMETER ComputerNames
		Entweder die Hostnamen, in dessen NinjaOne-Objekten ein Feld verändert werden soll

	.PARAMETER DeviceIDs
		Oder die NinjaOne DeviceIDs, in dessen NinjaOne-Objekten ein Feld verändert werden soll

	.PARAMETER FieldName
		Der Feldname, der im NinjaOne Coputer-Objekt gesetzt werden soll

	.PARAMETER FieldValue
		Der Feldwert, der gesetzt wird

	.PARAMETER Action
		Set				Ersetzt den Wert im Feld
		Add				Ergänzt den Wert im Feld (String-Concat)
		AddNewLine	Ergänzt den Wert im Feld mit einem NewLine

	.PARAMETER NinjaOneClientID
		Alternative zu -AuthHeader
		zB 'Gig.......................M'

	.PARAMETER NinjaOneClientSecret
		Alternative zu -AuthHeader
		zB 'aB..................................................1w'	

	.PARAMETER AuthHeader
		Anstatt -NinjaOneClientID und -NinjaOneClientSecret
		kann ein AuthHeader angegeben werden


	.PARAMETER Dbg	
		Wenn Debug-Infos angezeigt werden sollen


	.EXAMPLE
		./Set-NinjaOne-Device-Field-Secure.ps1 -ComputerNames 'AkMb011' -FieldName 'FieldName' -FieldValue 'Value' -Action Set -AuthHeader $AuthHeader
		Setzt das NinjaOne Custom Field


	.NOTES
		001, 250415, Tom
		002, 250612, Tom
			Neu:
				-Action
		003, 250612, Tom
			Neuer Name
				Set-NinjaOne-Device-Field-Secure.ps1
		004, 250616, Tom
		Neu:
			-CoputerID
			-AuthHeader 
#>

## Suppress PSScriptAnalyzer Warning
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidUsingCmdletAliases', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseApprovedVerbs', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidDefaultValueSwitchParameter', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidDefaultValueForMandatoryParameter', '')]
# [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSShouldProcess', '')]
# [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseSingularNouns', '')]

[CmdletBinding(SupportsShouldProcess, DefaultParameterSetName = 'GetHelp')]
Param(
	[Parameter(Mandatory, Position=0, ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName='ComputerName_WithAuthHeader')]
	[Parameter(Mandatory, Position=0, ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName='ComputerName_WoAuthHeader')]
	# Der Computername, in dessen NinjaOne-Objekt ein Feld verändert werden soll
	[String[]]$ComputerNames = @($Env:ComputerName),

	[Parameter(Mandatory, Position=0, ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName='DeviceID_WithAuthHeader')]
	[Parameter(Mandatory, Position=0, ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName='DeviceID_WoAuthHeader')]
	# Die NinjaOne DeviceIDs
	[String[]]$DeviceIDs,
	
	[Parameter(Mandatory, Position=1, ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName='ComputerName_WithAuthHeader')]
	[Parameter(Mandatory, Position=1, ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName='ComputerName_WoAuthHeader')]
	[Parameter(Mandatory, Position=1, ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName='DeviceID_WithAuthHeader')]
	[Parameter(Mandatory, Position=1, ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName='DeviceID_WoAuthHeader')]
	# Der Feldname, der im NinjaOne Coputer-Objekt gesetzt werden soll
	[String]$FieldName = 'ssfmahandynr',
	
	[Parameter(Mandatory, Position=2, ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName='ComputerName_WithAuthHeader')]
	[Parameter(Mandatory, Position=2, ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName='ComputerName_WoAuthHeader')]
	[Parameter(Mandatory, Position=2, ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName='DeviceID_WithAuthHeader')]
	[Parameter(Mandatory, Position=2, ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName='DeviceID_WoAuthHeader')]
	# Der Feldwert, der gesetzt wird
	[String]$FieldValue,

	[Parameter(ParameterSetName='ComputerName_WithAuthHeader')]
	[Parameter(ParameterSetName='ComputerName_WoAuthHeader')]
	[Parameter(ParameterSetName='DeviceID_WithAuthHeader')]
	[Parameter(ParameterSetName='DeviceID_WoAuthHeader')]
	[ValidateSet(IgnoreCase, 'Set', 'Add', 'AddNewLine')]
	# Set				Ersetzt den Wert im Feld
	# Add				Ergänzt den Wert im Feld (String-Concat)
	# AddNewLine	Ergänzt den Wert im Feld mit einem NewLine
	[String]$Action = 'Set',

	[Parameter(Mandatory, ParameterSetName='ComputerName_WoAuthHeader')]
	[Parameter(Mandatory, ParameterSetName='DeviceID_WoAuthHeader')]
	# zB 'Gig.......................M'
	[String]$NinjaOneClientID,
	
	[Parameter(Mandatory, ParameterSetName='ComputerName_WoAuthHeader')]
	[Parameter(Mandatory, ParameterSetName='DeviceID_WoAuthHeader')]
	# zB 'aB..................................................1w'	
	[String]$NinjaOneClientSecret,
	
	[Parameter(Mandatory, ParameterSetName='ComputerName_WithAuthHeader')]
	[Parameter(Mandatory, ParameterSetName='DeviceID_WithAuthHeader')]
	[Object]$AuthHeader,
	
	[Parameter(ParameterSetName='ComputerName_WithAuthHeader')]
	[Parameter(ParameterSetName='ComputerName_WoAuthHeader')]
	[Parameter(ParameterSetName='DeviceID_WithAuthHeader')]
	[Parameter(ParameterSetName='DeviceID_WoAuthHeader')]
	# Debug?
	[Switch]$Dbg,

	# Get-Help Parameter
	# 005, 200314
	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Switch]$GetHelp,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Switch]$GetHelpCls,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Nullable[Int]]$GetEx,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Nullable[Int]]$RunEx
)

# Write-Host $PsCmdlet.ParameterSetName




### Init
$ScriptDir = [IO.Path]::GetDirectoryName($MyInvocation.MyCommand.Path)

# Ist irgend ein GetHelp aktiv?
$IsGetHelpActive = (($GetHelp) -or ($GetHelpCls) -or ($GetEx -ne $Null) -or ($RunEx -ne $Null))

# Wenn kein GetHelp Parameter aktiv ist, aber das ParameterSet GetHelb gewählt ist,
# dann wird -GetHelp aktiviert
If(($PsCmdlet.ParameterSetName -eq 'GetHelp') -and (-not $IsGetHelpActive)) {
    $GetHelp = $True
    $IsGetHelpActive = $True
}


### Config
$TomLibMacOS_ps1 = 'TomLib-MacOS.ps1'
$NinjaOneBaseURL = 'https://eu.ninjarmm.com/'
$NinjaOneBaseURLAPI  = "$($NinjaOneBaseURL.TrimEnd('/'))/api/"



### Funcs



### Prepare
$TomLibMacOS_ps1 = Join-Path $ScriptDir $TomLibMacOS_ps1
# Lib laden
. $TomLibMacOS_ps1 $PSBoundParameters -ParseGetHelp


# Authentifizieren?
If ( $PsCmdlet.ParameterSetName.EndsWith('WoAuthHeader')) {
	$AuthHeader = Auth-NinjaOne -Dbg:$Dbg
}
# Return $AuthHeader



### Main

If ( $PsCmdlet.ParameterSetName.StartsWith('ComputerName')) {
	# Wir arbeiten mit Computernamen
	If ($Dbg) { Log 4 'Mode: ComputerName' -Fore DarkGray }
	
	ForEach($ComputerName in $ComputerNames) {
		
		# Das NinjaOne Gerät suchen
		$ThisDevice = Get-NinjaOne-Device-by-SystemName -AuthHeader $AuthHeader -SystemName $ComputerName -Dbg:$Dbg
		
		If ($ThisDevice) {
			# Vom NinjaOne Gerät ale Fields abrufen
			$Fields = Get-NinjaOne-Device-Fields -AuthHeader $AuthHeader -DeviceID $ThisDevice.ID -Dbg:$Dbg
			
			# Alle gefundenen Felder ausgeben?
			If ($Dbg) {
				Log 4 ("`n{0}: {1}" -f $ThisDevice.ID, $ThisDevice.SystemName) -Fore Green
				($Fields | FL | Out-String) -split "`n" | `
					? { -Not [String]::IsNullOrWhiteSpace($_) } | `
					% { Log 5 "  $_" }
			}

			# Haben wir einen Wert?
			$WertOri = $Fields.$FieldName
			$WertNeu = ''
			Switch ($Action) {
				'Set' {
					$WertNeu = $FieldValue
				}
				'Add' {
					$WertNeu = $WertOri + $FieldValue
				}
				'AddNewLine' {
					$WertNeu = ("{0}`n{1}" -f $WertOri, $FieldValue)
				}
				Default {
					Log 4 "Ungültige Action: $Action" -Fore Red
				}
			}

				
			# Das Feld setzen
			Set-NinjaOne-Device-Field -AuthHeader $AuthHeader -DeviceID $ThisDevice.ID `
									-FieldName $FieldName -FieldValue $WertNeu -Dbg:$Dbg
			
		} Else {
			Log 4 "Nicht gefunden: $ComputerName" -Fore Red
		}
	}
}


If ( $PsCmdlet.ParameterSetName.StartsWith('DeviceID')) {
	# Wir arbeiten mit DeviceIDs
	If ($Dbg) { Log 4 'Mode: DeviceIDs' -Fore DarkGray }
	
	ForEach($DeviceID in $DeviceIDs) {
		If ($Dbg) { Log 4 ('DeviceID: {0}' -f $DeviceID) -Fore Green }

		# Vom NinjaOne Gerät ale Fields abrufen
		$Fields = Get-NinjaOne-Device-Fields -AuthHeader $AuthHeader -DeviceID $DeviceID -Dbg:$Dbg
		
		# Alle gefundenen Felder ausgeben?
		If ($Dbg) {
			($Fields | FL | Out-String) -split "`n" | `
				? { -Not [String]::IsNullOrWhiteSpace($_) } | `
				% { Log 5 "  $_" }
		}

		# Haben wir einen Wert?
		$WertOri = $Fields.$FieldName
		$WertNeu = ''
		Switch ($Action) {
			'Set' {
				$WertNeu = $FieldValue
			}
			'Add' {
				$WertNeu = $WertOri + $FieldValue
			}
			'AddNewLine' {
				$WertNeu = ("{0}`n{1}" -f $WertOri, $FieldValue)
			}
			Default {
				Log 4 "Ungültige Action: $Action" -Fore Red
			}
		}

		If ($Dbg) { 
			Log 2 ('Wert aktuell: {0}' -f $WertOri)
			Log 2 ('Wert neu    : {0}' -f $WertNeu)
		}

		# Das Feld setzen
		Set-NinjaOne-Device-Field -AuthHeader $AuthHeader `
								-DeviceID $DeviceID `
								-FieldName $FieldName `
								-FieldValue $WertNeu `
								-Dbg:$Dbg
		
	}
}


