#!/usr/local/bin/pwsh
# !9^9 Unix (LF), nur UTF8 für MacOS, KEIN BOM

#Requires -Version 7



<#
	.SYNOPSIS
		Aktiviert die MacOS Disk-Verschlüsselung (FileVault)
		und speichert das Passwort in NinjaOne

		Wenn FileVault nicht aktiviert ist:
		- Script-Aufruf 1:
		FileVault wird aktiviert und der Recovery-Key in eine lokale Datei geschrieben
		Der Benutzer wird abgemeldet und muss sich wieder anmelden, 
		wo dann der FileVault aktiviert wird
		- Script-Aufruf 2:
		Der FileVault Recovery-Key wird zu NinjaOne übertragen

		Wenn FileVault aktiviert ist:
		Der Status wird angezeigt

	.PARAMETER TestAddNewRec_ssfClientTasksData
		Erzeugt bei NinjaOne das Test-Record mit einem Dummy MacOS FileVault Recovery-Key für Tests


	.EXAMPLE
		! .\Enable-MacOS-FileVault.ps1
		Wenn FileVault aktiviert ist, dann wird der Status angezeigt
		Sonst wird beim ersten Aufruf FileVault aktiviert
		und beim zweiten Aufruf der FileVault Recovery Key zu NinjaOne übertragen

	.EXAMPLE
		! .\Enable-MacOS-FileVault.ps1 -TestAddNewRec_ssfClientTasksData
		Erzeugt bei NinjaOne das Test-Record mit einem Dummy MacOS FileVault Recovery-Key für Tests


	.NOTES
		001, 250611, Tom
		002, 250708, Tom
			Neu: -GetHelp
#>

## Suppress PSScriptAnalyzer Warning
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidUsingCmdletAliases', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseApprovedVerbs', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidDefaultValueSwitchParameter', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidDefaultValueForMandatoryParameter', '')]
# [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSShouldProcess', '')]
# [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseSingularNouns', '')]
[CmdletBinding(DefaultParameterSetName = 'Script')]
Param(
	[Parameter(ParameterSetName = 'Script')]
	[Switch]$TestAddNewRec_ssfClientTasksData,

	# Get-Help Parameter
	# 005, 200314
	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Switch]$GetHelp,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Switch]$GetHelpCls,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Nullable[Int]]$GetEx,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Nullable[Int]]$RunEx
)

# Write-Host $PsCmdlet.ParameterSetName



### Init
$ScriptDir = [IO.Path]::GetDirectoryName($MyInvocation.MyCommand.Path)
$ThisScriptPath = $MyInvocation.MyCommand.Definition


### Root erzwingen
# 250620 1850
If ((id -u) -ne 0) {
    Write-Host 'Starte das Script als root'
	If ([String]::IsNullOrWhiteSpace($ThisScriptPath)) {
		Write-Error 'Variable $ThisScriptPath ist leer. Kann nicht fortfahren.'
	}	

    # Argumente aus den BoundParameters zusammensetzen
    $paramList = @()
    foreach ($kvp in $PSBoundParameters.GetEnumerator()) {
        if ($kvp.Value -is [switch] -and $kvp.Value.IsPresent) {
            $paramList += "-$($kvp.Key)"
        } else {
            $paramList += "-$($kvp.Key) `"$($kvp.Value)`""
        }
    }

    $pwsh = (Get-Command pwsh).Source

    if ($paramList.Count -gt 0) {
        # sudo $pwsh $ThisScriptPath @paramList
		$quotedArgs = $paramList -join ' '
		$cmd = "$pwsh `"$ThisScriptPath`" $quotedArgs"
		sudo bash -c "$cmd"
    } else {
        sudo $pwsh $ThisScriptPath
    }

    exit
}

# Ab hier läuft das Script als root
Write-Host "`nScript läuft jetzt als root`n`n" -Fore DarkGray



### Init

# Ermitteln des aktuell angemeldeten Benutzers auf macOS
# Die Ausgabe von ls -l /dev/console ist:
#	"crw-rw-rw- 1 loggedinuser wheel 3, 0 May 20 10:00 /dev/console"
# Zeigen root:
# 	$LoggedInUser = $(/bin/zsh -c "ls -l /dev/console | awk '{ print `$3 }'")
# 	$LoggedInUser = [System.Environment]::UserName
#
# Zeigen den User, der am Desktop angemeldet ist:
# echo $SUDO_USER
# logname
$LoggedInUser = $(logname)



### Config
$TomLibMacOS_ps1 = 'TomLib-MacOS.ps1'

# NinjaOne Custom Field Name
$FieldName_ssfClientTasksData = 'ssfClientTasksData'

# Pfad zur recovery.plist Datei
# WICHTIG
# - Der Recovery-Key FileVault Recovery Key gehört zum User, der ihn erzeugt hat!
# - FileVault kann nur mit Root-Rechten aktiviert werden
# Deshalb dieser Pfad:
$recoveryPlistPath = "/private/var/root/$loggedInUser-recovery.plist" 
# Log 4 $recoveryPlistPath

$SetLoginScript_sh = '/Library/Scripts/SSF/NinjaOne/Set-Login-Script.sh'
$NinjaCliPath = '/Applications/NinjaRMMAgent/programdata/ninjarmm-cli'



### Funcs



### Prepare

# Lib laden
$TomLibMacOS_ps1 = Join-Path $ScriptDir $TomLibMacOS_ps1
. $TomLibMacOS_ps1 $PSBoundParameters -ParseGetHelp

# Status bestimmen
$IsFileVaultEnabled = Is-FileVault-Enabled
$HasRecoveryPlistFile = Test-Path -LiteralPath $recoveryPlistPath -PathType Leaf
$IsRecoveryPlistFileMissing = -not $HasRecoveryPlistFile


$ResSummary = [Ordered]@{
	IsFileVaultEnabled =$IsFileVaultEnabled
}

If ($HasRecoveryPlistFile) {
	$ResSummary = @{ RecoveryPlistFile = $recoveryPlistPath }
} Else {
	$ResSummary = @{ RecoveryPlistFile = 'Nicht vorhanden' }
}



### Main

Log 0 'Prüfe FileVault-Status'

# Nur im Test-Modus?
If ($TestAddNewRec_ssfClientTasksData) {
	Log 1 'Resultat:' -Fore Magenta
	Log 2 'TestAddNewRec_ssfClientTasksData' -NewLineBefore
	Log 2 "Setze den Record im Custom Field: $($Script:FieldName_ssfClientTasksData)"
	Create-ssfClientTasksData-Rec -FileVaultRecoveryKey ((Get-Date).ToString('yyMMdd HHmmss'))
	Log 2 'Done'
	Return
}



# Ist FileVault bereits aktiv?
If ($IsFileVaultEnabled -and $IsRecoveryPlistFileMissing) {
	Log 1 'Resultat:' -Fore Magenta
	Log 2 'FileVault ist bereits aktiviert' -Fore DarkGreen
	Log 2 'Der Recovery-Key ist bereits in NinjaOne gesichert' -Fore DarkGreen
	Break Script
}



## FileVault ist noch nicht aktiv

# Wenn noch keine RecoveryPlist erzeugt wurde, wurde FileVault noch nicht aktiviert
If ($IsRecoveryPlistFileMissing) {
	
	# FileVault wurde noch nicht aktiviert
    Log 1 "Aktiviere FileVault für: $LoggedInUser"

	try {
		# Verwende Start-Process mit -Wait, um sicherzustellen, dass der Befehl abgeschlossen ist
		Start-Process -FilePath 'fdesetup' -ArgumentList 'enable', '-user', $LoggedInUser, '-defer', $recoveryPlistPath, '-keychain', '-forceatlogin', '0' -Wait -NoNewWindow -PassThru | Out-Null

		Log 2 'FileVault-Aktivierung eingeleitet'
		
		# Dieses Script als login Script definieren
		Log 2 'Aktiviere Login-Script' -NewLineBefore
		# Parameter für das Bash-Skript
		$Username = $LoggedInUser
		$PwshScriptToExecute = "$ThisScriptPath"
		$Action = 'add'
		# $Mode = 'runonce'
		$Mode = 'persistent'
		
		Try {
			# Option 1: Ohne optionalen Modus-Parameter
			& $SetLoginScript_sh $Username $PwshScriptToExecute $Action $Mode
			$LastExitOK, $LastExitNOK = Is-LastExitCode-AllOK $LastExitCode
			
			If ($LastExitOK) {
				Log 3 'Bash-Skript erfolgreich ausgeführt'
			} else {
				Log 4 'Bash-Skript wurde mit Exit-Code $LastExitCode beendet' -Fore Red
			}
		} Catch {
			Log 4 "Fehler beim Ausführen des Bash-Skripts: $($_.Exception.Message)" -Fore red
		}		

		Log 2 'Starte in 5s das Logoff' -Fore Red -NewLineBefore
		Log 2 'Bitte wieder anmelden und so den FileVault aktivieren!' -Fore Magenta
		
		Start-Sleep -Milliseconds 5500
		[void] (Read-Host "Drücke [Enter] zum Fortfahren")
		
		Logoff-MacOS-User -LoggedInUser $LoggedInUser
		
	} catch {
		Log 4 "Fehler bei der FileVault-Aktivierung: $($_.Exception.Message)" -Fore Red
	}
		
	
} Else {

	# FileVault wurde bereits aktiviert,
	# Aber der Recovery-key wurde noch nicht in NinjaOne gesichert
	
    Log 1 'Übertrage den FileVault Recovery Key zu NinjaOne'

    try {
        # Den RecoveryKey aus der Plist-Datei lesen
        $FileVaultRecoveryKey = (& /usr/libexec/PlistBuddy -c "Print :RecoveryKey" $recoveryPlistPath)
		
		Log 2 $FileVaultRecoveryKey
		
		$ResSummary = @{ RecoveryKey = $FileVaultRecoveryKey }


        if ([string]::IsNullOrEmpty($FileVaultRecoveryKey)) {
            Log 4 "Der Recovery Key konnte nicht aus $recoveryPlistPath gelesen werden" -Fore red
        } else {
			Log 2 'Hat NinjaOne den Recovery Key bereits?'
			$RecoveryKeyTransferOK = Assert-ssfClientTasksData-has-FileVaultRecoveryKey `
										-FileVaultRecoveryKey $FileVaultRecoveryKey `
										-TimeoutSec 10
			
			If ($RecoveryKeyTransferOK) {
				Log 3 'NinjaOne hat den Recovery Key bereits'
			} Else {
				Log 3 'Übergebe Recovery Key an NinjaOne'
				
				# Den Record erzeugen
				Create-ssfClientTasksData-Rec -FileVaultRecoveryKey $FileVaultRecoveryKey
				
				
				# Hat NinjaOne den VaultKey?
				$RecoveryKeyTransferOK = Assert-ssfClientTasksData-has-FileVaultRecoveryKey -FileVaultRecoveryKey $FileVaultRecoveryKey
				
				Log 4 "fileVaultKeyOK: $RecoveryKeyTransferOK" -NewLineBefore
			}
			
			
			## Wenn der Recovery-Key übertragen wurde,
			## die PlistBuddy Datei löschen
			$FileExists = Test-Path -LiteralPath $recoveryPlistPath -PathType Leaf
			
			If ($RecoveryKeyTransferOK -and $FileExists) {
				Log 3 ('Lösche Recovery Key Datei: {0}' -f $recoveryPlistPath)
				Remove-Item -Path $recoveryPlistPath -Force
				$ResSummary = @{ RecoveryPlistFileExists = $False }
			} Else {
				$ResSummary = @{ RecoveryPlistFileExists = $FileExists }
			}
			
			$ResSummary = @{ RecoveryKeyTransferOK = $RecoveryKeyTransferOK }
		
			Log 3 'done.'
			
        }
    } catch {
        Log 4 "Fehler beim Lesen der recovery.plist oder beim Übergeben an NinjaRMM: $($_.Exception.Message)" -Fore Red
    }	
}



## Resultat ausgeben
Log 1 "`nResultat:" -Fore Magenta -NewLineBefore
Log 2 ([PSCustomObject]$ResSummary | Out-String) -Fore Magenta

