#!/usr/local/bin/pwsh
# !9^9 Unix (LF), nur UTF8 für MacOS, KEIN BOM

#Requires -Version 7


<#
	.SYNOPSIS
		Schreibt Daten in ein NinjaOne Custom Text-Field
		und ergänzt eine Zeile

	.PARAMETER FieldName
		Name des NinjaOne Custom Text-Field

	.PARAMETER LineToAdd
		Testzeile, die im NinjaOne Custom Text-Field ergönzt werden soll

	.EXAMPLE
		! .\Add-NinjaOne-TextField-Line.ps1 -FieldName ssfMacOsFileVaultKey -LineToAdd TomTom
		Ergänzt bei NinjaOne das Custom Field "ssfMacOsFileVaultKey" um die Zeile "TomTom"

	.NOTES
		001, 250611, Tom
		002, 250708, Tom
			Neu: -GetHelp
#>

## Suppress PSScriptAnalyzer Warning
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidUsingCmdletAliases', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseApprovedVerbs', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidDefaultValueSwitchParameter', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidDefaultValueForMandatoryParameter', '')]
# [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSShouldProcess', '')]
# [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseSingularNouns', '')]
[CmdletBinding(DefaultParameterSetName = 'GetHelp')]
Param(
	[Parameter(Mandatory, ParameterSetName = 'Script')]
	[String]$FieldName,

	[Parameter(Mandatory, ParameterSetName = 'Script')]
	[String]$LineToAdd,

	# Get-Help Parameter
	# 005, 200314
	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Switch]$GetHelp,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Switch]$GetHelpCls,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Nullable[Int]]$GetEx,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Nullable[Int]]$RunEx
)

# Write-Host $PsCmdlet.ParameterSetName



### Init
$ScriptDir = [IO.Path]::GetDirectoryName($MyInvocation.MyCommand.Path)



### Config
$TomLibMacOS_ps1 = 'TomLib-MacOS.ps1'
$NinjaCliPath = '/Applications/NinjaRMMAgent/programdata/ninjarmm-cli'



### Funcs



### Prepare

# Lib laden
$TomLibMacOS_ps1 = Join-Path $ScriptDir $TomLibMacOS_ps1
. $TomLibMacOS_ps1 $PSBoundParameters -ParseGetHelp



### Main

Write-Verbose "Prüfe, ob das NinjaOne CLI-Tool existiert: '$NinjaCliPath'"
If (-not (Test-Path $NinjaCliPath)) {
	Log 4 "Das NinjaOne CLI-Tool wurde unter '$NinjaCliPath' nicht gefunden" -Fore Red
	Return
}

Write-Verbose "Rufe aktuellen Inhalt des Feldes '$FieldName' ab …"
try {
	# Der Backtick am Zeilenende ist für Multi-Line-Befehle in PowerShell
	# Die `Get-Content` umgeht potentielle Probleme mit der Zeilenumbruchinterpretation
	# bei der direkten Ausgabe in eine Variable.
	# $CurrentContent = & "$NinjaCliPath" get "$FieldName" | Out-String | ForEach-Object { $_.Trim() }
	$ResProc = Invoke-Process-WithTimeout -ExeFilePath $NinjaCliPath `
											-Arguments "get `"$FieldName`"" `
											-MaxWaitTimeSec 15

	# Return $ResProc
	If (Handle-Invoke-Process-Errors $ResProc '2jfaEa') {
		Log 4 'Abbruch' -Fore Red
		Break Script
	}

	# Den Wert im CustomField auslesen
	$CurrentContent = $ResProc.StdOut | % { $_.Trim() }
											
	# Behandeln, wenn das Feld (bisher) leer ist. 
	# Out-String kann einen leeren String oder nur Leerzeichen/Zeilenumbrüche zurückgeben.
	If ([string]::IsNullOrWhiteSpace($CurrentContent)) {
		$UpdatedContent = $LineToAdd
	} Else {
		Write-Verbose "Aktueller Inhalt des Feldes '$FieldName':`n$CurrentContent"
		# Füge die neue Zeile mit einem Zeilenumbruch hinzu
		$UpdatedContent = "$CurrentContent`n$LineToAdd"
	}

	Write-Verbose "Setze aktualisierten Inhalt für Feld '$FieldName'..."
	# Verwende die doppelten Anführungszeichen, um sicherzustellen, dass Leerzeichen und Zeilenumbrüche korrekt übergeben werden.
	# $cliOutput = & "$NinjaCliPath" set "$FieldName" "$UpdatedContent"
	# $LastExitOK, $LastExitNOK = Is-LastExitCode-AllOK $LastExitCode
	$ResProc = Invoke-Process-WithTimeout -ExeFilePath $NinjaCliPath `
											-Arguments "set `"$FieldName`" `"$UpdatedContent`"" `
											-MaxWaitTimeSec 15

	If (Handle-Invoke-Process-Errors $ResProc '2jfaEb') {
		Log 4 "Fehler beim Setzen des Feldes '$FieldName'. Exit Code: $ResProc.ExitCode. Output: $ResProc.StdErr" -Fore Red
		Log 4 'Abbruch' -Fore Red
		Break Script
	} Else {
		Log 1 "Zeile erfolgreich zu Feld '$FieldName' hinzugefügt." -Fore Green
		Write-Verbose "Der aktualisierte Inhalt ist: '$UpdatedContent'"
	}

} catch {
	Log 4 "Ein unerwarteter Fehler ist aufgetreten: $($_.Exception.Message)" -Fore Red
}

