#!/usr/local/bin/pwsh
# !9^9 Unix (LF), nur UTF8 für MacOS, KEIN BOM

#Requires -Version 7

## Suppress PSScriptAnalyzer Warning
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidUsingCmdletAliases', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseApprovedVerbs', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidDefaultValueSwitchParameter', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidDefaultValueForMandatoryParameter', '')]
# [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSShouldProcess', '')]
# [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseSingularNouns', '')]


# Lib für MacOS

# 001, 250701, Tom
# 002, 250704, Tom
# 003, 250704, Tom



# !Ex
#
### Init
# $ScriptDir = [IO.Path]::GetDirectoryName($MyInvocation.MyCommand.Path)
#
### Config
# $TomLibMacOS_ps1 = 'TomLib-MacOS.ps1'
#
### Prepare
# Lib laden
# $TomLibMacOS_ps1 = Join-Path $ScriptDir $TomLibMacOS_ps1
# . $TomLibMacOS_ps1 $PSBoundParameters


[CmdletBinding(SupportsShouldProcess, ConfirmImpact = 'Low')]
Param (
	# Optional $PSBoundParameters, 
	# um zu erkennen: IsVerbose, IsWhatIf 
	[HashTable]$PSBoundParameters_
)



### Init

# Haben wir Verbose / WhatIf?
If ($PSBoundParameters_) {
	$IsVerbose = $PSBoundParameters_.ContainsKey('Verbose') -and $PSBoundParameters_['Verbose']
	$IsWhatIf = $PSBoundParameters_.ContainsKey('WhatIf') -and $PSBoundParameters_['WhatIf'] -or $Force -eq $False
}



### Config



### Funcs


#Region String Funcs

# Macht, was man erwartet
# 220127
Function Is-NullOrEmpty([String]$Str) {
	[String]::IsNullOrWhiteSpace($Str)
}

Function Has-Value($Data) {
	If ($null -eq $Data) { Return $False }
	Switch ($Data.GetType().Name) {
		'String' {
			If ([String]::IsNullOrWhiteSpace($Data)) { Return $False } 
			Else { Return $True }
		}
		Default {
			Return $True
		}
	}
}

Function Is-Empty($Data) {
	Return !(Has-Value $Data)
}

Function Is-Verbose() {
	$VerbosePreference -eq 'Continue'
}

Function Is-WhatIf() {
	$WhatIfPreference
}

#Endregion String Funcs


#Region Log

# 250701
# 	Portiert der Windows-PowerShell Version:
# 		220123 222224
# 		Fixed: -ReplaceLine handling
$Script:LogColors = @('DarkMagenta', 'DarkGreen', 'DarkCyan', 'DarkBlue', 'Red')

# Mapping von ConsoleColor zu ANSI-Farbcodes
$Script:AnsiColorMap = @{
    Black      = 30; DarkBlue = 34; DarkGreen   = 32; 
	DarkCyan   = 36; DarkRed  = 31; DarkMagenta = 35; 
	DarkYellow   = 33; Gray     = 37; DarkGray    = 90; 
	Blue       = 94; Green    = 92; Cyan        = 96;
    Red        = 91; Magenta    = 95; Yellow      = 93; 
	White      = 97;
}

Function Convert-ToAnsiColor {
    Param (
        [ConsoleColor]$Color,
        [Switch]$Background
    )
    $ansi = $Script:AnsiColorMap[$Color.ToString()]
    If ($Background) {
        # Hintergrundfarben: 10er-Offset
        $ansi += 10
    }
    return "`e[${ansi}m"
}

# 250704 2114
#	[Switch]$IfVerbose
Function Log {
    Param (
        [Parameter(Position = 0)]
        [Int]$Indent = 0,

        [Parameter(Position = 1, ValueFromPipeline, ValueFromPipelineByPropertyName)]
		[Alias('Msg')]
		[String]$Message = '',

        [Parameter(Position = 2)]
        [ConsoleColor]$ForegroundColor,

        [Parameter(Position = 3)]
        [Switch]$NewLineBefore,

        [Parameter(Position = 4)]
        [Switch]$ReplaceLine = $false,

        [Parameter(Position = 5)]
        [Switch]$NoNewline = $false,

        [Parameter(Position = 6)]
        [Switch]$Append = $false,

        [Parameter(Position = 7)]
        [Switch]$ClrToEol = $false,

		# Ausgabe erfolgt nur, wenn Verbose aktiv ist
		[Switch]$IfVerbose,

        [Parameter(Position = 8)]
        [ConsoleColor]$BackgroundColor
    )

    Begin {
        If ($Script:LogDisabled -eq $true) { return }

        If (-not $ForegroundColor) {
            If ($Indent -lt $Script:LogColors.Count) {
                $ForegroundColor = [ConsoleColor]::Parse([ConsoleColor], $Script:LogColors[$Indent])
            } else {
                $ForegroundColor = [ConsoleColor]::White
            }
        }

        $AnsiStart = Convert-ToAnsiColor -Color $ForegroundColor
        If ($BackgroundColor) {
            $AnsiStart += Convert-ToAnsiColor -Color $BackgroundColor -Background
        }
        $AnsiEnd = "`e[0m"

		# $Script:LogMaxWindowSizeWidth
		If ($null -eq $Script:LogMaxWindowSizeWidth) {
			Try {
				$ThisWidth = $Host.UI.RawUI.WindowSize.Width
				If ($Null -eq $ThisWidth) {
					$Script:LogMaxWindowSizeWidth = 132
				} Else {
					$Script:LogMaxWindowSizeWidth = $ThisWidth
				}
			} Catch {
				$Script:LogMaxWindowSizeWidth = 132
			}
		}

    }

    Process {
		If ($Script:LogDisabled -eq $True) { Return }

		# Wenn Verbose gewünscht aber nicht aktiv, dann sind wir fertig
		If ($IfVerbose -and (Is-Verbose) -eq $False) { Return }

		# Wenn -Indent < 0, sind wir fertig
		If ($Indent -lt 0) { Return }

		If ([String]::IsNullOrEmpty($Message)) { $Message = '' }

        If ($NewLineBefore) { Write-Host '' }

		If ($Append) {
			$Msg = $Message
			If ($ClrToEol) {
				If ($Msg.Length -lt $Script:LogMaxWindowSizeWidth) {
					$Msg += ' ' * ($Script:LogMaxWindowSizeWidth - $Msg.Length)
				}
			}
		}
		Else {
			Switch ($Indent) {
				0 {
					$Msg = "* $Message"
					If ($NoNewline -and $ClrToEol) {
						If ($Msg.Length -lt $Script:LogMaxWindowSizeWidth) {
							$Msg += ' ' * ($Script:LogMaxWindowSizeWidth - $Msg.Length)
						}
					}
					If (!($ReplaceLine)) {
						$Msg = "`n$Msg"
					}
				}
				Default {
					$Msg = $(' ' * ($Indent * 2) + $Message)
					If ($NoNewline -and $ClrToEol) {
						# Rest der Zeile mit Leerzeichen überschreiben
						$Width = $Script:LogMaxWindowSizeWidth
						If ($Msg.Length -lt $Script:LogMaxWindowSizeWidth) {
							$Msg += ' ' * ($Script:LogMaxWindowSizeWidth - $Msg.Length)
						}
					}
				}
			}
		}

        If ($ReplaceLine) { $Msg = "`r`e[0K$Msg" }

        If ($NoNewline) {
            Write-Host "$AnsiStart$Msg$AnsiEnd" -NoNewline
        } else {
            Write-Host "$AnsiStart$Msg$AnsiEnd"
        }
    }
}

#Endregion Log


#Region Shell Farben


# Shell Styles
Enum ShellColorStyle {
    Text; H1; H2; H3; H4; Err1; Err2
}

# Die Shell-Farben
$Script:ShellColors = @{
    [ShellColorStyle]::Text = @{ ForegroundColor = 'DarkGray'; BackgroundColor = 'White' }
    
    [ShellColorStyle]::H1 = @{ ForegroundColor = 'Blue'; BackgroundColor = 'White' }
    [ShellColorStyle]::H2 = @{ ForegroundColor = 'Magenta'; BackgroundColor = 'White' }
    [ShellColorStyle]::H3 = @{ ForegroundColor = 'Green'; BackgroundColor = 'White' }
    [ShellColorStyle]::H4 = @{ ForegroundColor = 'Black'; BackgroundColor = 'White' }
    
    [ShellColorStyle]::Err1 = @{ ForegroundColor = 'Red'; BackgroundColor = 'White' }
    [ShellColorStyle]::Err2 = @{ ForegroundColor = 'DarkRed'; BackgroundColor = 'White' }
}


# Die Shell-Farben mit Enum-Werten als Keys
$Script:ShellColors = @{
    [ShellColorStyle]::Text = @{ ForegroundColor = 'DarkGray'; BackgroundColor = 'White' }
    
    [ShellColorStyle]::H1 = @{ ForegroundColor = 'Blue'; BackgroundColor = 'White' }
    [ShellColorStyle]::H2 = @{ ForegroundColor = 'Magenta'; BackgroundColor = 'White' }
    [ShellColorStyle]::H3 = @{ ForegroundColor = 'Green'; BackgroundColor = 'White' }
    [ShellColorStyle]::H4 = @{ ForegroundColor = 'Black'; BackgroundColor = 'White' }
    
    [ShellColorStyle]::Err1 = @{ ForegroundColor = 'Red'; BackgroundColor = 'White' }
    [ShellColorStyle]::Err2 = @{ ForegroundColor = 'DarkRed'; BackgroundColor = 'White' }
}


# Schreibt eine voll eingefärbte Zeile auf die Shell,.
# um die Lesbarkeit zu perfektionieren
Function Print-Line {
    [CmdletBinding(DefaultParameterSetName = 'StyleSet')]
    Param (
        [Parameter(Position = 0)]
        [Int]$Indent = 0,

        [Parameter(Position = 1)]
        [Alias('Message')]
        [String]$Msg = '',

		[Parameter(Position = 2, ParameterSetName = 'StyleSet')]
        [ShellColorStyle]$Style = [ShellColorStyle]::Text,

        [Parameter(Position = 2, ParameterSetName = 'DirectColorSet')]
        [Alias('ForegroundColor')]
        [Nullable[ConsoleColor]]$Fore = $null,

        [Parameter(Position = 2, ParameterSetName = 'DirectColorSet')]
        [Alias('BackgroundColor')]
        [Nullable[ConsoleColor]]$Back = $null
    )

	# Write-Host $PsCmdlet.ParameterSetName

    # Farben basierend auf dem verwendeten ParameterSet festlegen
    if ($PSCmdlet.ParameterSetName -eq 'StyleSet') {
        $Fore = $Script:ShellColors[$Style].ForegroundColor
        $Back = $Script:ShellColors[$Style].BackgroundColor
    }

    # $width = (tput cols)
    $width = $Host.UI.RawUI.WindowSize.Width

    $AnsiStart = ''
    $AnsiEnd = ''
    If ($null -ne $Fore) {
        $AnsiStart = Convert-ToAnsiColor -Color $Fore
    }
    If ($null -ne $Back) {
        $AnsiStart += Convert-ToAnsiColor -Color $Back -Background
    }
    If ($AnsiStart -ne '') {
        # Farbe nur beenden, wenn auch definiert
        $AnsiEnd = "`e[0m"
    }

    $OutStr = $(' ' * ($Indent * 2) + $Msg).PadRight($width)
    $OutStr = "$($AnsiStart)$($OutStr)$($AnsiEnd)"

    Write-Host $OutStr -NoNewline
    # Zeile mit den Standard-Farben abschliessen
    Write-Host ''
}



Function Print-Line_02 {
    [CmdletBinding(DefaultParameterSetName = 'StyleSetOhneIndent')]
    Param (
        [Parameter(Position = 0, ParameterSetName = 'StyleSetMitIndent')]
        [Parameter(Position = 0, ParameterSetName = 'DirectColorSetMitIndent')]
        [Int]$Indent = 0,

        [Parameter(Position = 0, ParameterSetName = 'StyleSetOhneIndent')]
        [Parameter(Position = 0, ParameterSetName = 'DirectColorSetOhneIndent')]
        [Parameter(Position = 1, ParameterSetName = 'StyleSetMitIndent')]
        [Parameter(Position = 1, ParameterSetName = 'DirectColorSetMitIndent')]
        [Alias('Message')]
        [String]$Msg = '',

        [Parameter(Position = 1, ParameterSetName = 'StyleSetOhneIndent')]
        [Parameter(Position = 2, ParameterSetName = 'StyleSetMitIndent')]
        [ShellColorStyle]$Style = [ShellColorStyle]::Text,

        [Parameter(Position = 1, ParameterSetName = 'DirectColorSetOhneIndent')]
        [Parameter(Position = 2, ParameterSetName = 'DirectColorSetMitIndent')]
        [Alias('ForegroundColor')]
        [Nullable[ConsoleColor]]$Fore = $null,

        [Parameter(Position = 2, ParameterSetName = 'DirectColorSetOhneIndent')]
        [Parameter(Position = 3, ParameterSetName = 'DirectColorSetMitIndent')]
        [Alias('BackgroundColor')]
        [Nullable[ConsoleColor]]$Back = $null
    )

    # Farben basierend auf dem verwendeten ParameterSet festlegen
    if ($PSCmdlet.ParameterSetName -eq 'StyleSet') {
        $Fore = $Script:ShellColors[$Style].ForegroundColor
        $Back = $Script:ShellColors[$Style].BackgroundColor
    }

    # $width = (tput cols)
    $width = $Host.UI.RawUI.WindowSize.Width

    $AnsiStart = ''
    $AnsiEnd = ''
    If ($null -ne $Fore) {
        $AnsiStart = Convert-ToAnsiColor -Color $Fore
    }
    If ($null -ne $Back) {
        $AnsiStart += Convert-ToAnsiColor -Color $Back -Background
    }
    If ($AnsiStart -ne '') {
        # Farbe nur beenden, wenn auch definiert
        $AnsiEnd = "`e[0m"
    }

    $OutStr = $(' ' * ($Indent * 2) + $Msg).PadRight($width)
    $OutStr = "$($AnsiStart)$($OutStr)$($AnsiEnd)"

    Write-Host $OutStr -NoNewline
    # Zeile mit den Standard-Farben abschliessen
    Write-Host ''
}


Function Print-Line_01() {
    Param (
        [Parameter(Position = 0)]
        [Int]$Indent = 0,

        [Parameter(Position = 1)]
		[Alias('Msg')]
		[String]$Message = '',

        [Parameter(Position = 2)]
        [ConsoleColor]$ForegroundColor = $Null,

        [Parameter(Position = 3)]
        [ConsoleColor]$BackgroundColor = $Null
    )

	# $width = (tput cols)
	$width = $Host.UI.RawUI.WindowSize.Width

	$AnsiStart = ''
	$AnsiEnd = ''
	If ($ForegroundColor -ne $Null) {
		$AnsiStart = Convert-ToAnsiColor -Color $ForegroundColor
	}
	If ($BackgroundColor -ne $Null) {
		$AnsiStart += Convert-ToAnsiColor -Color $BackgroundColor -Background
	}
	If ($AnsiStart -ne '') {
		# Farbe nur beenden, wenn auch definiert
		$AnsiEnd = "`e[0m"
	}

	$OutStr = $(' ' * ($Indent * 2) + $Message).PadRight($width)
	$OutStr = "$($AnsiStart)$($OutStr)$($AnsiEnd)"

	# Dbg
	# Write-Host $OutStr.Replace("`e", "[ESC]") -Fore Cyan

	Write-Host $OutStr -NoNewline
	# Zeile mit den Standard-Farben abschliessen
	Write-Host ''
}


Function Print-Help() {

	# $Text = @{ ForegroundColor = 'DarkGray'; BackgroundColor = 'White' }

	# $H1 = @{ ForegroundColor = 'Blue'; BackgroundColor = 'White' }
	# $H2 = @{ ForegroundColor = 'Magenta'; BackgroundColor = 'White' }
	# $H3 = @{ ForegroundColor = 'Green'; BackgroundColor = 'White' }
	# $H4 = @{ ForegroundColor = 'Black'; BackgroundColor = 'White' }
	
	# $Err1 = @{ ForegroundColor = 'Red'; BackgroundColor = 'White' }
	# $Err2 = @{ ForegroundColor = 'DarkRed'; BackgroundColor = 'White' }

	Print-Line -Msg 'Überschrift 1' -Style H1
	Print-Line -Msg 'Lorem ipsum dolor sit Amet, consetetur sadipscing elitr, Sed diam nonumy'
	
	Print-Line -Msg 'Überschrift 2' -Style H2
	Print-Line -Msg 'Lorem ipsum dolor sit Amet, consetetur sadipscing elitr, Sed diam nonumy' -Style Text

	Print-Line -Msg 'Überschrift 3' -Style H3
	Print-Line -Msg 'Lorem ipsum dolor sit Amet, consetetur sadipscing elitr, Sed diam nonumy' 

	Print-Line -Msg 'Überschrift 4' -Style H4
	Print-Line -Msg 'Lorem ipsum dolor sit Amet, consetetur sadipscing elitr, Sed diam nonumy' 


	Print-Line -Msg 'Fehler 1' -Style Err1
	Print-Line -Msg 'Lorem ipsum dolor sit Amet, consetetur sadipscing elitr, Sed diam nonumy' 
	
	Print-Line -Msg 'Fehler 2' -Style Err2
	Print-Line -Msg 'Lorem ipsum dolor sit Amet, consetetur sadipscing elitr, Sed diam nonumy' 
	
}




#Endregion Shell Farben


#Region Shell Hilfsfunktionen

# !Ex
#	$LastExitOK, $LastExitNOK = Is-LastExitCode-AllOK
Function Is-LastExitCode-AllOK() {
    Param ([Parameter(Mandatory)][Int32]$LastExitCode)
	$IsNOK = $LastExitCode -ne $null -and $LastExitCode -ne 0
	Return (-not $IsNOK), $IsNOK
}

#Endregion Shell Hilfsfunktionen


#Region String

# Konverstiert einen SecureString
Function ConvertFrom-SecureStringToPlainText {
    Param (
        [Parameter(Mandatory)]
        [System.Security.SecureString]$SecureString
    )

    $bstr = [System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($SecureString)
    try {
        [System.Runtime.InteropServices.Marshal]::PtrToStringBSTR($bstr)
    }
    finally {
        [System.Runtime.InteropServices.Marshal]::ZeroFreeBSTR($bstr)
    }
}


# Erzeugt aus einem '' einen SecureString
Function Get-Empty-SecureString {
    Return New-Object System.Security.SecureString
}

#Endregion String


#Region User Input

# Liefert $True, wenn Ja gewählt wurde
Function Ask-JaNein($Frage) {
	$Frage = $Frage.TrimEnd('?')
	
	do {
		$response = Read-Host "$($Frage)? (j/n)"
	} while ($response -notmatch '^(j|n)$')

	Return $response -eq 'j'
}


# Fragt den Benutzer nach dem Benutzernamen
# !Ex
# $UserName = Ask-Username -Prompt 'Bitte Domain-Join Benutzernamen eingeben' -MustBeDomainUser
Function Ask-UserName() {
	Param (
		[String]$Prompt = 'Bitte Benutzernamen eingeben',
		[Switch]$MustBeDomainUser
	)

	# Write-Host $Prompt -ForegroundColor Cyan
	Do {
		# Write-Host $Prompt -ForegroundColor Cyan
		$UserName = Read-Host -Prompt $Prompt

		# Prüfen, ob es ein Domain-Benutzer sein muss
		If ($MustBeDomainUser) {
			# Ein Domain-Benutzername muss ein @ enthalten
			If ($UserName -notmatch '@' `
				-and $UserName -notmatch '\\') {
				Log 1 'Bitte Domain-Usernamen (Domain\User oder User@Domain)' -ForegroundColor Red
				$UserName = ''
			}
		}

	} While ([String]::IsNullOrWhiteSpace($UserName))
	
	Return $UserName.Trim()
}


# Fragt den Benutzer nach einem Passwort
# Optional wird ein leeres Passwort erlaubt
# !Ex
# $ScurePW = Ask-secure-Password -Prompt 'Bitte Domain-Join PW eingeben'
Function Ask-secure-Password() {
	Param (
		[String]$Prompt = 'Bitte das Passwort eingeben',
		[Switch]$AllowEmptyPW
	)

	# Write-Host $Prompt -ForegroundColor Cyan

	# So lange wiederholen, bis ein Passwort eingegeben wurde
	If ($AllowEmptyPW) {
		$SecurePW = Read-Host -AsSecureString -Prompt $Prompt
	} Else {
		# Solange wiederholen, bis ein Passwort eingegeben wurde
		Do {
			# Write-Host $Prompt -ForegroundColor Cyan
			$SecurePW = Read-Host -AsSecureString -Prompt $Prompt
		} While ($SecurePW.Length -eq 0)
	}

	Return $SecurePW
}


# Der Funktion können UserName, PW oder SecurePW übergeben werden
# und sie fragt ab, was fehlt
Function Resolve-Credentials {
    Param (
		[Parameter(Mandatory)]
        [AllowNull()][AllowEmptyString()]
        [String]$UserName,

		[Parameter(Mandatory)]
        [AllowNull()][AllowEmptyString()]
        [Object]$PWTextOrSecureStr,

		[Switch]$AllowEmptyPW,
        [Switch]$MustBeDomainUser,
        [String]$UsernamePrompt = 'Benutzername',
        [String]$PWPrompt = 'Passwort'
    )

	# Den UserName validieren
    If ([String]::IsNullOrWhiteSpace($UserName)) {
        $UserName = Ask-Username -Prompt $UsernamePrompt -MustBeDomainUser:$MustBeDomainUser
    }

	# Analyse von $PWTextOrSecureStr
	$SecurePW = $Null
	$PW = $Null
	If ($Null -ne $PWTextOrSecureStr -and $PWTextOrSecureStr.GetType().Name -eq 'String') {
		$SecurePW = $Null
		$PW = $PWTextOrSecureStr
	} ElseIf ($Null -ne $PWTextOrSecureStr -and $PWTextOrSecureStr.GetType().Name -eq 'SecureString') {
		$SecurePW = $PWTextOrSecureStr
		$PW = $Null
	}

    If ($null -eq $SecurePW) {
        # Ein leeres PW wäre OK, $Null aber nicht
		If ($null -eq $PW -or
			(-not $AllowEmptyPW -and [String]::IsNullOrEmpty($PW)) ) {
            $SecurePW = Ask-secure-Password -Prompt $PWPrompt -AllowEmptyPW:$AllowEmptyPW
        } Else {
			# Wenn $PW leer ist, muss der SecureString speziell erzeugt werden
			If ([String]::IsNullOrEmpty($PW)) {
				$SecurePW = Get-Empty-SecureString
			} Else {
				$SecurePW = ConvertTo-SecureString $PW -AsPlainText -Force
			}
        }
	}

    Return [pscustomobject]@{
        UserName = $UserName
        SecurePW = $SecurePW
    }
}

#Endregion User Input


#Region Root User

# True, wenn wir Root-Rechte haben
Function Is-Root() {
    Param ([Switch]$Dbg)
	$userID = $(id -u)
	If( $Dbg ) {
		Log 4 "Is-Root(): UserID = $userID" -Fore DarkGray
	}
    Return $userID -eq 0
}

Function Is-not-Root() {
    Param ([Switch]$Dbg)
	Return -not (Is-Root -Dbg:$Dbg)
}

#Endregion Root User


#Region MacOS User handling

#  Meldet den aktuellen macOS User ab
Function Logoff-MacOS-User($LoggedInUser) {
	try {
		# Diese Befehle erfordern sudo, um andere Benutzer abzumelden.
		# Für den angemeldeten Benutzer selbst reicht oft der direkte Aufruf ohne sudo aus,
		# aber da das Skript als sudo läuft, ist es gut, den launchctl-Ansatz zu verwenden.
		# Alternativ: & osascript -e 'tell app "System Events" to log out'
		# Die launchctl bootout Methode ist robuster für Skripte, die als root laufen.
		
		# UID des angemeldeten Benutzers ermitteln
		$LoggedInUserUID = $(id -u $LoggedInUser)
		# Logoff
		& launchctl bootout user/$LoggedInUserUID
		Log 0 'Benutzerabmeldung erfolgreich eingeleitet.'
	} catch {
		Log 4 "Fehler beim Einleiten der Benutzerabmeldung: $($_.Exception.Message)" -Fore red
		Log 5 "Bitte melden Sie sich manuell ab und wieder an, um FileVault zu aktivieren" -Fore Magenta
	}	
}


#Endregion MacOS User handling


#Region Hostname

# Gibt auf der Konsole die Info der Hostnamen aus
# !Ex
# $oHostnamesInfo = Get-HostNames-Info
# Print-HostNamesInfo -oHostnamesInfo $oHostnamesInfo
Function Print-HostNamesInfo($oHostnamesInfo) {

	Log 0 'HostNamesInfo'

	Log 1 'Domain-Join : ' -NoNewline
	If ($oHostnamesInfo.IsDomainJoined) {
		Log -Append -Msg 'Ja' -Fore Red
	} Else {
		Log -Append -Msg 'Nein' -Fore Red
	}
	
    Log 1 "JoinedDomain: $($oHostnamesInfo.JoinedDomain)"

	Log 1 'Aktuelle Hostnamen'
    Log 2 "  MacOS ComputerName   : $($oHostnamesInfo.MacOS_ComputerName)"
	# HostName == FQDN
    Log 2 "  MacOS HostName (Fqdn): $($oHostnamesInfo.MacOS_HostName)"

    Log 2 "  MacOS LocalHostName  : $($oHostnamesInfo.MacOS_LocalHostName)"
    Log 2 "  > Effektiv           : $($oHostnamesInfo.MacOS_LocalHostNameEffektiv)"

	Log 1 'FQDN (berechnet aus HostName)'
	Log 2 "  FQDN                 : $($oHostnamesInfo.FQDN)"
	Log 2 "  FQDN HostName        : $($oHostnamesInfo.FQDN_HostName)"
	Log 2 "  FQDN Domain          : $($oHostnamesInfo.FQDN_Domain)"
}


# Berechnet alle Infos zum MacOS Hostnamen
# $oHostnamesInfo = Get-HostNames-Info
Function Get-HostNames-Info() {
	# Hostnamen bestimmen
	# HostName == FQDN
	$MacOS_HostName = "$(scutil --get HostName 2>$null)"

	$oHostnamesInfo = [PSCustomObject][Ordered]@{
		IsDomainJoined = Is-Domain-joined
		JoinedDomain = Get-Joined-Domain
		MacOS_ComputerName = "$(scutil --get ComputerName 2>$null)"
		# HostName == FQDN
		MacOS_HostName = $MacOS_HostName
		MacOS_LocalHostName = "$(scutil --get LocalHostName 2>$null)"
		MacOS_LocalHostNameEffektiv = "$(scutil --get LocalHostName 2>$null).local"
		# FQDN == HostName
		FQDN = $MacOS_HostName
		FQDN_HostName = $MacOS_HostName.Split('.')[0]
		FQDN_Domain = ($MacOS_HostName.Split('.')[1..99] -join '.')
	}
	Return $oHostnamesInfo
}


# Ermittelt die aktuellen Hostnamen durch Analyse von $oHostnamesInfo
# $CurrentHostName = Get-Current-HostName $oHostnamesInfo
Function Get-Current-HostName() {
	Param(
		# Optional
		[PSCustomObject]$oHostnamesInfo
	)

	## Init
	If ($oHostnamesInfo -eq $null) {
		# Wenn kein oHostnamesInfo übergeben wurde, dann holen wir die Infos
		$oHostnamesInfo = Get-HostNames-Info
	}

	# Zuerst den LocalHostName verwenden, wenn er gesetzt ist
	If (Has-Value $oHostnamesInfo.MacOS_LocalHostName) {
		Return $oHostnamesInfo.MacOS_LocalHostName
	}
	# Sonst ComputerName
	If (Has-Value $oHostnamesInfo.MacOS_ComputerName) {
		Return $oHostnamesInfo.MacOS_ComputerName
	}
	# Sonst das erste Element von HostName (=FQDN)
	Return $oHostnamesInfo.MacOS_HostName.Split('.')[0]
}


# Setzt die MacOS Hostnamen
Function Set-HostNames {
	Param (
		[AllowEmptyString()][String]$NewHostName,
		[AllowEmptyString()][String]$DomainName,
		[Switch]$Force,
		[Switch]$Dbg
	)
	

	# 📌 Empfohlen bei domain-joined Macs:
	# Wenn dein Mac Mitglied einer Active Directory-Domain ist, 
	# ist es besonders wichtig, die Namen konsistent zu setzen:
	#
	# sudo scutil --set HostName "myhost.mydomain.com"
	# sudo scutil --set ComputerName "myhost"
	# sudo scutil --set LocalHostName "myhost"


	## Check
	If ($Dbg) {
		Log 4 'Set-HostNames(), vor Bereinigung:'
		Log 5 "NewHostName: '$NewHostName'"
		Log 5 "DomainName : '$DomainName'"
	}

	If ((Is-NullOrEmpty $NewHostName) -and (Is-NullOrEmpty $DomainName)) {
		Log 4 'Set-HostNames(): Nichts zu tun: Weder Hostname noch Domain angegeben' -Fore Red
		Log 5 "NewHostName: $NewHostName" -Fore Red
		Log 5 "DomainName : $DomainName" -Fore Red
		Return
	}


	## Init
	# Hostnamen lesen
	$oThisHostnamesInfo = Get-HostNames-Info

	# Params bereinigen
	If (Has-Value $NewHostName) { $NewHostName = $NewHostName.Trim() }
	If (Has-Value $DomainName) { $DomainName = $DomainName.Trim() }

	If ($Dbg) {
		Log 4 'Set-HostNames(), nach Bereinigung:'
		Log 5 "NewHostName: '$NewHostName'"
		Log 5 "DomainName : '$DomainName'"
	}


	# Ändert der Hostname?
	If (Is-NullOrEmpty $NewHostName) {
		# Kein neuer Hostname gesetzt - den alten verwenden
		$NewHostName = Get-Current-HostName $oThisHostnamesInfo

		If ($Dbg) {
			Log 4 'Set-HostNames(), Kein neuer Hostname angegeben, nütze den aktuell gesetzten:'
			Log 5 "NewHostName: '$NewHostName'"
		}
	}


	# Den DQDN bestimmen
	# Ohne Domäne ist der full-qualified Domain Name (FQDN) nur der Hostname
	$FQDN = $NewHostName
	If (Has-Value $DomainName) {
		$FQDN = ('{0}.{1}' -f $NewHostName, $DomainName.Trim())	
	}
	If ($Dbg) {
		Log 4 'Set-HostNames(), berechneter FQDN:'
		Log 5 "FQDN: '$FQDN'"
	}


	If ($Force) {
		# Write-Host "Setze Hostnamen: '$NewHostName'.."
		If ($Dbg) { Log 4 '-Force aktiv' }

		# Änderte der ComputerName?
		If ($oThisHostnamesInfo.MacOS_ComputerName -ne $NewHostName) {
			If ($Dbg) { Log 5 "Aktualisiere ComputerName: '$NewHostName'" }
			sudo scutil --set ComputerName "$NewHostName"
		} Else {
			If ($Dbg) { Log 5 'ComputerName: Schon aktuell' }
		}

		# Änderte der HostName?
		# HostName == FQDN
		If ($oThisHostnamesInfo.MacOS_HostName -ne $FQDN) {
			If ($Dbg) { Log 5 "Aktualisiere HostName: '$FQDN'" }
			sudo scutil --set HostName "$FQDN"
		} Else {
			If ($Dbg) { Log 5 'HostName: Schon aktuell' }
		}

		# Änderte der LocalHostName?
		If ($oThisHostnamesInfo.MacOS_LocalHostName -ne $NewHostName) {
			# Erzeugt automatisch $NewHostName.local
			If ($Dbg) { Log 5 "Aktualisiere LocalHostName: '$NewHostName'" }
			sudo scutil --set LocalHostName "$NewHostName"
		} Else {
			If ($Dbg) { Log 5 'LocalHostName: Schon aktuell' }
		}

		Log 2 'Hostnamen wurden gesetzt' -Fore Green
		
	} Else {
		# Write-Host "Würde Hostnamen setzen:"
		If ($Dbg) { Log 4 '-Force NICHT aktiv' }
		
		Log 2 'MacOS ComputerName : ' -NoNewLine
		If ($oThisHostnamesInfo.MacOS_ComputerName -eq $NewHostName) {
			Log -Append -Msg ('{0} (Schon OK)' -f $NewHostName) -Fore Green
		} Else {
			Log -Append -Msg $NewHostName -Fore Red
		}
		
		# HostName == FQDN
		Log 2 "MacOS HostName     : " -NoNewLine
		If ($oThisHostnamesInfo.MacOS_HostName -eq $FQDN) {
			Log -Append -Msg ('{0} (Schon OK)' -f $FQDN) -Fore Green
		} Else {
			Log -Append -Msg $FQDN -Fore Red
		}
		
		Log 2 "MacOS LocalHostName: " -NoNewLine
		If ($oThisHostnamesInfo.MacOS_LocalHostName -eq $NewHostName) {
			Log -Append -Msg ('{0} (Schon OK)' -f $NewHostName) -Fore Green
		} Else {
			Log -Append -Msg $NewHostName -Fore Red
		}
		
	}
}

#Endregion Hostname


#Region Domain-Join

# Ist das MacOS Domain-Joined?
Function Is-Domain-joined() {

    #Region dsconfigad -show

    # Active Directory Forest          = akros.ch
    # Active Directory Domain          = akros.ch
    # Computer Account                 = anb289$

    # Advanced Options - User Experience
    #   Create mobile account at login = Enabled
    #      Require confirmation        = Enabled
    #   Force home to startup disk     = Enabled
    #      Mount home as sharepoint    = Enabled
    #   Use Windows UNC path for home  = Enabled
    #      Network protocol to be used = smb
    #   Default user Shell             = /bin/bash

    # Advanced Options - Mappings
    #   Mapping UID to attribute       = not set
    #   Mapping user GID to attribute  = not set
    #   Mapping group GID to attribute = not set
    #   Generate Kerberos authority    = Enabled

    # Advanced Options - Administrative
    #   Preferred Domain controller    = not set
    #   Allowed admin groups           = not set
    #   Authentication from any domain = Enabled
    #   Packet signing                 = allow
    #   Packet encryption              = allow
    #   Password change interval       = 14
    #   Restrict Dynamic DNS updates   = not set
    #   Namespace mode                 = domain

    #Endregion dsconfigad -show

	$adInfo = dsconfigad -show 2>$null
	$LastExitOK, $LastExitNOK = Is-LastExitCode-AllOK $LastExitCode
	$ADInfoMissing = [String]::IsNullOrWhiteSpace($adInfo)
	$IsDomainJoined = $LastExitOK -and $ADInfoMissing -eq $False

	# Write-Host "adInfo        : $adInfo"
	# Write-Host "LastExitOK    : $LastExitOK"
	# Write-Host "ADInfoMissing : $ADInfoMissing"
	# Write-Host "IsDomainJoined: $IsDomainJoined"
	# Write-Host "IsDomainJoined: $IsDomainJoined"

	Return $IsDomainJoined
}


# Liest die Domäne, zu der das MacOS joined ist
# !Ex
#	$JoinedDomain = Get-Joined-Domain
Function Get-Joined-Domain {
    $dsconfigadOutput = & dsconfigad -show 2>$null
	$LastExitOK, $LastExitNOK = Is-LastExitCode-AllOK $LastExitCode
	If ($LastExitOK -and $dsconfigadOutput) {
        $Domain = ($dsconfigadOutput | ? { $_ -match "^Active Directory Domain" }) -replace "^.*=\s*", ""
		Return $Domain.Trim()
	} Else {
        Return $Null
	}
}


# Domain-Join für MacOS
# Wenn UserName oder PW fehlt, fragt das Script danach
Function Join-MacOS-to-Domain {
	[CmdletBinding(DefaultParameterSetName = 'Secure')]
    Param (
		[Parameter(Mandatory)]
        [String]$DomainName,
		# Optional ein UserName
        [Object]$UserName,
		# Optional ein PW
        [Object]$PW
    )

	Log 0 'Starte Domain-Join'
	
	### Prepare
	$DomainName = $DomainName.Trim()

	## UserName und PW prüfen
	$Cred = Resolve-Credentials -UserName $UserName `
								-PWTextOrSecureStr $PW `
								-MustBeDomainUser `
								-UsernamePrompt 'Domain-Join Benutzernamen (user@domain.com)' `
								-PWPrompt 'Domain-Join Passwort' `

	### Checks
	If (Is-Domain-joined) {
		$JoinedDomain = Get-Joined-Domain
		If ($JoinedDomain -eq $DomainName) {
			Log 2 "Computer ist schon in der Domäne: $DomainName" -Fore Green
			Return $True
		} Else {
			Log 4 'Fehler beim Domain-Join!' -Fore Red
			Log 5 "Computer ist bereits in einer anderen Domäne: $JoinedDomain" -Fore Magenta
			Return $False
		}
	}

	## Init
	If (Is-not-Root) {
		Log 4 'Domain-Join benötigt Root' -Fore Red
		Log 5 'Abbruch' -Fore Magenta
		Return $False
	}


	# Aktuelle Hostnamen abrufen
	$oHostnamesInfo = Get-HostNames-Info
	$ComputerName = $oHostnamesInfo.MacOS_LocalHostName
	
	$ResDomainJoinOK = $False
    Try {
		# Das PW im Klartext
		$PW = ConvertFrom-SecureStringToPlainText -SecureString $Cred.SecurePW

		# Domain Join ausführen
        # 2>&1 stderr nach stdout

		# Write-Host "DomainName   : $DomainName"
		# Write-Host "Cred.UserName: $($Cred.UserName)"
		# Write-Host "Cred.SecurePW: $($Cred.SecurePW)"
		# Write-Host "PW           : $PW"

        $ResCmdline = & /usr/sbin/dsconfigad -add "$DomainName" -computer "$ComputerName" -force -username "$($Cred.UserName)" -password "$PW" 2>&1
		$LastExitOK, $LastExitNOK = Is-LastExitCode-AllOK $LastExitCode		

		### Fehlerauswertung
		## dsconfigad: Meldungen und Fehler
		# 	Alles OK
		# 	> keine antwort / ausgabe!

		$HasNoErrors = $True
		Switch ($ResCmdline) {
			# ❯ Falsches PW
			# dsconfigad: Invalid credentials supplied for binding to the server
			{ ([Regex]::Matches($_, '\b(invalid|credentials?|supplied|username|password)\b').Count -ge 2) } {
				$HasNoErrors = $False
				Log 5 $ResCmdline -Fore DarkGray
				Log 4 'Domain-Join failed:' -Fore Red
				Log 4 'Benutzername oder Passwort ungültig' -Fore Red
				Break
			}
			# ❯ Bereits joined
			# dsconfigad: This computer is already 'bound' to Active Directory. You must 'unbind' with '-remove' first
			{ ([Regex]::Matches($_, '\b(already|bound)\b').Count -ge 2) } {
				$HasNoErrors = $False
				Log 5 $ResCmdline -Fore DarkGray
				Log 4 'Domain-Join failed:' -Fore Red
				Log 4 'Computer ist schon domain-joined' -Fore Red
				Break
			}
			# ❯ Domäne wird nicht gefunden
			# dsconfigad: Node name wasn't found. (2000)
			{ ([Regex]::Matches($_, "\b(Node|wasn't|found|not)\b").Count -ge 3) } {
				$HasNoErrors = $False
				Log 5 $ResCmdline -Fore DarkGray
				Log 4 'Domain-Join failed:' -Fore Red
				Log 4 "Domäne nicht gefunden: '$DomainName'" -Fore Red
				Break
			}
			# ❯ Hostname existiert schon
			# Computer account already exists! Bind to Existing? (y/n):
			# > -force angeben!
			{ ([Regex]::Matches($_, "\b(account|already|exists)\b").Count -ge 2) } {
				$HasNoErrors = $False
				Log 5 $ResCmdline -Fore DarkGray
				Log 4 'Domain-Join failed:' -Fore Red
				Log 4 "Computer existiert schon in der Domäne!" -Fore Red
				Log 5 '-force nützen!' -Fore Magenta
				Break
			}
			default {
				If ($LastExitNOK) {
					$HasNoErrors = $False
					Log 4 "Unbekannter Fehler beim Hinzufügen zur Domäne" -Fore Red
					Log 5 "Details: $ResCmdline" -Fore Magenta
				}
			}
		}

        # War der Domain-Join OK?
        If ($LastExitOK -and $HasNoErrors) {
            Log 3 "Mac erfolgreich zur Domäne hinzugefügt" -ForegroundColor Green
            $ResDomainJoinOK = $True
        } else {
			Log 4 "Fehler beim Hinzufügen des Macs zur Domäne" -Fore Red
            $ResDomainJoinOK = $False
        }

    } Catch {
		Log 4 'Exception:' -Fore Red
		Log 5 "$($_.Exception.Message)" -Fore Magenta
        $ResDomainJoinOK = $False
    
	} Finally {
		# Sicheres Löschen des Passworts aus dem Speicher
        if ($Cred -and $Cred.SecurePW) {
            [System.Runtime.InteropServices.Marshal]::ZeroFreeBSTR([System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($Cred.SecurePW))
        }
        
        # Sicheres Löschen des Klartext-Passworts
        if ($PW) {
            # Überschreiben mit leerer Zeichenfolge und dann mit $null
            $PW = ""
            $PW = $null
        }
        
        # Garbage Collection forcieren
        [System.GC]::Collect()		
    }
}


# MacOS: Löscht den Domain-Join
Function Unjoin-MacOS-from-Domain {
    Param (
		# Optional ein Username
        [Object]$UserName,
		# Optional ein PW
        [Object]$PW
    )

    Log 0 'Entferne Mac aus der Domäne'

	### Checks
	If (-not (Is-Domain-joined)) {
		Log 2 'Computer ist in keiner Domäne' -Fore Green
		Return $True
	}

	If (Is-not-Root) {
		Log 4 'Domain-UnJoin benötigt Root' -Fore Red
		Log 5 'Abbruch' -Fore Magenta
		Return $False
	}

	## UserName und PW prüfen
	$Cred = Resolve-Credentials -UserName $UserName `
								-PWTextOrSecureStr $PW `
								-MustBeDomainUser `
								-UsernamePrompt 'Domain-Join Benutzernamen (user@domain.com)' `
								-PWPrompt 'Domain-Join Passwort' `

    
    try {
		# Das PW im Klartext
		$PW = ConvertFrom-SecureStringToPlainText -SecureString $Cred.SecurePW

		# Domäne trennen
        # 2>&1 stderr nach stdout
        $ResCmdline = & /usr/sbin/dsconfigad -remove -force -u "$($Cred.UserName)" -p "$PW" 2>&1
		$LastExitOK, $LastExitNOK = Is-LastExitCode-AllOK $LastExitCode


		## Fehlerauswertung
		# » Falsches PW:
		# 	KEINE Meldung, auf dem MacOS wird der Join entfernt
		$HasNoErrors = $True
		Switch -regex ($ResCmdline) {
			# ❯ Ist gar nicht joined:
			#	dsconfigad: This computer is not Bound to Active Directory. You must bind it first.
			{ ([Regex]::Matches($_, '\b(not|bound)\b').Count -ge 2) } {
				$HasNoErrors = $False
				Log 4 'Computer ist nicht domain-joined' -For Red
				Break
			}
			default {	
				If ($LastExitNOK) {
					$HasNoErrors = $False
					Log 4 "Unbekannter Fehler Trennen der Domäne" -Fore Red
					Log 5 "Details: $ResCmdline" -Fore Magenta
				}
			}
		}

        # Überprüfe den Exit-Code des letzten Befehls und die Ausgabe auf Fehlermuster.
        If ($LastExitOK -and $HasNoErrors) {
            Log 1 'Mac erfolgreich aus der Domäne entfernt' -Fore Green
			Start-Sleep -Milliseconds 4500
            Return $True
        } else {
			Log 4 'Fehler beim Trennen der Domäne' -Fore Red
			Log 5 '- Netzverbindung zur Domäne OK?' -Fore Magenta
			Log 5 '- Credential OK?' -Fore Magenta
			Log 5 'Abbruch' -Fore Magenta
            return $False
        }

    } Catch {
		Log 4 'Exception:' -Fore Red
		Log 5 "$($_.Exception.Message)" -Fore Magenta
        Return $False

	} Finally {
		# Sicheres Löschen des Passworts aus dem Speicher
        if ($Cred -and $Cred.SecurePW) {
            [System.Runtime.InteropServices.Marshal]::ZeroFreeBSTR([System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($Cred.SecurePW))
        }
        
        # Sicheres Löschen des Klartext-Passworts
        if ($PW) {
            # Überschreiben mit leerer Zeichenfolge und dann mit $null
            $PW = ""
            $PW = $null
        }
        
        # Garbage Collection forcieren
        [System.GC]::Collect()		
    }
}

#Endregion Domain-Join


#Region MacOS File Vault

# Liefert $True, wenn FileVault aktiviert ist
Function Is-FileVault-Enabled {
    try {
		# fdesetup status
		# 	FileVault is Off.
		# 	Deferred enablement appears to be active for user 'root'
		# 	> kann so wieder ausgeschaltet werden:
		# 		sudo fdesetup disable -user root
        $fileVaultStatus = (& fdesetup status) -join "`n"
        If ($fileVaultStatus -match "FileVault is On") {
            return $true
        } else {
            return $false
        }
    } catch {
        Write-Error "Fehler beim Abrufen des FileVault-Status: $($_.Exception.Message)"
        return $false
    }
}


#Endregion MacOS File Vault


#Region URLs

# !Ex
# 	Join-URLs 'https://eu.ninjarmm.com/', 'oauth/token/'
# » 
#	https://eu.ninjarmm.com/oauth/token
Function Join-URLs() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[String[]]$URLs,
		[Switch]$AddSlash
	)
	
	$Res = $Null
	ForEach($URL in $URLs) {
		$URL = $URL.Trim()
		If ($Null -eq $Res) {
			$Res = $URL.TrimEnd('/')
		} Else {
			$Res = '{0}/{1}' -f $Res, $URL.TrimEnd('/')
		}
	}
	
	If ($AddSlash) {
		Return $Res + '/'
	} Else {
		Return $Res
	}
}

#Endegion URLs


#Region NinjaOne

# Authentifiziert sich am NinjaOne REST API
# Liefert den AuthHeader für folgeaufrufe zurück
# -Dbg zeigt Details zum Auth-Token an
# 
# !M
# https://app.NinjaOne.com/apidocs-beta/authorization/overview
# 250415
Function Auth-NinjaOne() {
	Param(
		[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]

		# Zugriffs-Scopes
		# Wenn nicht definiert, werden alle gewährt
		#  @('monitoring', 'management', 'control')
		# !Ex https://github.com/homotechsual/NinjaOne/blob/bfbc4832473fef510fe11b5a759f2d91f529b999/Source/Public/PSModule/Connect/Connect-NinjaOne.ps1#L142
		[ValidateSet('monitoring', 'management', 'control')]
		[String[]]$Scopes,
		[Switch]$Dbg
	)
	
	If ($Dbg) {
		Log 4 'Auth-NinjaOne()' -Fore DarkGray
	}
	
	## Init
	# Allenfalls alle Scopes definieren
	If ($Null -eq $Scopes) {
		$Scopes = @('monitoring', 'management', 'control')
	}
	
	
	# Sie müssen alle kleingeschrieben sein
	$Scopes = $Scopes | % { $_.ToLower() }
	# Write-Host $Scopes | Out-String
	
	
	## Config
	$AuthURL = Join-URLs @($Script:NinjaOneBaseURL, 'oauth/token/')

	$Body = @{
		grant_type = 'client_credentials'
		client_id = $Script:NinjaOneClientID
		client_secret = $Script:NinjaOneClientSecret
		redirect_uri = 'https://localhost'
		scope = ($Scopes -join ' ')
	}
	
	
	## Prepare

	$AuthHeaders = New-Object 'System.Collections.Generic.Dictionary[[String],[String]]'
	# Muss kleingeschrieben sein!
	$AuthHeaders.Add('Accept', 'application/json')
	$AuthHeaders.Add('Content-Type', 'application/x-www-form-urlencoded')
	
	
	## Main

	$AuthToken = Invoke-RestMethod -Uri $AuthURL -Method POST `
												-Headers $AuthHeaders `
												-Body $body
	# Return $AuthToken

	If ($Dbg) {
		Log 4 ($AuthToken | Out-String)
	}

	$AccessToken = $AuthToken | Select-Object -ExpandProperty 'access_token' -EA 0

	$AuthHeader = New-Object 'System.Collections.Generic.Dictionary[[String],[String]]'
	$AuthHeader.Add('accept', 'application/json')
	$AuthHeader.Add('Authorization', "Bearer $AccessToken")

	Return $AuthHeader
}


# Sucht das NinjaOne-Objekt für ein Gerät
Function Get-NinjaOne-Device-by-SystemName() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[Object]$AuthHeader,
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=1)]
		[String]$SystemName,
		[Switch]$Dbg
	)
	
	# Config
	$Limit = 10
	
	# https://app.NinjaOne.com/apidocs-beta/core-resources/operations/search
	$SearchDevicesURL = 'v2/devices/search'

	$ApiURL = Join-URLs @($Script:NinjaOneBaseURLAPI, $SearchDevicesURL)
	
	$QryURL = Join-URLs @($ApiURL, `
								('?q={0}&limit={1}' -f [URI]::EscapeDataString($SystemName), $Limit))
	
	If ($Dbg) { 
		log 4 'QryURL'
		log 5 $QryURL
	}
	
	
	# Die Geräte suchen
	$Devices = Invoke-RestMethod -Uri $QryURL -Method GET -Headers $AuthHeader
	# Return $Devices

	# Das Gerät filtern
	$MyDevice = $Devices.Devices | ? systemName -eq $SystemName
	# $ID = $MyDevice.id
	# Write-Host "ID: $ID"
	Return $MyDevice
}


# Sucht von einem NinjaOne Geräte-Objekt
# alle Felder, die es hat
Function Get-NinjaOne-Device-Fields() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[Object]$AuthHeader,
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=1)]
		[Int]$DeviceID,
		[Switch]$Dbg
	)
	
	## Config

	# https://app.NinjaOne.com/apidocs-beta/core-resources/operations/search
	$GetURL = "v2/device/$DeviceID/custom-fields"

	$ApiURL = Join-URLs @($Script:NinjaOneBaseURLAPI, $GetURL)
	If ($Dbg) { 
		Log 4 'ApiURL'
		Log 5 $ApiURL
	}

	# Die Fields auslesen
	$Fields = Invoke-RestMethod -Uri $ApiURL -Method GET -Headers $AuthHeader
	Return $Fields
}


# Setzt den Wert eines Custom Field
Function Set-NinjaOne-Device-Field() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[Object]$AuthHeader,
	
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=1)]
		[Int]$DeviceID,

		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName)]
		[String]$FieldName = 'ssfmahandynr',

		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName)]
		[String]$FieldValue,
		[Switch]$Dbg
	)
	
	## Config

	# https://app.NinjaOne.com/apidocs/?links.active=core#/devices/getNodeCustomFields
	$GetURL = "v2/device/$DeviceID/custom-fields"

	$ApiURL = Join-URLs @($Script:NinjaOneBaseURLAPI, $GetURL)
	If ($Dbg) { 
		Log 4 'ApiURL'
		Log 5 $ApiURL
	}
	
	$BodyProps = @{}
	$BodyProps.Add($FieldName, $FieldValue)
	# Return $BodyProps
	# Return (ConvertTo-Json -InputObject $BodyProps -Depth 100)

	# Die Fields Setzen
	$Res = Invoke-RestMethod -Uri $ApiURL -Method Patch `
									-Headers $AuthHeader `
									-ContentType 'application/json' `
									-Body (ConvertTo-Json -InputObject $BodyProps -Depth 100)

	Return $Res
}

#Endregion NinjaOne



#Region Netzwerk


# Löscht den DNS Cache
Function Clear-DNS-Cache() {
	If (Is-not-Root) {
		Log 4 'Benötigt Root: Clear-DNS-Cache()' -Fore Red
		Log 5 'Abbruch' -Fore Magenta
		Return $False
	}

	# Leert den DNS Cache
	# https://www.macworld.com/article/671688/how-to-clear-dns-cache-macos.html
	# sudo dscacheutil -flushcache; sudo killall -HUP mDNSResponder
	& dscacheutil -flushcache
	& killall -HUP mDNSResponder
}


# Liefert die erste IPv4 Adresse für eine Domäne
# !Ex, liefert: [System.Net.IPAddress]
# $oIPv4 = Resolve-Domain-get-IPv4 -Domain 'akros.ch'
# $oIPv4.IPAddressToString
Function Resolve-Domain-get-IPv4($Domain) {
	# Punkt am Ende erzwingen
	$Domain = ('{0}.' -f $Domain.Trim().TrimEnd('.'))

	Try {
		$oHost = [System.Net.Dns]::GetHostEntry($Domain)
		# Erste IPv4
		$oFirstIp = $oHost.AddressList | ? { $_.AddressFamily -eq 'InterNetwork' } | Select-Object -First 1
		
		Return $oFirstIp
	} Catch {
		Return $Null
	}
}


# Prüft, ob die Domäne mit einer LAN IP-Adresse aufgelöst wird
Function Is-Computer-in-Domain-LAN($Domain) {
	$oIPv4 = Resolve-Domain-get-IPv4 -Domain $Domain
	If (-not $oIPv4) { Return $False }
	
	$FirstOctett = $oIPv4.IPAddressToString.Split('.')[0]
	# Sind wir im 10.x Netzwerk?
	Return ($FirstOctett -eq 10)
}

#Endregion Netzwerk


#Region NinjaOne Funcs

# Gibt die automatischen Variablen von NinjaOne aus
# NINJA_AGENT_NODE_ID ist die DeviceID
# 
# !Ex
# 	NINJA_AGENT_MACHINE_ID: V5;C69HWWJ1Q1;84:2F:57:3B:2B:41
# 	NINJA_AGENT_NODE_ID: 167
# 	NINJA_AGENT_VERSION_INSTALLED: 8.0.3261
# 	NINJA_COMPANY_NAME: Noser Engineering AG
# 	NINJA_DATA_PATH: /Applications/NinjaRMMAgent/programdata
# 	NINJA_EXECUTING_PATH: /Applications/NinjaRMMAgent/programfiles
# 	NINJA_LOCATION_ID: 20
# 	NINJA_LOCATION_NAME: (Kein Standort)
# 	NINJA_ORGANIZATION_ID: 2
# 	NINJA_ORGANIZATION_NAME: AKROS AG
# 	NINJA_PATCHER_VERSION_INSTALLED: 8.0.3261
# 	NINJA_SECURE_CHAIN_TOKEN: d61c462d-182b-4df7-9e4c-603e287c1f15
Function Print-NinjaOne-AutomaticVars {
	Log 0 'Automatische Variablen von NinjaOne'
	# Alle automatischen NinjaOne Variablen ausgeben
	Get-ChildItem Env: | Where-Object { $_.Name -like 'NINJA*' } | ForEach-Object { Log 1 "$($_.Name): $($_.Value)" }	
}


# Authentifiziert sich am NinjaOne REST API
# Liefert den AuthHeader für folgeaufrufe zurück
# -Dbg zeigt Details zum Auth-Token an
# 
# !M
# https://app.NinjaOne.com/apidocs-beta/authorization/overview
# 250415
Function Auth-NinjaOne() {
	Param(
		[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]

		# Zugriffs-Scopes
		# Wenn nicht definiert, werden alle gewährt
		#  @('monitoring', 'management', 'control')
		# !Ex https://github.com/homotechsual/NinjaOne/blob/bfbc4832473fef510fe11b5a759f2d91f529b999/Source/Public/PSModule/Connect/Connect-NinjaOne.ps1#L142
		[ValidateSet('monitoring', 'management', 'control')]
		[String[]]$Scopes,
		[Switch]$Dbg
	)
	
	
	## Init
	# Allenfalls alle Scopes definieren
	If ($Null -eq $Scopes) {
		$Scopes = @('monitoring', 'management', 'control')
	}
	
	
	# Sie müssen alle kleingeschrieben sein
	$Scopes = $Scopes | % { $_.ToLower() }
	# Write-Host $Scopes | Out-String
	
	
	## Config
	$AuthURL = Join-URLs @($Script:NinjaOneBaseURL, 'oauth/token/')

	$Body = @{
		grant_type = 'client_credentials'
		client_id = $Script:NinjaOneClientID
		client_secret = $Script:NinjaOneClientSecret
		redirect_uri = 'https://localhost'
		scope = ($Scopes -join ' ')
	}
	
	
	## Prepare

	$AuthHeaders = New-Object 'System.Collections.Generic.Dictionary[[String],[String]]'
	# Muss kleingeschrieben sein!
	$AuthHeaders.Add('Accept', 'application/json')
	$AuthHeaders.Add('Content-Type', 'application/x-www-form-urlencoded')
	
	
	## Main

	$AuthToken = Invoke-RestMethod -Uri $AuthURL -Method POST `
												-Headers $AuthHeaders `
												-Body $body
	# Return $AuthToken

	If ($Dbg) {
		Log 4 ($AuthToken | Out-String)
	}

	$AccessToken = $AuthToken | Select-Object -ExpandProperty 'access_token' -EA 0

	$AuthHeader = New-Object 'System.Collections.Generic.Dictionary[[String],[String]]'
	$AuthHeader.Add('accept', 'application/json')
	$AuthHeader.Add('Authorization', "Bearer $AccessToken")

	Return $AuthHeader
}


# Sucht das NinjaOne-Objekt für ein Gerät
Function Get-NinjaOne-Device-by-SystemName() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[Object]$AuthHeader,
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=1)]
		[String]$SystemName,
		[Switch]$Dbg
	)
	
	# Config
	$Limit = 10
	
	# https://app.NinjaOne.com/apidocs-beta/core-resources/operations/search
	$SearchDevicesURL = 'v2/devices/search'

	$ApiURL = Join-URLs @($Script:NinjaOneBaseURLAPI, $SearchDevicesURL)
	
	$QryURL = Join-URLs @($ApiURL, `
								('?q={0}&limit={1}' -f [URI]::EscapeDataString($SystemName), $Limit))
	
	If ($Dbg) { 
		Log 4 'QryURL'
		Log 5 $QryURL
	}
	
	
	# Die Geräte suchen
	$Devices = Invoke-RestMethod -Uri $QryURL -Method GET -Headers $AuthHeader
	# Return $Devices

	# Das Gerät filtern
	$MyDevice = $Devices.Devices | ? systemName -eq $SystemName
	# $ID = $MyDevice.id
	# Write-Host "ID: $ID"
	Return $MyDevice
}


# Sucht von einem NinjaOne Geräte-Objekt
# alle Felder, die es hat
Function Get-NinjaOne-Device-Fields() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[Object]$AuthHeader,
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=1)]
		[Int]$DeviceID,
		[Switch]$Dbg
	)
	
	## Config

	# https://app.NinjaOne.com/apidocs-beta/core-resources/operations/search
	$GetURL = "v2/device/$DeviceID/custom-fields"

	$ApiURL = Join-URLs @($Script:NinjaOneBaseURLAPI, $GetURL)
	If ($Dbg) { 
		Log 4 'ApiURL'
		Log 5 $ApiURL
	}

	# Die Fields auslesen
	$Fields = Invoke-RestMethod -Uri $ApiURL -Method GET -Headers $AuthHeader
	Return $Fields
}


# Setzt den Wert eines Custom Field
Function Set-NinjaOne-Device-Field() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[Object]$AuthHeader,
	
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=1)]
		[Int]$DeviceID,

		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName)]
		[String]$FieldName_ssfClientTasksData = 'ssfmahandynr',

		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName)]
		[String]$FieldValue,
		[Switch]$Dbg
	)
	
	## Config

	# https://app.NinjaOne.com/apidocs/?links.active=core#/devices/getNodeCustomFields
	$GetURL = "v2/device/$DeviceID/custom-fields"

	$ApiURL = Join-URLs @($Script:NinjaOneBaseURLAPI, $GetURL)
	If ($Dbg) { 
		Log 4 'ApiURL'
		Log 5 $ApiURL
	}
	
	$BodyProps = @{}
	$BodyProps.Add($FieldName_ssfClientTasksData, $FieldValue)
	# Return $BodyProps
	# Return (ConvertTo-Json -InputObject $BodyProps -Depth 100)

	# Die Fields Setzen
	$Res = Invoke-RestMethod -Uri $ApiURL -Method Patch `
									-Headers $AuthHeader `
									-ContentType 'application/json' `
									-Body (ConvertTo-Json -InputObject $BodyProps -Depth 100)

	Return $Res
}


#Region NinjaOne, Funcs für ssfClientTasksData


## Parser für den Inhalt von ssfClientTasksData

# Parst von *einem* Block die Zeilen und erzeugt ein PSCustomObject
Function Parse-Block-Lines() {
    Param (
        [String[]]$BlockLines,
		[Switch]$Dbg
    )

	# Sicherstellen, dass wir wirklich Zeilen haben
	$BlockLines = $BlockLines -split "`n" | ? { -Not [String]::IsNullOrWhiteSpace( $_ ) }

	If ($Dbg) {
		Log 4 '2jbdjK Parse-Block-Lines(), $BlockLines:' -Fore Magenta
		Log 5 ($BlockLines | Out-String)
	}

    ## Header-Zeile analysieren
    $headerLine = $BlockLines[0]
	If ($Dbg) {
		Log 4 'Parse-Block-Lines(), $headerLine:' -Fore Magenta
		Log 5 ($headerLine | Out-String)
	}

    $headerPattern = '^(?<Timestamp>\d{6}\s+\d{4,6})\s+(?<Key>\S+)\s*(?<KeyAddon>.*)$'

    If ($headerLine -notmatch $headerPattern) {
        Log 4 "Ungültige Headerzeile: $headerLine" -Fore Red
		Return $Null
    }

    $timestampRaw = $matches['Timestamp']
    $key          = $matches['Key']
    $KeyAddon      = $matches['KeyAddon']

    # Konvertierung in [datetime]
    $datetime = $null
    If ($timestampRaw -match '^(?<date>\d{6})\s+(?<time>\d{4,6})$') {
        $datePart = $matches['date']
        $timePart = $matches['time']

        $yy = [int]$datePart.Substring(0, 2)
        $mm = [int]$datePart.Substring(2, 2)
        $dd = [int]$datePart.Substring(4, 2)

        $hh = [int]$timePart.Substring(0, 2)
        $mi = [int]$timePart.Substring(2, 2)
        $ss = If ($timePart.Length -ge 6) { [int]$timePart.Substring(4, 2) } else { 0 }

        $year = If ($yy -lt 100) { 2000 + $yy } else { $yy }

        $datetime = Get-Date -Year $year -Month $mm -Day $dd -Hour $hh -Minute $mi -Second $ss
    }

    # Comment
	$Comment = ''
	# Vxxx
	$Version = $null
    # Key-Value Werte
	$KeyValueData = @{}
    $AdditionalLines = @()

    ForEach ($Line in $BlockLines[1..($BlockLines.Count - 1)]) {
		$Line = $Line.Trim()
        If ($Line -match '^V\d{3}$' -and -not $Version) {
            $Version = $Line
        }
        Elseif ($Line -match '^\s*(?<PKey>[^:]+)\s*:\s*(?<PValue>.+)$') {
            $keyName = $matches['PKey'].Trim()
            $value = $matches['PValue'].Trim()
            $KeyValueData[$keyName] = $value
        }
        Elseif ($Line -match '^#(?<Comment>.*)') {
			$Comment = $Matches['Comment'].Trim()
		}		
        Else {
            $AdditionalLines += $Line
        }
    }

    # Objekt erstellen und Root-Eigenschaften zusammenbauen
    $baseObject = [Ordered]@{
        Processed   = $False
        Key         = $key
        KeyAddon    = $KeyAddon
		Comment		= $Comment
        Version     = $Version
        Timestamp   = $datetime
    }

    # Properties hinzufügen
    ForEach ($kvp in $KeyValueData.GetEnumerator()) {
        $baseObject[$kvp.Key] = $kvp.Value
    }
	
	$baseObject['AdditionalLines'] = $AdditionalLines
	$baseObject['OriLines'] = $BlockLines

    return [PSCustomObject]$baseObject
}



# Parst alle Blöcke und erzeugt für jeden je ein PSCustomObject
# !Ex
# 	$Blöcke = Get-TextBlocks -Text BlockLines
# 	$oBlöcke = Parse-ClientTasksData -Blocks $Blöcke
Function Parse-ClientTasksData {
    Param (
        [String[]]$Text,
		[Switch]$Dbg
    )

	# String in Linien Splitten und leere Zeilen entfernen
	$Lines = $Text -split "`n" | ? { -Not [String]::IsNullOrWhiteSpace( $_ ) }
	
	If ($Dbg) {
		Log 4 'Parse-ClientTasksData(), $Lines:' -Fore Magenta
		Log 5 ($Lines | Out-String)
	}

	$Blöcke = @(Get-TextBlocks -Lines $Lines -Dbg:$Dbg)

	If ($Dbg) {
		Log 4 'Parse-ClientTasksData(), $Blöcke:' -Fore Magenta
		For($idx = 0; $idx -lt $Blöcke.Count; $idx++) {
			Log 5 "Block: $idx"
			Log 6 ($Blöcke[$idx] | Out-String)
		}
	}

	$Res = @()
    ForEach ($BlockLines in $Blöcke) {
		$Res += Parse-Block-Lines -BlockLines $BlockLines -Dbg:$Dbg
	}
	Return $Res
}



# Trennt die Zeilen in Blöcke, e.g.:
# 	$Text = @'
# 		250616 1246 LocalUserPWChanged lk flaksjf lkas df
# 		V002
# 		Prop1: x
# 		Prop2: y
# 		Prop3: y
# 		Testzeile 1
# 		Testzeile 2
#	'@
Function Get-TextBlocks {
    param (
        [String[]]$Lines,
		[Switch]$Dbg
    )
	
	## Init
	# Sicherstellen, dass wir wirklich Zeilen haben
	$Lines = $Lines -split "`n" | ? { -Not [String]::IsNullOrWhiteSpace( $_ ) }

	If ($Dbg) {
		Log 4 'Get-TextBlocks(), $Lines:' -Fore Magenta
		Log 5 ($Lines | Out-String)
	}

	## Config
    # Regex für den Start eines Blocks
    $startPattern = '^\d{6}\s+\d{4,6}\s+\S.+$'

	## Main
    $blocks = @()
    $currentBlock = @()

    foreach ($Line in $Lines -split "`n") {
        If ($Line.Trim() -eq '') {
            continue # Leerzeilen vollständig ignorieren
        }

        If ($Line -match $startPattern) {
            If ($currentBlock.Count -gt 0) {
                $blocks += ,@($currentBlock)
                $currentBlock = @()
            }
        }

        $currentBlock += $Line
    }

    # Letzten Block hinzufügen, falls vorhanden
    If ($currentBlock.Count -gt 0) {
        $blocks += ,@($currentBlock)
    }

	If ($Dbg) {
		Log 4 'Get-TextBlocks(), $blocks:' -Fore Magenta
		For($idx = 0; $idx -lt $blocks.Count; $idx++) {
			Log 5 "Block: $idx"
			Log 6 ($blocks[$idx] | Out-String)
		}
	}

    Return $blocks
}


# Konvertiert nicht verarbeitete Blöcke wieder zu Zeilen
# Um sie wieder ins ssfClientTasksData zu schreiben, wenn wir Fehler haben
# In -Comment kann der Fehler mitgegeben werden
Function Get-Block-unprocessed-AsLines {
    Param (
        [PSCustomObject[]]$Blocks,
        [String]$Comment
    )
	
	# Neue Liste erzeugen
	$Res = @()

	ForEach ($Block in $Blocks | ? Processed -eq $False) {
		$Idx = -1
		ForEach ($Line in $Block.OriLines) {
			$Idx++
			$Res += [String]$Line
			If ($Idx -eq 0) {
				$Res += ('# {0}' -f $Comment.TrimStart('#').Trim())
			}
		}
		# Leere Zeile als Trenner
		$Res += ''
	}
	Return $Res -join "`n"
}


# Trennt die Zeilen in Blöcke, e.g.:
# 	$Text = @'
# 		250616 1246 LocalUserPWChanged lk flaksjf lkas df
# 		V002
# 		Prop1: x
# 		Prop2: y
# 		Prop3: y
# 		Testzeile 1
# 		Testzeile 2
#	'@
Function Get-TextBlocks {
    param (
        [String[]]$Lines,
		[Switch]$Dbg
    )
	
	## Init
	# Sicherstellen, dass wir wirklich Zeilen haben
	$Lines = $Lines -split "`n" | ? { -Not [String]::IsNullOrWhiteSpace( $_ ) }

	If ($Dbg) {
		Log 4 'Get-TextBlocks(), $Lines:' -Fore Magenta
		Log 5 ($Lines | Out-String)
	}

	## Config
    # Regex für den Start eines Blocks
    $startPattern = '^\d{6}\s+\d{4,6}\s+\S.+$'

	## Main
    $blocks = @()
    $currentBlock = @()

    foreach ($Line in $Lines -split "`n") {
        If ($Line.Trim() -eq '') {
            continue # Leerzeilen vollständig ignorieren
        }

        If ($Line -match $startPattern) {
            If ($currentBlock.Count -gt 0) {
                $blocks += ,@($currentBlock)
                $currentBlock = @()
            }
        }

        $currentBlock += $Line
    }

    # Letzten Block hinzufügen, falls vorhanden
    If ($currentBlock.Count -gt 0) {
        $blocks += ,@($currentBlock)
    }

	If ($Dbg) {
		Log 4 'Get-TextBlocks(), $blocks:' -Fore Magenta
		For($idx = 0; $idx -lt $blocks.Count; $idx++) {
			Log 5 "Block: $idx"
			Log 6 ($blocks[$idx] | Out-String)
		}
	}

    Return $blocks
}


# Konvertiert nicht verarbeitete Blöcke wieder zu Zeilen
# Um sie wieder ins ssfClientTasksData zu schreiben, wenn wir Fehler haben
# In -Comment kann der Fehler mitgegeben werden
Function Get-Block-unprocessed-AsLines {
    Param (
        [PSCustomObject[]]$Blocks,
        [String]$Comment
    )
	
	# Neue Liste erzeugen
	$Res = @()

	ForEach ($Block in $Blocks | ? Processed -eq $False) {
		$Idx = -1
		ForEach ($Line in $Block.OriLines) {
			$Idx++
			$Res += [String]$Line
			If ($Idx -eq 0) {
				$Res += ('# {0}' -f $Comment.TrimStart('#').Trim())
			}
		}
		# Leere Zeile als Trenner
		$Res += ''
	}
	Return $Res -join "`n"
}

#Endregion NinjaOne, Funcs für ssfClientTasksData


#Region NinjaOne, Funcs für das Custom Field: ssfClientTasksData


## Prozessor für die einzelnen ssfClientTasksData Elemente

# Speichert den FileVault Recovery Key
# Block-Struktur:
# 	250616 094302 SaveFileVaultRecoveryKey
#	V001
#	RecoveryKey: x
# 	
Function Save-FileVaultRecoveryKey() {
	Param (
        [PSCustomObject]$oBlock,
		[Object]$AuthHeader,
		[String]$DeviceID,
		[Switch]$Dbg
    )
	
	Switch($oBlock.Version) {
		'V001' {
			If ($Dbg) { 
				Log 4 'Save-FileVaultRecoveryKey(): V001' -Fore Magenta
				Log 5 ('FieldName : {0}' -f 'ssfMacOsFileVaultKey')
				Log 5 ('FieldValue: {0}' -f $oBlock.RecoveryKey)
			}
			# Zeile ergänzen
			
			$RecoveryData = ("{0}:`n{1}" -f (Get-Date).ToString('yyMMdd HHmm'), $oBlock.RecoveryKey)
			
			& $SetNinjaOneDeviceFieldSecure_ps1 -DeviceIDs $DeviceID `
			-FieldName 'ssfMacOsFileVaultKey' `
			-FieldValue $RecoveryData `
			-Action 'AddNewLine' `
			-AuthHeader $AuthHeader
		}
		
		Default {
			Log 4 '2jbb5z Fehler in: Save-FileVaultRecoveryKey()' -NewLineBefore -Fore Red 
			Log 5 ('Version unbekannt: {0}' -f $oBlock.Version)
		}
	}
	
}


# Verarbeitet alle Elemente im Feld ssfClientTasksData
Function Transfer-ssfClientTasksData() {
	Param (
        [PSCustomObject[]]$oBlöcke,
		[Object]$AuthHeader,
		[String]$DeviceID,
		[Switch]$Dbg
    )

	ForEach($oBlock in $oBlöcke) {
		If ($Dbg) {
			Log 4 'Block:' -Fore Magenta
			Log 5 $oBlock
		}
		Switch ($oBlock.Key) {
			# Den FileFault RecoveryKey
			'SaveFileVaultRecoveryKey' {
				If ($Dbg) {
					Log 4 'Verarbeite: SaveFileVaultRecoveryKey' -Fore Magenta
				}
				Log 2 'Speichere den FileVault Recovery Key'
				Save-FileVaultRecoveryKey -oBlock $oBlock `
										  -AuthHeader $AuthHeader `
										  -DeviceID $DeviceID `
										  -Dbg:$Dbg
				$oBlock.Processed = $True				
			}
			
			Default {
				If ($Dbg) {
					Log 4 "Switch-Fehler!: $($oBlock.Key)" -Fore Red
				}
				Log 4 '2jbb5y Fehler in: Transfer-ssfClientTasksData()' -NewLineBefore -Fore Red
				Log 5 ('Unbekannter Key: {0}' -f $oBlock.Key)
			}
		}
	}
}

#Endregion NinjaOne, Funcs für das Custom Field: ssfClientTasksData


#Region NinjaOne, Funcs für das Custom Field: SaveFileVaultRecoveryKey


# Liefert den Record für das Custom Field:
#
# 250616 1707 SaveFileVaultRecoveryKey
# V001
# RecoveryKey: x1
Function New-ssfClientTasksData-Record-SaveFileVaultRecoveryKey() {
	Param (
		[Parameter(Mandatory)]
		[String]$FileVaultRecoveryKey
	)
	
	# Config
	$RecKey = 'SaveFileVaultRecoveryKey'

	$Rec = @"
$((Get-Date).ToString('yyMMdd HHmm')) $RecKey
V001
RecoveryKey: $FileVaultRecoveryKey
"@	
	
	Return $Rec
}


# Erzeugt im NinjaOne Custom Field ssfClientTasksData einen Record
Function Create-ssfClientTasksData-Rec() {
	Param(
		[Parameter(Mandatory)]
		[String]$FileVaultRecoveryKey
	)

	# Den Record erzeugen
	$RecData = New-ssfClientTasksData-Record-SaveFileVaultRecoveryKey `
				-FileVaultRecoveryKey $FileVaultRecoveryKey
	
	# Dbg
	# Log 4 "`nRecData:"
	# Log 5 ($RecData | Out-String)
	
	# Den aktuellen Wert lesen
	# !9 Liefert ein String[]
	$WertOri = & "$Script:NinjaCliPath" get "$Script:FieldName_ssfClientTasksData"
	$WertOri = $WertOri -Join "`n"
	
	
	# Den Rec ergänzen
	$WertNeu = ("{0}`n`n{1}" -f $WertOri, $RecData)
	# Dbg
	# Log 4 "`nWertNeu:"
	# Log 5 ($WertNeu | Out-String)
	

	# Pfad zum NinjaRMM-Agenten
	& "$Script:NinjaCliPath" set "$Script:FieldName_ssfClientTasksData" "$WertNeu"
	
}


# Prüft, dass das NinjaOne Custom Field den FileVaultRecoveryKey hat
# Liefert $True, wenn der Key vorhanden ist
Function Assert-ssfClientTasksData-has-FileVaultRecoveryKey() {
	Param(
		[Parameter(Mandatory)]
		# eg EDK4-JR4M-LRW2-GHQQ-47PL-39EQ
		[String]$FileVaultRecoveryKey,
		[Int]$RetryMs = 8000,
		[Int]$TimeoutSec = 120
	)

	Log 2 'Prüfe, ob NinjaOne den FileVault Recovery Key hat' -NewLineBefore

	$StartTime = Get-Date
	$FileVaultKeyFound = $False
	Do {
		
		# Den aktuellen Wert lesen
		# !9 Liefert ein String[]
		$WertOri = & "$Script:NinjaCliPath" get "$Script:FieldName_ssfClientTasksData"
		$WertOri = $WertOri -Join "`n"
		
		# Wenn wir einen Wert erhalten haben
		If (-not [string]::IsNullOrWhiteSpace($WertOri)) {
			$FileVaultKeyFound = $WertOri.IndexOf($FileVaultRecoveryKey, [StringComparison]::OrdinalIgnoreCase) -ge 0
		}

		# Wenn der VaultKey fehlt, warten
		If ($FileVaultKeyFound) {
			Log 3 'OK, gefunden' -Fore Green
		} Else {
			Log 3 ('Noch nicht gefunden, warte {0}ms' -f $RetryMs) -Fore Magenta
			Start-Sleep -Milliseconds $RetryMs
		}
		
		$Elapsed = (Get-Date) - $StartTime
		
	} While (-not $FileVaultKeyFound -and $Elapsed.TotalSeconds -lt $TimeoutSec)
	
	Return $FileVaultKeyFound
}

#Endregion NinjaOne, Funcs für das Custom Field: SaveFileVaultRecoveryKey

#Endregion NinjaOne Funcs


