#!/usr/local/bin/pwsh
# !9^9 Unix (LF), nur UTF8 für MacOS, KEIN BOM

#Requires -Version 7



<#
	.SYNOPSIS
		Liest die verschiedenen Hostnamen vom MacOS und zeigt sie an

	.PARAMETER GetDetails
		Zeigt die Details von dsconfigad -show an

	.PARAMETER Print
		Gibt die Hostnamen-Infos auf der Konsole aus,
		sonst wird das Resultat nur in der Pipeline zurückgegeben

	.EXAMPLE
		! .\Get-Hostname-2.ps1
		Gibt in der Pipeline die Hostnamen-Infos zurück

	.EXAMPLE
		! .\Get-Hostname-2.ps1 -Print
		Gibt in der Pipeline die Hostnamen-Infos zurück und gibt sie auf der Konsole aus

	.NOTES
		001, 250610, Tom
		002, 250611, Tom
		  Shebang angepasst
		003, 250627, Tom
			Liefert infos zum Domain-join
#>
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidUsingCmdletAliases', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseApprovedVerbs', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidDefaultValueSwitchParameter', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidDefaultValueForMandatoryParameter', '')]
Param (
	# Liefert Infos von dsconfigad -show
	[Switch]$GetDetails,
	[Switch]$Print,

	# Get-Help Parameter
	# 005, 200314
	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Switch]$GetHelp = $false,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Switch]$GetHelpCls = $false,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[int]$GetEx,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[int]$RunEx
)

# Write-Host $PsCmdlet.ParameterSetName





### Init
$ScriptDir = [IO.Path]::GetDirectoryName($MyInvocation.MyCommand.Path)

#Region Toms Tools: Get-Help
# 006, 210202

$Script:GetHelpColors = @{
		ScriptName = @{
			ForegroundColor = 'Green'
			BackgroundColor = 'Black'
		}
		ExampleColor = 'Cyan'
		CommandColor = 'Yellow'
		CommentColor = 'White'
}

# Liefert $True, wenn $TestString vermutlich ein PowerShell Script / Kommando ist,
# sonst ist es vermutlich ein normaler Text
Function Is-PowerShell-Command($TestString) {
	$Token = $Null
	$ParseError = $Null

	$Res = [System.Management.Automation.Language.Parser]::ParseInput($TestString, [ref]$Token, [ref]$ParseError)

	# Wenn wir anderes als nur Tokens haben, haben wir vermutlich ein Script
	If (($Token | ? { ($_.GetType()).Name -ne 'Token' }).Count -gt 0) {
		# Vermutlich ein Script
		$True
	} Else {
		# Vermutlich Text
		$False
	}
}


# Analysier das Code-Beispiel
# Syntax, e.g.
# 	i3lines ! .\Get-My-ActivityStream.ps1 -GetCurrentMonth -JiraTicketsInteractive -npid xxxx
#
# i3lines	» Das Code-Beispiel hat 3 Zeilen
# !			» Das Code-Beispiel ist direkt ausführbar
Function Analyze-CodeExample($Code, $ScriptFileName, $ScriptFullName) {

	# Syntax:
	# i<n>Lines ! .\…
	$Rgx = '(?<NoOfLinesInfo>i(?<NoOfLines>\d+)lines)|(?<IsExecutable>\!\s+)'

	# 250708, MacOS
	$NoOfLines = 1
	$MyCode = $Code
	$IsDirectExecutable = $false
	$MyMatches = $MyCode | Select-String $Rgx -AllMatches
	While ($MyMatches) {
		$MyMatches | % {
			# Haben wir das Element gefunden, das die Anzahl der Zeilen angibt?
			$NoOfLinesInfo = $_.Matches[0].Groups['NoOfLinesInfo']
			If ($NoOfLinesInfo.Success) {
				# Vom Code-Beispiel den gefundenen Parameter entfernen
				$MyCode = $MyCode.Remove($NoOfLinesInfo.Index, $NoOfLinesInfo.Length).Trim()
				$NoOfLines = $_.Matches[0].Groups['NoOfLines'].Value
				# Pipeline stoppen: Sicherstellen, dass wir neu den Regex ausführen
				Return
			}

			# Haben wir das Element gefunden, das angibt, dass das Beispiel direkt ausführbar ist?
			$IsExecutable = $_.Matches[0].Groups['IsExecutable']
			If ($IsExecutable.Success) {
				# Vom Code-Beispiel den gefundenen Parameter entfernen
				$MyCode = $MyCode.Remove($IsExecutable.Index, $IsExecutable.Length).Trim()
				$IsDirectExecutable = $true
				# Pipeline stoppen: Sicherstellen, dass wir neu den Regex ausführen
				Return
			}

		}
		# Den Rest wieder Parsen
		$MyMatches = $MyCode | Select-String $Rgx -AllMatches
	}

  	# 250708 Code-Zeilen
	$MyCodeLines = $MyCode -Split "`n"
	# Von den Code-Zeilen nur die ersten $NoOfLines Zeilen behalten
	$OnlyCodeLines = ($MyCodeLines[0..($NoOfLines - 1)]) -join "`n"
	$CodeCommentLines = ($MyCodeLines[$NoOfLines..($MyCodeLines.Count - 1)]) -join "`n"

	[PSCustomObject][Ordered]@{
		FullCommandLineOri = $OnlyCodeLines
		# Den Script-Namen mit dem vollen Pfad ersetzen
		FullCommandLine	 = ($OnlyCodeLines.Replace(".\$ScriptFileName", $ScriptFullName))
		# Nur die Kommandozeilen-Parameter
		Parameters		    = ($OnlyCodeLines.Replace(".\$ScriptFileName", '')).Trim()
		IsDirectExecutable = $IsDirectExecutable
		NoOfLines			= $NoOfLines
		CodeCommentLines  		= $CodeCommentLines
	}
}


# Zeigt eine nützliche Formatierung von Get-Help -Examples
Function Show-Help($ScriptFullName, $HelpText, $RecognizeCode = $false) {
	$ScriptFileName = [IO.Path]::GetFileName($ScriptFullName)

	# Die Synopsis (den Header) anzeigen
	Write-Host $HelpText.Synopsis -ForegroundColor Green
	Write-Host ''

	$ExNo = 1
	ForEach ($Example In $HelpText.Examples.Example) {
		# Jedes Beispiel anzeigen
		# Den Titel
		Write-Host ("Example #{0}" -f ($ExNo++)) -ForegroundColor Cyan

		## Das Code-Beispiel analysieren
		$Code = $Example.Code

		$CodeInfo = Analyze-CodeExample $Code $ScriptFileName $ScriptFullName
		If ($CodeInfo.IsDirectExecutable) {
			Write-Host "! $($CodeInfo.FullCommandLine)" -ForegroundColor Yellow
		} Else {
			Write-Host $CodeInfo.FullCommandLine -ForegroundColor Yellow
		}

		### Die Kpommentare anzeigen
		## Die Kommentare, die der PS-Parser selber fand, scheint in PS5 und PS7 unterschieldich zu sein
		# Die Leeren Zeilen am Ende löschen
		$Comments = $Example.Remarks.Text -replace "(?s)`n\s*$"
		# Die ersten Zeilen im Kommentar allenfalls als Code anzeigen
		$CommentStarted = $False
		$CodeLines = $CodeInfo.NoOfLines -1 # Die erste Zeile wurde bereits angezeigt
		ForEach ($Comment In $Comments) {
			# Die Comments in Zeilen aufteilen und leere Zeilen ignorieren
			$Lines = ($Comment -split "`r`n|`r|`n") | ? { -not [String]::IsNullOrEmpty($_) }
			ForEach ($Line In $Lines) {
				# Wir haben noch eine Code-Zeile
				If (($CodeLines--) -gt 0) {
					# Der Kommentar ist vermutlich ein PowerShell Befehl
					Write-Host $Line.Replace(".\$ScriptFileName", $ScriptFullName) -ForegroundColor Yellow
				} Else {
					# Der Kommentar ist vermutlich Text
					If ($CommentStarted -eq $False) {
						$CommentStarted = $True
						Write-Host ''
					}
					Write-Host $Line
				}
			}
		}


		## Die Kommentare ausgeben, die mein eigener Parser fand
		$CommentStarted = $False
		# Leere Zeilen ignorieren
		ForEach ($Line In $CodeInfo.CodeCommentLines | ? { -not [String]::IsNullOrEmpty($_) }) {
			If ($CommentStarted -eq $False) {
				$CommentStarted = $True
				Write-Host ''
			}
			Write-Host $Line
		}
		Write-Host ''
	}
}


# Schreibt eines der Examples ins ClipBoard
# oder liefert @($ExampleIsExecutable, $ScriptFullName, $Parameters)
Function Get-Ex($GetEx, $ScriptFullName, $HelpText, [Switch]$ToClipboard = $True) {
	# 250708, MacOS
	# $HelpText ist PSCustomObject
	# Return $HelpText

	$ScriptFileName = [IO.path]::GetFileName($ScriptFullName)
	# $Example ist PSCustomObject
	$Example = $HelpText.Examples.Example[($GetEx - 1)]
	# Return $Example
	$Comments = $Example.Remarks.Text -replace "(?s)`n\s*$"
	# Return $Comments

	## Das Code-Beispiel analysieren
	$Code = $Example.Code
	# Return $Code
	$CodeInfo = Analyze-CodeExample $Code $ScriptFileName $ScriptFullName

	# Haben wir mehrere Zeilen im Code-Beispiel?
	$CodeExample = @()
	$CodeExample += $CodeInfo.FullCommandLine
	If ($CodeInfo.NoOfLines -gt 1) {
		$CodeLines = $CodeInfo.NoOfLines - 1 # Die erste Zeile wurde bereits angezeigt
		ForEach ($Comment In $Comments) {
			# Die Comments in Zeilen aufteilen und leere Zeilen ignorieren
			$Lines = ($Comment -split "`r`n|`r|`n") | ? { -not [String]::IsNullOrEmpty($_) }
			ForEach ($Line In $Lines) {
				# Wir haben noch eine Code-Zeile
				If (($CodeLines--) -gt 0) {
					$CodeExample += $Line
				}
			}
		}
	}

	If ($ToClipboard) {
		Set-Clipboard -Value ($CodeExample -join "`n")
	} Else {
		Return @($CodeInfo.IsDirectExecutable, $CodeExample)
	}
}


If ($GetHelp -or $GetHelpCls) {
	If ($GetHelpCls) { CLS }
	Show-Help $MyInvocation.InvocationName (Get-Help -Examples $MyInvocation.InvocationName)
	Break Script
}

# Kopiert eines der Examples ins Clipboard
If ($GetEx) {
	$ResGetEx = Get-Ex $GetEx $MyInvocation.InvocationName (Get-Help -Examples $MyInvocation.InvocationName)
	# Dbg
	# Return $ResGetEx
	Break Script
}

# Führt eines der Examples direkt aus
If ($RunEx) {
	$ExampleIsExecutable, $CodeExample = Get-Ex $RunEx $MyInvocation.InvocationName (Get-Help -Examples $MyInvocation.InvocationName) -ToClipboard:$False
	If ($ExampleIsExecutable) {
		# Bei mehr als zwei Zeilen: Notepad öffnen und nur die erste Zeile ausführen
		Invoke-Expression -Command $CodeExample[0]
	} Else {
		Write-Host "Example $RunEx kann nicht direkt ausgeführt werden - es wurde in die Zwischenablage kopiert"
		Set-Clipboard -Value ("{0} {1}" -f $ScriptFullName, $Parameters)
	}
	Break Script
}

#Endregion Toms Tools: Get-Help




### Config
$TomLibMacOS_ps1 = 'TomLib-MacOS.ps1'



### Funcs


### Prepare

# Lib laden
$TomLibMacOS_ps1 = Join-Path $ScriptDir $TomLibMacOS_ps1
. $TomLibMacOS_ps1 $PSBoundParameters


# Hostnamen bestimmen
$oHostnamesInfo = Get-HostNames-Info
# Return $oHostnamesInfo



### Main

If ($Print) {
	Print-HostNamesInfo -oHostnamesInfo $oHostnamesInfo
}

If ($GetDetails) {
	Log 0 'dsconfigad -show'
	$dsconfigadOut = & dsconfigad -show 2>$null
	$dsconfigadOut.Split("?n") | % {
		Log 1 $_ -Fore White
	}
}

Return $oHostnamesInfo


