#!/usr/local/bin/pwsh
# !9^9 Nur UTF8 für MacOS, KEIN BOM, Unix (LF)

#Requires -Version 7

## Suppress PSScriptAnalyzer Warning
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidUsingCmdletAliases', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseApprovedVerbs', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidDefaultValueSwitchParameter', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidDefaultValueForMandatoryParameter', '')]
# [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSShouldProcess', '')]
# [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseSingularNouns', '')]


# Verbindet das MacBook von einer Domäne
# Fragt nach Usernamen und PW, wenn nicht angegeben
#
# Returns:
#	$True	Domain-Join i.O.
#	$False	Domain-Join nicht i.O.

# !Ex
# /Library/Scripts/SSF/NinjaOne/Join-Domain.ps1 -ZielDomain 'akros.ch' -UserName 'ad@akros.ch' -PW 'xxx'


# 001, 250701, Tom

[CmdletBinding(SupportsShouldProcess, ConfirmImpact = 'Low')]
Param (
    [Parameter(Mandatory, Position = 0)]
	[String]$ZielDomain,
	# Optional ein UserName
	[Object]$UserName,
	# Optional ein PW
	[Object]$PW
)



### Init
$ScriptDir = [IO.Path]::GetDirectoryName($MyInvocation.MyCommand.Path)
$ThisScriptPath = $MyInvocation.MyCommand.Definition

$ZielDomain = $ZielDomain.Trim()


### Config
$TomLibMacOS_ps1 = 'TomLib-MacOS.ps1'


### Root erzwingen
# 250620 1850
If ((id -u) -ne 0) {
    Write-Host 'Starte das Script als root'

    # Argumente aus den BoundParameters zusammensetzen
    $paramList = @()
    ForEach ($kvp in $PSBoundParameters.GetEnumerator()) {
        If ($kvp.Value -is [switch] -and $kvp.Value.IsPresent) {
            $paramList += "-$($kvp.Key)"
        } Else {
            $paramList += "-$($kvp.Key) `"$($kvp.Value)`""
        }
    }

    $pwsh = (Get-Command pwsh).Source

    If ($paramList.Count -gt 0) {
        # sudo $pwsh $ThisScriptPath @paramList
		$quotedArgs = $paramList -join ' '
		$cmd = "$pwsh `"$ThisScriptPath`" $quotedArgs"
		sudo bash -c "$cmd"
    } Else {
        sudo $pwsh $ThisScriptPath
    }
    Exit
}

# Ab hier läuft das Script als root
Write-Host "`nScript läuft jetzt als root`n`n" -Fore DarkGray


### Funcs



### Prepare

# Lib laden
$TomLibMacOS_ps1 = Join-Path $ScriptDir $TomLibMacOS_ps1
. $TomLibMacOS_ps1 $PSBoundParameters

# Aktuelle Hostnamen abrufen
$oHostnamesInfo = Get-HostNames-Info
# Print-HostNamesInfo -oHostnamesInfo $oHostnamesInfo



### Main
If ($IsVerbose) { Log 0 'Join-Domain' }


# Sind wir schon Domain-Joined?
If ($oHostnamesInfo.IsDomainJoined) {
	If ($IsVerbose) { Log 1 'MacOS ist Domain-Joined' }

	If ($oHostnamesInfo.JoinedDomain -eq $ZielDomain) {
		If ($IsVerbose) { Log 2 "Ist bereits in der Domäne: $ZielDomain"  -Fore Green }
		Return [PSCustomObject]@{
			Resultat = $True
			Message  = "Ist bereits in der Domäne: $ZielDomain"
		}
		
	} Else {
		If ($IsVerbose) { Log 2 "Ist in einer anderen Domäne: $($oHostnamesInfo.JoinedDomain)" -Fore Red } 
		Return [PSCustomObject]@{
			Resultat = $False
			Message  = "Ist in einer anderen Domäne: $($oHostnamesInfo.JoinedDomain)"
		}
	}
}


## Das Notebook ist nicht Domain-Joined

## Kann das Notebook die Domäne anpingen?

$IsComputerInDomainLAN = Is-Computer-in-Domain-LAN -Domain $ZielDomain

If (-not $IsComputerInDomainLAN) {
	Return [PSCustomObject]@{
		Resultat = $False
		Message  = "Notebook kann die Domäne nicht anpingen: $ZielDomain"
	}
}


## Hat der MacOS HostName die Domäne schon?
# Oder ist sie falsch?
If (Is-Empty $oHostnamesInfo.FQDN_Domain `
	-or $oHostnamesInfo.JoinedDomain -ne $ZielDomain) {
	# Den Hostnamen anpassen
	Log 1 "Ergänze den Hostnamen um die Domäne: $ZielDomain"
	# Den Hostnamen um die Domäne ergänzen
	Set-HostNames -NewHostName $Null `
					-DomainName $ZielDomain `
					-Force:$Force
}


## Join Domain
$JoinOK = Join-MacOS-to-Domain -DomainName $ZielDomain -UserName $UserName -PW $PW
Return $JoinOK
