#!/usr/local/bin/pwsh
# !9^9 Nur UTF8 für MacOS, KEIN BOM

# !9 Exakt wie:
# 	Set-NinjaOne-Device-Field.ps1
# jedoch müssen dem Script die REST API Hashs 
# als Parameter übergeben werden,
# damit ein Hacker dieses Script nicht missbrauchen kann
# 
# 
# Nützt das NinjaOne REST API
# 	Sucht bei NinjaOne das Objekt mit dem -ComputerName
# 	Setzt im NinjaOne Computer-Objekt das Property -FieldName 
# 	auf den Wert -FieldValue
# 
# Unterstützte Aktionen
# -Action Set
# 	Wert = Wert aus -FieldValue
# -Action Add
# 	Wert = Alter Wert + Wert aus -FieldValue
# -Action AddNewLine
# 	Wert = Alter Wert + \n + Wert aus -FieldValue
# 

# !Q
# https://ninjarmm.zendesk.com/hc/en-us/community/posts/30319558075405-Script-Share-Retrieving-and-Updating-values-in-Custom-Fields-via-API-Windows-PowerShell


# !Ex
# c:\Scripts\PowerShell\NinjaOne\Get-NinjaOne-Devices-Fields.ps1 -ComputerNames anb038
# 


# 001, 250415, Tom
# 002, 250612, Tom
# 	Neu:
#		-Action
# 003, 250612, Tom
# 	Neuer Name
#		Set-NinjaOne-Device-Field-Secure.ps1
# 004, 250616, Tom
#   Neu:
#	-CoputerID
#	-AuthHeader 

[CmdletBinding(SupportsShouldProcess, DefaultParameterSetName = 'ComputerName')]
Param(
	[Parameter(Mandatory, Position=0, ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName='ComputerName_WithAuthHeader')]
	[Parameter(Mandatory, Position=0, ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName='ComputerName_WoAuthHeader')]
	# Der Computername, in dessen NinjaOne-Objekt ein Feld verändert werden soll
	[String[]]$ComputerNames = @($Env:ComputerName),

	[Parameter(Mandatory, Position=0, ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName='DeviceID_WithAuthHeader')]
	[Parameter(Mandatory, Position=0, ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName='DeviceID_WoAuthHeader')]
	# Die NinjaOne DeviceIDs
	[String[]]$DeviceIDs,
	
	[Parameter(Mandatory, Position=1, ValueFromPipeline, ValueFromPipelineByPropertyName)]
	# Der Feldname, der im NinjaOne Coputer-Objekt gesetzt werden soll
	[String]$FieldName = 'ssfmahandynr',
	
	[Parameter(Mandatory, Position=2, ValueFromPipeline, ValueFromPipelineByPropertyName)]
	# Der Feldwert, der gesetzt wird
	[String]$FieldValue,

	[ValidateSet(IgnoreCase, 'Set', 'Add', 'AddNewLine')]
	# Set				Ersetzt den Wert im Feld
	# Add				Ergänzt den Wert im Feld (String-Concat)
	# AddNewLine	Ergänzt den Wert im Feld mit einem NewLine
	[String]$Action = 'Set',

	[Parameter(Mandatory, ParameterSetName='ComputerName_WoAuthHeader')]
	[Parameter(Mandatory, ParameterSetName='DeviceID_WoAuthHeader')]
	# zB 'Gig.......................M'
	[String]$NinjaOneClientID,
	
	[Parameter(Mandatory, ParameterSetName='ComputerName_WoAuthHeader')]
	[Parameter(Mandatory, ParameterSetName='DeviceID_WoAuthHeader')]
	# zB 'aB..................................................1w'	
	[String]$NinjaOneClientSecret,
	
	[Parameter(Mandatory, ParameterSetName='ComputerName_WithAuthHeader')]
	[Parameter(Mandatory, ParameterSetName='DeviceID_WithAuthHeader')]
	[Object]$AuthHeader,
	
	# Debug?
	[Switch]$Dbg
)

# Write-Host $PsCmdlet.ParameterSetName



### Init
$ScriptDir = [IO.Path]::GetDirectoryName($MyInvocation.MyCommand.Path)



### Config
$NinjaOneBaseURL = 'https://eu.ninjarmm.com/'
$NinjaOneBaseURLAPI  = "$($NinjaOneBaseURL.TrimEnd('/'))/api/"



### Funcs



### Prepare

# Authentifizieren?
If ( $PsCmdlet.ParameterSetName.EndsWith('WoAuthHeader')) {
	$AuthHeader = Auth-NinjaOne -Dbg:$Dbg
}
# Return $AuthHeader



### Main

If ( $PsCmdlet.ParameterSetName.StartsWith('ComputerName')) {
	# Wir arbeiten mit Computernamen
	If ($Dbg) { Write-Host 'Mode: ComputerName' -Fore Gray }
	
	ForEach($ComputerName in $ComputerNames) {
		
		# Das NinjaOne Gerät suchen
		$ThisDevice = Get-NinjaOne-Device-by-SystemName -AuthHeader $AuthHeader -SystemName $ComputerName -Dbg:$Dbg
		
		If ($ThisDevice) {
			# Vom NinjaOne Gerät ale Fields abrufen
			$Fields = Get-NinjaOne-Device-Fields -AuthHeader $AuthHeader -DeviceID $ThisDevice.ID -Dbg:$Dbg
			
			# Alle gefundenen Felder ausgeben?
			If ($Dbg) {
				Write-Host ("`n{0}: {1}" -f $ThisDevice.ID, $ThisDevice.SystemName) -Fore Green
				($Fields | FL | Out-String) -split "`n" | `
					? { -Not [String]::IsNullOrWhiteSpace($_) } | `
					% { Write-Host "  $_" }
			}

			# Haben wir einen Wert?
			$WertOri = $Fields.$FieldName
			$WertNeu = ''
			Switch ($Action) {
				'Set' {
					$WertNeu = $FieldValue
				}
				'Add' {
					$WertNeu = $WertOri + $FieldValue
				}
				'AddNewLine' {
					$WertNeu = ("{0}`n{1}" -f $WertOri, $FieldValue)
				}
				Default {
					Write-Error "Ungültige Action: $Action"
				}
			}

				
			# Das Feld setzen
			Set-NinjaOne-Device-Field -AuthHeader $AuthHeader -DeviceID $ThisDevice.ID `
									-FieldName $FieldName -FieldValue $WertNeu -Dbg:$Dbg
			
		} Else {
			Write-Host "Nicht gefunden: $ComputerName" -Fore Red
		}
	}
}


If ( $PsCmdlet.ParameterSetName.StartsWith('DeviceID')) {
	# Wir arbeiten mit DeviceIDs
	If ($Dbg) { Write-Host 'Mode: DeviceIDs' -Fore Gray }
	
	ForEach($DeviceID in $DeviceIDs) {
		If ($Dbg) { Write-Host ('DeviceID: {0}' -f $DeviceID) -Fore Green }

		# Vom NinjaOne Gerät ale Fields abrufen
		$Fields = Get-NinjaOne-Device-Fields -AuthHeader $AuthHeader -DeviceID $DeviceID -Dbg:$Dbg
		
		# Alle gefundenen Felder ausgeben?
		If ($Dbg) {
			($Fields | FL | Out-String) -split "`n" | `
				? { -Not [String]::IsNullOrWhiteSpace($_) } | `
				% { Write-Host "  $_" }
		}

		# Haben wir einen Wert?
		$WertOri = $Fields.$FieldName
		$WertNeu = ''
		Switch ($Action) {
			'Set' {
				$WertNeu = $FieldValue
			}
			'Add' {
				$WertNeu = $WertOri + $FieldValue
			}
			'AddNewLine' {
				$WertNeu = ("{0}`n{1}" -f $WertOri, $FieldValue)
			}
			Default {
				Write-Error "Ungültige Action: $Action"
			}
		}

		If ($Dbg) { 
			Write-Host ('Wert aktuell: {0}' -f $WertOri)
			Write-Host ('Wert neu    : {0}' -f $WertNeu)
		}

		# Das Feld setzen
		Set-NinjaOne-Device-Field -AuthHeader $AuthHeader `
								-DeviceID $DeviceID `
								-FieldName $FieldName `
								-FieldValue $WertNeu `
								-Dbg:$Dbg
		
	}
}


