#!/usr/local/bin/pwsh

# !9^9 Nur UTF8 für MacOS, KEIN BOM

# Setzt für den lokalen MacOS User ein neues PW
# Und aktualisiert bei NinjaOne die Custom Fields

# 001, 250611, Tom

[CmdletBinding(SupportsShouldProcess, ConfirmImpact='High')]
Param(
	[Parameter(Mandatory)]
	[string]$TargetUserName,

	[Parameter(Mandatory)]
	[string]$NewPassword,

	[Parameter(Mandatory)]
	[string]$AdminUserName,

	[Parameter(Mandatory)]
	[string]$AdminPassword
)



### Config

$dsclPath = "/usr/bin/dscl"
$sysadminctlPath = "/usr/sbin/sysadminctl"



### Funcs




### Prepare


# Prüfen, ob der Befehl mit root-Rechten ausgeführt wird
if ($env:SUDO_USER -eq $null) {
	Write-Error "Diese Funktion muss mit Administratorrechten (sudo pwsh) ausgeführt werden."
	return
}

if (-not (Test-Path $sysadminctlPath)) {
	Write-Error "Der 'sysadminctl'-Befehl wurde unter '$sysadminctlPath' nicht gefunden."
	return
}
	
	
# Prüfen, ob der dscl-Befehl existiert
if (-not (Test-Path $dsclPath)) {
	Write-Error "Der 'dscl'-Befehl wurde unter '$dsclPath' nicht gefunden."
	return
}

# Überprüfen, ob der Benutzer existiert (optional, aber empfohlen)
Write-Verbose "Überprüfe Existenz des Benutzers '$UserName'..."
try {
	$userCheck = & "$dsclPath" . -read "/Users/$UserName" 2>&1
	if ($LASTEXITCODE -ne 0 -or $userCheck -match "No such key: Users") {
		Write-Error "Benutzer '$UserName' nicht gefunden. Bitte überprüfen Sie den Benutzernamen."
		return
	}
} catch {
	Write-Error "Fehler beim Überprüfen der Benutzer-Existenz: $($_.Exception.Message)"
	return
}



### Main

if ($PSCmdlet.ShouldProcess("Benutzer '$TargetUserName'", "Passwort setzen")) {
	Write-Verbose "Versuche Passwort für Benutzer '$TargetUserName' mit Admin-Konto '$AdminUserName' zu setzen..."
	try {
		# Argumentliste für sysadminctl
		$arguments = @(
			"-resetPasswordFor", $TargetUserName,
			"-newPassword", "-", # - als Platzhalter für stdin
			"-adminUser", $AdminUserName,
			"-adminPassword", "-" # - als Platzhalter für stdin
		)

		Write-Verbose "Starte Prozess: '$sysadminctlPath' mit Argumenten: '$($arguments -join ' ')'"

		$processInfo = New-Object System.Diagnostics.ProcessStartInfo
		$processInfo.FileName = $sysadminctlPath
		$processInfo.Arguments = $arguments -join ' '
		$processInfo.UseShellExecute = $false
		$processInfo.RedirectStandardInput = $true
		$processInfo.RedirectStandardOutput = $true
		$processInfo.RedirectStandardError = $true
		$processInfo.CreateNoWindow = $true

		$process = New-Object System.Diagnostics.Process
		$process.StartInfo = $processInfo

		$process.Start() | Out-Null # Startet den Prozess im Hintergrund

		# Sende Passwörter an stdin
		$process.StandardInput.WriteLine($NewPassword) # Direkte Übergabe des Klartext-Passworts
		$process.StandardInput.WriteLine($AdminPassword) # Direkte Übergabe des Klartext-Passworts
		$process.StandardInput.Close() # Wichtig: stdin schließen, damit der Prozess fortfährt

		$process.WaitForExit()

		$exitCode = $process.ExitCode
		$output = $process.StandardOutput.ReadToEnd()
		$errorOutput = $process.StandardError.ReadToEnd()

		Write-Verbose "sysadminctl Exit Code: $exitCode"
		if (![string]::IsNullOrWhiteSpace($output)) {
			Write-Verbose "sysadminctl Standard Output:`n$output"
		}
		if (![string]::IsNullOrWhiteSpace($errorOutput)) {
			Write-Warning "sysadminctl Standard Error:`n$errorOutput"
		}

		if ($exitCode -ne 0) {
			Write-Error "Fehler beim Setzen des Passworts für Benutzer '$TargetUserName'. Exit Code: $exitCode. Details: $errorOutput"
		} else {
			Write-Host "Passwort für Benutzer '$TargetUserName' erfolgreich gesetzt."
		}

	} catch {
		Write-Error "Ein unerwarteter Fehler ist aufgetreten: $($_.Exception.Message)"
	} finally {
		# In dieser Version gibt es keine SecureStrings zu nullen,
		# aber die Passwörter sind nach der Ausführung im Speicher der Prozess-Klasse
		# und können in der Historie des aufrufenden Scripts landen.
	}
}


Return


if ($PSCmdlet.ShouldProcess("Passwort für Benutzer '$UserName'", "erzwingen")) {
	Write-Verbose "Setze neues Passwort für Benutzer '$UserName'..."
	try {
		# Der Befehl zum erzwungenen Setzen des Passworts
		# Syntax: dscl . -passwd /Users/username newpassword
		# Hier verwenden wir die Methode über das "Password" Attribut, die das alte Passwort nicht benötigt.
		$commandOutput = & "$dsclPath" . -create "/Users/$UserName" Password "$NewPassword" 2>&1

		if ($LASTEXITCODE -ne 0) {
			Write-Error "Fehler beim Setzen des Passworts für Benutzer '$UserName'. Exit Code: $LASTEXITCODE. Output: $commandOutput"
			Write-Error "Möglicherweise wurde der Befehl bereits ausgeführt und es gab ein Problem beim 'create' auf ein bereits existierendes Feld. Versuche 'change'."
			
			# Wenn 'create' fehlschlägt, weil das Feld bereits existiert, versuchen wir 'change'
			$commandOutput = & "$dsclPath" . -change "/Users/$UserName" Password "******" "$NewPassword" 2>&1 # Das alte Passwort hier ist ein Platzhalter "******"
			if ($LASTEXITCODE -ne 0) {
				 Write-Error "Fehler beim Ändern des Passworts für Benutzer '$UserName' mit 'change'. Exit Code: $LASTEXITCODE. Output: $commandOutput"
			} else {
				 Write-Host "Passwort für Benutzer '$UserName' erfolgreich gesetzt."
			}
		} else {
			Write-Host "Passwort für Benutzer '$UserName' erfolgreich gesetzt."
		}
	} catch {
		Write-Error "Ein unerwarteter Fehler ist aufgetreten: $($_.Exception.Message)"
	}
}
