#!/usr/local/bin/pwsh
# !9^9 Unix (LF), nur UTF8 für MacOS, KEIN BOM

#Requires -Version 7



<#
	.SYNOPSIS
		Verbindet das MacBook von einer Domäne
		Fragt nach Usernamen und PW, wenn nicht angegeben
		Returns:
		$True   Domain-Join i.O.
		$False  Domain-Join nicht i.O.

	.PARAMETER ZielDomain
		Die Ziel-Domäne
	.PARAMETER NewHostName
		Optional der neue Hostname, zB 'AkMb016'
	.PARAMETER DomainAdUserName
		Der AD-Join Username, Optional.
		Wenn nicht angegeben, wird danach gefragt
	.PARAMETER DomainAdUserPW
		Das AD-Join Passwort, Optional.
		Wenn nicht angegeben, wird danach gefragt
	.PARAMETER Dbg
		Gibt Debug-Infos aus

	.EXAMPLE
		! ./Join-Domain.ps1 -ZielDomain 'akros.ch' -DomainAdUserName 'ad@akros.ch'
		 Domain-Join mit dem bestehenden Hostnamen, das PW wird abgefragt

	.EXAMPLE
		! ./Join-Domain.ps1 -ZielDomain 'akros.ch' -DomainAdUserName 'ad@akros.ch' -DomainAdUserPW 'xxx'
		Domain-Join mit PW und dem bestehenden Hostnamen

	.EXAMPLE
		! ./Join-Domain.ps1 -ZielDomain 'akros.ch' -DomainAdUserName 'ad@akros.ch' -NewHostName AkMb009a
		Domain-Join neuem Hostnamen, PW wird abgefragt

	.NOTES
		001, 250701, Tom
#>

## Suppress PSScriptAnalyzer Warning
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidUsingCmdletAliases', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseApprovedVerbs', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidDefaultValueSwitchParameter', '')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidDefaultValueForMandatoryParameter', '')]
# [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSShouldProcess', '')]
# [Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSUseSingularNouns', '')]

[CmdletBinding(SupportsShouldProcess, ConfirmImpact = 'Low', DefaultParameterSetName = 'Script')]
Param (
    [Parameter(Mandatory, Position=0, ParameterSetName='Script')]
	[String]$ZielDomain,
	# Optional der HostName
	[String]$NewHostName,
	# Optional ein UserName
	[Object]$DomainAdUserName,
	# Optional ein PW
	[Object]$DomainAdUserPW,
	[Switch]$Dbg,

	# Get-Help Parameter
	# 005, 200314
	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Switch]$GetHelp,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Switch]$GetHelpCls,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Nullable[Int]]$GetEx,

	[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, ParameterSetName = 'GetHelp')]
	[Nullable[Int]]$RunEx
)



### Init
$ScriptDir = [IO.Path]::GetDirectoryName($MyInvocation.MyCommand.Path)
$ThisScriptPath = $MyInvocation.MyCommand.Definition



### Config
$TomLibMacOS_ps1 = 'TomLib-MacOS.ps1'

# Root nur prüfen, wenn wir die Hilfe nicht abrufen
If ((-Not $GetHelp) -and (-Not $GetHelpCls) -and ($Null -eq $GetEx) -and ($Null -eq $RunEx)) {

	### Root erzwingen
	# 250620 1850
	If ((id -u) -ne 0) {
		Write-Host 'Starte das Script als root'
		If ([String]::IsNullOrWhiteSpace($ThisScriptPath)) {
			Write-Error 'Variable $ThisScriptPath ist leer. Kann nicht fortfahren.'
		}	
	
		# Argumente aus den BoundParameters zusammensetzen
		$paramList = @()
		ForEach ($kvp in $PSBoundParameters.GetEnumerator()) {
			If ($kvp.Value -is [switch] -and $kvp.Value.IsPresent) {
				$paramList += "-$($kvp.Key)"
			} Else {
				$paramList += "-$($kvp.Key) `"$($kvp.Value)`""
			}
		}
	
		$pwsh = (Get-Command pwsh).Source
	
		If ($paramList.Count -gt 0) {
			# sudo $pwsh $ThisScriptPath @paramList
			$quotedArgs = $paramList -join ' '
			$cmd = "$pwsh `"$ThisScriptPath`" $quotedArgs"
			sudo bash -c "$cmd"
		} Else {
			sudo $pwsh $ThisScriptPath
		}
		Exit
	}
	
	# Ab hier läuft das Script als root
	Write-Host "`nScript läuft jetzt als root`n`n" -Fore DarkGray
}



### Funcs



### Prepare

# Lib laden
$TomLibMacOS_ps1 = Join-Path $ScriptDir $TomLibMacOS_ps1
. $TomLibMacOS_ps1 $PSBoundParameters -ParseGetHelp

# Aktuelle Hostnamen abrufen
$oHostnamesInfo = Get-HostNames-Info
# Print-HostNamesInfo -oHostnamesInfo $oHostnamesInfo

# Params bereinigen
$ZielDomain = $ZielDomain.Trim()
If(Has-Value $NewHostName) { $NewHostName = $NewHostName.Trim() }



### Main
If ($IsVerbose) { Log 0 'Join-Domain' }


# Sind wir schon Domain-Joined?
If ($oHostnamesInfo.IsDomainJoined) {
	If ($IsVerbose) { Log 1 'MacOS ist Domain-Joined' }

	If ($oHostnamesInfo.JoinedDomain -eq $ZielDomain) {
		If ($IsVerbose) { Log 2 "Ist bereits in der Domäne: $ZielDomain"  -Fore Green }
		Return [PSCustomObject]@{
			Resultat = $True
			Message  = "Ist bereits in der Domäne: $ZielDomain"
		}
		
	} Else {
		If ($IsVerbose) { Log 2 "Ist in einer anderen Domäne: $($oHostnamesInfo.JoinedDomain)" -Fore Red } 
		Return [PSCustomObject]@{
			Resultat = $False
			Message  = "Ist in einer anderen Domäne: $($oHostnamesInfo.JoinedDomain)"
		}
	}
}


## Das Notebook ist nicht Domain-Joined

## Kann das Notebook die Domäne anpingen?

$IsComputerInDomainLAN = Is-Computer-in-Domain-LAN -Domain $ZielDomain

If (-not $IsComputerInDomainLAN) {
	Return [PSCustomObject]@{
		Resultat = $False
		Message  = "Notebook kann die Domäne nicht anpingen: $ZielDomain"
	}
}


## Müssen wir den MacOS HostNamen ändern?
$IsFQDN_Domain_Missing = Is-Empty $oHostnamesInfo.FQDN_Domain
$IsWrongDomain = $oHostnamesInfo.JoinedDomain -ne $ZielDomain

$IsNewHostName = $False
If (Has-Value $NewHostName) {
	$CurrentMacOSHostName = Get-Current-HostName $oHostnamesInfo
	$IsNewHostName = $NewHostName -ne $CurrentMacOSHostName
} Else {
	# Keinen neuen Hostnamen, wir ändern ihn nicht
	$NewHostName = $Null
}


# Allenfalls den Hostnamen anpassen
If ($IsFQDN_Domain_Missing `
	-or $IsWrongDomain `
	-or $IsNewHostName) {

	# Den Hostnamen anpassen
	Log 1 "Setze den neuen Hostnamen allenfalls mit der Domäne: $ZielDomain"

	# Den Hostnamen um die Domäne ergänzen
	Set-HostNames -NewHostName $NewHostName `
					-DomainName $ZielDomain `
					-Force:$True `
					-Dbg:$False

	If ($Dbg) {
		$oHostnamesInfoDbg = Get-HostNames-Info
		Print-HostNamesInfo -oHostnamesInfo $oHostnamesInfoDbg
	}
}


## Join Domain
$JoinOK = Join-MacOS-to-Domain -DomainName $ZielDomain `
								-DomainAdUserName $DomainAdUserName `
								-DomainAdUserPW $DomainAdUserPW

Log 0 'Neue MacOS Hostnamen'
$oHostnamesInfoNeu = Get-HostNames-Info
Print-HostNamesInfo -oHostnamesInfo $oHostnamesInfoNeu

Return $JoinOK
