#!/usr/bin/env zsh
# Format: UTF-8, Linux EOL

# Skript bei Fehler abbrechen
set -e


# Format: UTF8, Linux EOL
# 006, 250407, tom@jig.ch
# 007, 250610, tom@jig.ch
# 008, 250611, tom@jig.ch
#	Wenn Tools schon installiert sind,
#	werden wenn vorhanden Updates installiert
# 009, 250627, tom@jig.ch
#	neu zsh
# 010, 250709, tom@jig.ch
#	neu programmiert: _compare_versions


# Bereitet MacOS für die Automatisierung mit NinjaOne vor.
#
# Installiert diese Tools, wenn sie nicht schon installiert sind:
# - Xcode Command Line Tools
# - Homebrew
#   » Ist ein Albtraum Paket-"Manager":
#     - Man kann Homebrew nicht als root installieren, 
#       um ihn allen Usern zur Verfügung stellen
#     - Man kann ihn auch nicht ad-hoc in NinjaOne installieren,
#       weil NinjaOne Scripts als Root laufen
# - MacPorts
#   Geniale Lib mit vielen für MacOS portierte Linux Tools
# - jq (Command-line JSON processor)
# - PowerShell Core

# Installation auf NinjaOne
# https://eu.ninjarmm.com/#/administration/library/scripting/scripts



### Root-Forcierung
if [[ $EUID -ne 0 ]]; then
	echo "Dieses Script benötigt root-Rechte. Es wird erneut mit sudo ausgeführt..."

	# Das Script mit sudo neu starten.
	# "$0" ist der Pfad zum aktuellen Script.
	# "$@" sind alle Argumente, die dem Script ursprünglich übergeben wurden.
	# So wird sichergestellt, dass z.B. './script.sh --arg1 value1' zu
	# 'sudo ./script.sh --arg1 value1' wird.
	sudo zsh "$0" "$@"

	# Den ursprünglichen (nicht-root) Prozess beenden.
	# $? enthält den Exit-Code des sudo-Befehls.
	exit $?
fi

# Ab hier: Root



### Config
# Standardwert für PortCmd
PortCmd="/opt/local/bin/port"



### Funcs

## Wenn noch nicht vorhanden, installieren: 
## Xcode Command Line Tools
Install_or_Update_XCode_CLI_Tools() {
	echo "\nPrüfe: Xcode Command Line Tools"
	
	if ! xcode-select -p &>/dev/null; then
		echo "  > Beginne die Installation"

		# Verwenden Sie den SoftwareUpdate-Mechanismus, um zu installieren
		touch /tmp/.com.apple.dt.CommandLineTools.installondemand.in-progress

		# PROD=$(softwareupdate -l | grep "\*.*Command Line" | head -n 1 | awk -F"*" '{print $2}' | sed 's/^ *//')
		# Die neuste Version ist die letzte Zeile
		PROD=$(softwareupdate -l \
			| grep '\*.*Command Line Tools for Xcode' \
			| tail -n1 \
			| awk -F"Label: " '{print $2}' \
			| sed 's/^ *//')
		# echo $PROD

		softwareupdate --install "$PROD" --verbose

		# Bereinigen
		rm /tmp/.com.apple.dt.CommandLineTools.installondemand.in-progress

		# Warten, bis die Command Line Tools verfügbar sind
		until xcode-select -p &>/dev/null; do
			echo "  > Warte auf Abschluss der Installation"
			sleep 5
		done

		echo "  > OK, Xcode Command Line Tools erfolgreich installiert"

	else
		echo "  > OK, ist bereits installiert."
		# Haben wir Updates?
		echo "  > Prüfe auf Updates..."

		if softwareupdate -l | grep -q "Command Line Tools Update"; then
			echo "  > Update verfügbar. Installiere..."
			# Hier wird der spezifische Label für das Update gesucht und installiert
			UPDATE_PROD=$(softwareupdate -l \
				| grep '\*.*Command Line Tools Update' \
				| tail -n1 \
				| awk -F"Label: " '{print $2}' \
				| sed 's/^ *//')

			if [[ -n $UPDATE_PROD ]]; then
				softwareupdate --install "$UPDATE_PROD" --verbose
				echo "  > Update erfolgreich installiert."
			else
				echo "  > Konnte das Update-Label nicht finden."
			fi
		else
			echo "  > Keine Updates verfügbar."
		fi
	fi
}


## MacPorts installieren
## https://guide.macports.org
Install_or_Update_MacPorts() {
	echo "\nPrüfe: MacPorts"

	# Prüfen, ob der Befehl "port" existiert
	if command -v port &>/dev/null || command -v /opt/local/bin/port &>/dev/null; then
		echo "  > OK, ist bereits installiert"
		echo "  > Prüfe und installiere Updates"
		/opt/local/bin/port -v selfupdate
		/opt/local/bin/port -v upgrade outdated
		echo "  > MacPorts Update-Prüfung abgeschlossen."
		return
	fi

	# macOS Versionsnummer auslesen, z.B. "10.14.6" oder "11.2.3"
	OS_VERSION=$(sw_vers -productVersion)
	OS_MAJOR=${OS_VERSION%%.*}
	OS_MINOR=${${OS_VERSION#*.}%%.*}

	CODENAME="Unbekannt"
	MacPortsUrl=""

	# Codename bestimmen
	case "$OS_MAJOR.$OS_MINOR" in
		10.12) CODENAME="Sierra"        ;;
		10.13) CODENAME="High Sierra"  ;;
		10.14) CODENAME="Mojave"       ;;
		10.15) CODENAME="Catalina"     ;;
		11.*)  CODENAME="Big Sur"      ;;
		12.*)  CODENAME="Monterey"     ;;
		13.*)  CODENAME="Ventura"      ;;
		14.*)  CODENAME="Sonoma"       ;;
		*)     CODENAME="Unbekannt"    ;;
	esac

	# Download-URLs:
	# https://guide.macports.org/#installing.macports
	case "$CODENAME" in
		"Sierra")       MacPortsUrl="https://github.com/macports/macports-base/releases/download/v2.10.5/MacPorts-2.10.5-10.12-Sierra.pkg" ;;
		"High Sierra")  MacPortsUrl="https://github.com/macports/macports-base/releases/download/v2.10.5/MacPorts-2.10.5-10.13-HighSierra.pkg" ;;
		"Mojave")       MacPortsUrl="https://github.com/macports/macports-base/releases/download/v2.10.5/MacPorts-2.10.5-10.14-Mojave.pkg" ;;
		"Catalina")     MacPortsUrl="https://github.com/macports/macports-base/releases/download/v2.10.5/MacPorts-2.10.5-10.15-Catalina.pkg" ;;
		"Big Sur")      MacPortsUrl="https://github.com/macports/macports-base/releases/download/v2.10.5/MacPorts-2.10.5-11-BigSur.pkg" ;;
		"Monterey")     MacPortsUrl="https://github.com/macports/macports-base/releases/download/v2.10.5/MacPorts-2.10.5-12-Monterey.pkg" ;;
		"Ventura")      MacPortsUrl="https://github.com/macports/macports-base/releases/download/v2.10.5/MacPorts-2.10.5-13-Ventura.pkg" ;;
		"Sonoma")       MacPortsUrl="https://github.com/macports/macports-base/releases/download/v2.10.5/MacPorts-2.10.5-14-Sonoma.pkg" ;;
		*)              MacPortsUrl="" ;;
	esac

	echo "  > macOS Version: $OS_VERSION, Codename: $CODENAME"

	# Wenn wir die Version haben, das Paket installieren
	if [[ -n $MacPortsUrl ]]; then
		echo "  > OS Codename: $CODENAME"
		echo "  > Beginne die Installation"
		InstallerPath="/tmp/macports.pkg"

		echo "  > Download Installer…"
		curl -L -o "$InstallerPath" "$MacPortsUrl"

		if [[ $? -ne 0 ]]; then
			echo -e "\n  » Fehler beim Herunterladen von MacPorts!\n"
			exit 1
		fi

		# Paket installieren
		echo "  > Installiere MacPorts…"
		installer -pkg "$InstallerPath" -target /

		if [[ $? -ne 0 ]]; then
			echo -e "\n  » Fehler bei der Installation von MacPorts!\n"
			exit 1
		fi

		# Aufräumen
		rm -f "$InstallerPath"

		# Nach der Installation: Port-Befehl erneut prüfen
		if ! command -v /opt/local/bin/port &>/dev/null; then
			echo "\n  » Installation abgeschlossen, aber der Befehl 'port' wurde nicht gefunden.\n"
		else
			echo "  > Prüfe und installiere Updates"
			/opt/local/bin/port -v selfupdate
			/opt/local/bin/port -v upgrade outdated
			echo "  > MacPorts Update-Prüfung abgeschlossen."
		fi
	fi
}


# Hilfsfunktion zum Vergleich von Versionsnummern
# ResCompareVer == 0, wenn Version1 = Version2
# ResCompareVer == 1, wenn Version1 > Version2
# ResCompareVer == 2, wenn Version1 < Version2
# ResCompareVer == 3, wenn Fehlerhafte Eingabe
# Beispiel: _compare_versions "7.4.2" "7.4.1" -> gibt 1 zurück
_compare_versions() {
    # Vergleichsergebnis global zurückgeben
    typeset -g ResCompareVer=3

    if [[ -z $1 || -z $2 ]]; then
        echo "Fehler: Zwei Versionsnummern müssen angegeben werden." >&2
        return 0
    fi

    local v1_full=$1
    local v2_full=$2

    # Haupt- und Prerelease-Anteile extrahieren
    local v1_main=${v1_full%%-*}
    local v1_pre=${v1_full#*-}
    [[ $v1_main == $v1_full ]] && v1_pre=""

    local v2_main=${v2_full%%-*}
    local v2_pre=${v2_full#*-}
    [[ $v2_main == $v2_full ]] && v2_pre=""

    local IFS='.'
    local -a v1_parts
    local -a v2_parts
    v1_parts=(${(s:.:)v1_main})
    v2_parts=(${(s:.:)v2_main})

    local max_len=$(( ${#v1_parts} > ${#v2_parts} ? ${#v1_parts} : ${#v2_parts} ))

    for (( i = 1; i <= max_len; i++ )); do
        local num1=${v1_parts[i]:-0}
        local num2=${v2_parts[i]:-0}

        num1=$(( 10#$num1 ))
        num2=$(( 10#$num2 ))

        if (( num1 > num2 )); then
            ResCompareVer=1
            return 0
        elif (( num1 < num2 )); then
            ResCompareVer=2
            return 0
        fi
    done

    # Pre-Release-Vergleich
    if [[ -z $v1_pre && -n $v2_pre ]]; then
        ResCompareVer=1
    elif [[ -n $v1_pre && -z $v2_pre ]]; then
        ResCompareVer=2
    elif [[ -n $v1_pre && -n $v2_pre ]]; then
        if [[ $v1_pre > $v2_pre ]]; then
            ResCompareVer=1
        elif [[ $v1_pre < $v2_pre ]]; then
            ResCompareVer=2
        else
            ResCompareVer=0
        fi
    else
        ResCompareVer=0
    fi

    return 0
}



## Hilfsfunktion zum Installieren einer spezifischen PowerShell-Version
Install_PowerShell_Version_() {
	local PS_VERSION_TO_INSTALL=$1

	if [[ -z $PS_VERSION_TO_INSTALL ]]; then
		echo "\n  » Interner Fehler: Keine PowerShell-Version angegeben!\n"
		return 1
	fi

	echo "  > Installiere PowerShell Version: $PS_VERSION_TO_INSTALL"

	# Bestimmen der Systemarchitektur: x86_64 (Intel) oder arm64 (Apple Silicon)
	local ARCH=$(uname -m)

	echo "  > Bestimme Download-URL für Architektur: $ARCH"

	local RELEASES_API_URL="https://api.github.com/repos/PowerShell/PowerShell/releases"
	
	[[ $PS_VERSION_TO_INSTALL != "latest" ]] \
		&& RELEASES_API_URL="${RELEASES_API_URL}/tags/v${PS_VERSION_TO_INSTALL}" \
		|| RELEASES_API_URL="${RELEASES_API_URL}/latest"

	local PKG_URL

	if [[ $ARCH == "x86_64" ]]; then
		PKG_URL=$(curl -s "$RELEASES_API_URL" \
			| jq -r '.assets[] | select(.name | test("osx-x64.pkg$")) | .browser_download_url')
	elif [[ $ARCH == "arm64" ]]; then
		PKG_URL=$(curl -s "$RELEASES_API_URL" \
			| jq -r '.assets[] | select(.name | test("osx-arm64.pkg$")) | .browser_download_url')
	else
		echo "\n  » Unbekannte Architektur: $ARCH\n"
		return 1
	fi

	if [[ -z $PKG_URL ]]; then
		echo "\n  » Paket für die Architektur nicht gefunden!\n"
		return 1
	fi

	echo "  > Lade PowerShell-Paket von: $PKG_URL"
	local TMP_PKG="/tmp/powershell.pkg"
	curl -L -o "$TMP_PKG" "$PKG_URL"

	if [[ $? -ne 0 ]]; then
		echo "\n  » Download des Pakets fehlgeschlagen!\n"
		rm -f "$TMP_PKG"
		return 1
	fi

	echo "\n  > Installiere PowerShell $PS_VERSION_TO_INSTALL …"
	installer -pkg "$TMP_PKG" -target /

	if [[ $? -ne 0 ]]; then
		echo "\n  » PowerShell-Installation fehlgeschlagen!\n"
		rm -f "$TMP_PKG"
		return 1
	fi

	rm -f "$TMP_PKG"
	echo "PowerShell $PS_VERSION_TO_INSTALL erfolgreich installiert."
	return 0
}


## PowerShell (pwsh)
## Installieren oder aktualisieren
Install_or_Update_PowerShell() {
	echo "\nPrüfe: PowerShell (pwsh)"

	# Dieses Skript muss als root ausgeführt werden.
	if [[ $(id -u) -ne 0 ]]; then
		echo "\n  » Muss als root ausgeführt werden!\n"
		return 1
	fi

	# Prüfen, ob jq installiert ist
	if ! command -v jq &>/dev/null; then
		echo "\n  » jq ist nicht installiert! Kann keine Installations-URL ermitteln.\n"
		return 1
	fi

	if ! command -v pwsh &>/dev/null; then
		echo "  > Beginne die Installation"
		Install_PowerShell_Version_ "latest" || {
			echo "\n  » Installation der neuesten PowerShell-Version fehlgeschlagen!\n"
			exit 1
		}
	else
		echo "  > OK, pwsh ist bereits installiert."
		echo "  > Prüfe auf Updates..."

		# Installierte PowerShell-Version ermitteln
		INSTALLED_PS_VERSION=$(pwsh -Command '$PSVersionTable.PSVersion.ToString()' | sed 's/^v//')

		# Neueste verfügbare PowerShell-Version ermitteln
		LATEST_PS_VERSION=$(curl -s https://api.github.com/repos/PowerShell/PowerShell/releases/latest \
			| jq -r .tag_name \
			| sed 's/^v//')

		echo "  > Installierte Version: $INSTALLED_PS_VERSION"
		echo "  > Neueste Version: $LATEST_PS_VERSION"

		# Versionsvergleich
		# echo "Start Versionsvergleich: $INSTALLED_PS_VERSION vs $LATEST_PS_VERSION"
		_compare_versions "$INSTALLED_PS_VERSION" "$LATEST_PS_VERSION"
		# echo "Ergebnis: $ResCompareVer"  # Erwartet: 1 (weil 7.5.2 > 7.5.1)

		if [[ $ResCompareVer -eq 2 ]]; then
			# Installierte Version ist älter als die neueste
			echo "  > Update verfügbar! Installiere Version $LATEST_PS_VERSION..."
			
			Install_PowerShell_Version_ "$LATEST_PS_VERSION" || {
				echo "\n  » Installation des PowerShell-Updates fehlgeschlagen!\n"
				exit 1
			}
		elif [[ $ResCompareVer -eq 1 ]]; then
			# Installierte Version ist neuer als die neueste 
			# (z.B. eine Vorabversion ist installiert)
			echo "  > Eine neuere oder Vorabversion ($INSTALLED_PS_VERSION) ist bereits installiert."
		else
			echo "  > PowerShell ist bereits aktuell."
		fi
	fi
}



## Installieren: jq (Command-line JSON processor)
## https://github.com/jqlang/jq?tab=readme-ov-file
Install_jq() {
	echo "\nPrüfe: jq"

	# Prüfen, ob der Befehl "port" existiert und "jq" installiert ist
	if command -v port &>/dev/null && port installed jq &>/dev/null; then
		echo "  > jq ist bereits installiert."
		echo "  > Prüfe auf Updates für jq..."

		if port outdated jq | grep -q 'jq @'; then
			port -N -q upgrade jq \
				&& echo "  > jq erfolgreich aktualisiert." \
				|| echo "  » Fehler beim Aktualisieren von jq!"
		else
			echo "  > jq ist auf dem neuesten Stand."
		fi

		return
	fi

	# Wenn hierher gekommen, bedeutet das entweder:
	# 1. 'port' ist nicht installiert, oder
	# 2. 'port' ist installiert, 
	#		aber 'jq' wurde damit noch nicht installiert.
	if ! command -v jq &>/dev/null; then
		echo "  > Beginne die Installation von jq."

		if ! command -v port &>/dev/null; then
			echo "  » MacPorts ist nicht installiert. Bitte zuerst MacPorts installieren."
			return 1
		fi

		# jq installieren
		echo "  > Installiere jq via MacPorts..."
		port -N -q install jq \
			&& echo "  > jq erfolgreich installiert." \
			|| { echo "  » Fehler bei der Installation von jq!"; return 1 }
	else
		# Dieser Zweig sollte eigentlich nicht mehr erreicht werden, da die obige Logik alles abdeckt.
		# Er dient als Fallback, falls jq ohne MacPorts installiert wurde, was hier nicht erwartet wird.
		echo "  > jq ist bereits installiert (möglicherweise nicht via MacPorts)."
	fi
}



### Main

## Debug in der Shell:
# Install_or_Update_XCode_CLI_Tools false
# Install_or_Update_PowerShell false

Install_or_Update_XCode_CLI_Tools
Install_or_Update_MacPorts

# Den Befehlt port suchen
# Zuerst im Pfad
if command -v port &>/dev/null; then
	PortCmd=$(command -v port)
elif [[ -x /opt/local/bin/port ]]; then
	# Sonst das Default Setup-Dir nützen
	PortCmd="/opt/local/bin/port"
fi

Install_jq
Install_or_Update_PowerShell


