#!/usr/local/bin/pwsh
# !9^9 Nur UTF8 für MacOS, KEIN BOM, Unix (LF)

# Lib für MacOS

# 001, 250701, Tom


# !Ex
#
### Init
# $ScriptDir = [IO.Path]::GetDirectoryName($MyInvocation.MyCommand.Path)
#
### Config
# $TomLibMacOS_ps1 = 'TomLib-MacOS.ps1'
#
### Prepare
# Lib laden
# $TomLibMacOS_ps1 = Join-Path $ScriptDir $TomLibMacOS_ps1
# . $TomLibMacOS_ps1 $PSBoundParameters


[CmdletBinding(SupportsShouldProcess, ConfirmImpact = 'Low')]
Param (
	# Optional $PSBoundParameters, 
	# um zu erkennen: IsVerbose, IsWhatIf 
	[HashTable]$PSBoundParameters_
)



### Init

# Haben wir Verbose / WhatIf?
If ($PSBoundParameters_) {
	$IsVerbose = $PSBoundParameters_.ContainsKey('Verbose') -and $PSBoundParameters_['Verbose']
	$IsWhatIf = $PSBoundParameters_.ContainsKey('WhatIf') -and $PSBoundParameters_['WhatIf'] -or $Force -eq $False
}



### Config



### Funcs

#Region Root User

# True, wenn wir Root-Rechte haben
Function Is-Root {
    return (id -u) -eq 0
}

#Endregion Root User


#Region MacOS User handling

#  Meldet den aktuellen macOS User ab
Function Logoff-MacOS-User($LoggedInUser) {
	try {
		# Diese Befehle erfordern sudo, um andere Benutzer abzumelden.
		# Für den angemeldeten Benutzer selbst reicht oft der direkte Aufruf ohne sudo aus,
		# aber da das Skript als sudo läuft, ist es gut, den launchctl-Ansatz zu verwenden.
		# Alternativ: & osascript -e 'tell app "System Events" to log out'
		# Die launchctl bootout Methode ist robuster für Skripte, die als root laufen.
		
		# UID des angemeldeten Benutzers ermitteln
		$LoggedInUserUID = $(id -u $LoggedInUser)
		# Logoff
		& launchctl bootout user/$LoggedInUserUID
		Write-Host 'Benutzerabmeldung erfolgreich eingeleitet.'
	} catch {
		Write-Error "Fehler beim Einleiten der Benutzerabmeldung: $($_.Exception.Message)"
		Write-Host "Bitte melden Sie sich manuell ab und wieder an, um FileVault zu aktivieren."
	}	
}


#Endregion MacOS User handling


#Region Log

# 250701
# 	Portiert der Windows-PowerShell Version:
# 		220123 222224
# 		Fixed: -ReplaceLine handling
$Script:LogColors = @('DarkMagenta', 'DarkGreen', 'DarkCyan', 'DarkBlue', 'Red')

# Mapping von ConsoleColor zu ANSI-Farbcodes
$Script:AnsiColorMap = @{
    Black      = 30; DarkBlue = 34; DarkGreen   = 32; 
	DarkCyan   = 36; DarkRed  = 31; DarkMagenta = 35; 
	DarkYellow = 33; Gray     = 37; DarkGray    = 90; 
	Blue       = 94; Green    = 92; Cyan        = 96;
    Red        = 91; Magenta  = 95; Yellow      = 93; 
	White      = 97;
}

Function Convert-ToAnsiColor {
    param (
        [ConsoleColor]$Color,
        [Switch]$Background
    )
    $ansi = $Script:AnsiColorMap[$Color.ToString()]
    if ($Background) {
        # Hintergrundfarben: 10er-Offset
        $ansi += 10
    }
    return "`e[${ansi}m"
}
Function Log {
    Param (
        [Parameter(Position = 0)]
        [Int]$Indent = 0,

        [Parameter(Position = 1, ValueFromPipeline, ValueFromPipelineByPropertyName)]
        [String]$Message = '',

        [Parameter(Position = 2)]
        [ConsoleColor]$ForegroundColor,

        [Parameter(Position = 3)]
        [Switch]$NewLineBefore,

        [Parameter(Position = 4)]
        [Switch]$ReplaceLine = $false,

        [Parameter(Position = 5)]
        [Switch]$NoNewline = $false,

        [Parameter(Position = 6)]
        [Switch]$Append = $false,

        [Parameter(Position = 7)]
        [Switch]$ClrToEol = $false,

        [Parameter(Position = 8)]
        [ConsoleColor]$BackgroundColor
    )

    begin {
        if ($Script:LogDisabled -eq $true) { return }

        if (-not $ForegroundColor) {
            if ($Indent -lt $Script:LogColors.Count) {
                $ForegroundColor = [ConsoleColor]::Parse([ConsoleColor], $Script:LogColors[$Indent])
            } else {
                $ForegroundColor = [ConsoleColor]::White
            }
        }

        $AnsiStart = Convert-ToAnsiColor -Color $ForegroundColor
        if ($BackgroundColor) {
            $AnsiStart += Convert-ToAnsiColor -Color $BackgroundColor -Background
        }
        $AnsiEnd = "`e[0m"
    }

    process {
        if ($NewLineBefore) { Write-Host "" }

        if (-not $Append) {
            switch ($Indent) {
                0 { $Message = "* $Message" }
                default { $Message = (' ' * ($Indent * 2)) + $Message }
            }
        }

        if ($ReplaceLine) {
            $Message = "`r`e[0K$Message"
        }

        if ($ClrToEol) {
            try {
                $Width = $Host.UI.RawUI.WindowSize.Width
                if ($Message.Length -lt $Width) {
                    $Message += ' ' * ($Width - $Message.Length)
                }
            } catch { }
        }

        if ($NoNewline) {
            Write-Host "$AnsiStart$Message$AnsiEnd" -NoNewline
        } else {
            Write-Host "$AnsiStart$Message$AnsiEnd"
        }
    }
}

#Endregion Log



#Region Hostname

# Gibt auf der Konsole die Info der Hostnamen aus
# !Ex
# $oHostnamesInfo = Get-HostNames-Info
# Print-HostNamesInfo -oHostnamesInfo $oHostnamesInfo
Function Print-HostNamesInfo($oHostnamesInfo) {

	Log 0 'HostNamesInfo'

	Log 1 'Domain-Join : ' -NoNewline
	If ($oHostnamesInfo.IsDomainJoined) {
		Log -Append -Message 'Ja' -Fore Red
	} Else {
		Log -Append -Message 'Nein' -Fore Red
	}
	
    Log 1 "DomainJoined: $($oHostnamesInfo.DomainJoined)"

	Log 1 'Aktuelle Hostnamen'
    Log 2 "  ComputerName   : $($oHostnamesInfo.ComputerName)"
    Log 2 "  HostName       : $($oHostnamesInfo.HostName)"

    Log 2 "  LocalHostName  : $($oHostnamesInfo.LocalHostName)"
    Log 2 "  > Effektiv     : $($oHostnamesInfo.LocalHostNameEffektiv)"

}


# Berechnet alle Infos zum MacOS Hostnamen
# $oHostnamesInfo = Get-HostNames-Info
Function Get-HostNames-Info() {
	# Hostnamen bestimmen
	$oHostnamesInfo = [PSCustomObject][Ordered]@{
		IsDomainJoined = Is-Domain-joined
		DomainJoined = Get-Joined-Domain
		ComputerName = "$(scutil --get ComputerName 2>$null)"
		HostName = "$(scutil --get HostName 2>$null)"
		LocalHostName = "$(scutil --get LocalHostName 2>$null)"
		LocalHostNameEffektiv = "$(scutil --get LocalHostName 2>$null).local"
	}
	Return $oHostnamesInfo
}


# Setzt die MacOS Hostnamen
Function Set-HostNames {
	Param (
		[String]$Name,
		[PSCustomObject]$oHostnamesInfo,
		[Switch]$Force
	)
	
	$Name = $Name.Trim()

	If ($Force) {
		# Write-Host "Setze Hostnamen: '$Name'..."
		
		sudo scutil --set ComputerName "$Name"
		sudo scutil --set HostName "$Name"
		# Erzeugt automatisch $Name.local
		sudo scutil --set LocalHostName "$Name"

		Write-Host "Hostnamen wurden gesetzt."
		
	} Else {
		# Write-Host "Würde Hostnamen setzen:"
		
		Write-Host "ComputerName : " -NoNewLine
		If ($oHostnamesInfo.ComputerName -eq $Name) {
			Write-Host 'Schon OK' -Fore Green
		} Else {
			Write-Host $Name -Fore Red
		}
		
		Write-Host "HostName     : " -NoNewLine
		If ($oHostnamesInfo.HostName -eq $Name) {
			Write-Host 'Schon OK' -Fore Green
		} Else {
			Write-Host $Name -Fore Red
		}
		
		Write-Host "LocalHostName: " -NoNewLine
		If ($oHostnamesInfo.LocalHostName -eq $Name) {
			Write-Host 'Schon OK' -Fore Green
		} Else {
			Write-Host $Name -Fore Red
		}
		
	}
}

#Endregion Hostname


#Region Domain-Join

# Ist das MacOS Domain-Joined?
Function Is-Domain-joined {

    #Region dsconfigad -show

    # Active Directory Forest          = akros.ch
    # Active Directory Domain          = akros.ch
    # Computer Account                 = anb289$

    # Advanced Options - User Experience
    #   Create mobile account at login = Enabled
    #      Require confirmation        = Enabled
    #   Force home to startup disk     = Enabled
    #      Mount home as sharepoint    = Enabled
    #   Use Windows UNC path for home  = Enabled
    #      Network protocol to be used = smb
    #   Default user Shell             = /bin/bash

    # Advanced Options - Mappings
    #   Mapping UID to attribute       = not set
    #   Mapping user GID to attribute  = not set
    #   Mapping group GID to attribute = not set
    #   Generate Kerberos authority    = Enabled

    # Advanced Options - Administrative
    #   Preferred Domain controller    = not set
    #   Allowed admin groups           = not set
    #   Authentication from any domain = Enabled
    #   Packet signing                 = allow
    #   Packet encryption              = allow
    #   Password change interval       = 14
    #   Restrict Dynamic DNS updates   = not set
    #   Namespace mode                 = domain

    #Endregion dsconfigad -show

	$adInfo = dsconfigad -show 2>$null
	$IsDomainJoined = $LASTEXITCODE -eq 0 -and $adInfo
	Return $IsDomainJoined
}


# Liest die Domäne, zu der das MacOS joined ist
Function Get-Joined-Domain {
    $dsconfigadOutput = & dsconfigad -show 2>$null
	If ($LastExitCode -eq 0 -and $dsconfigadOutput) {
        $Domain = ($dsconfigadOutput | ? { $_ -match "^Active Directory Domain" }) -replace "^.*=\s*", ""
		Return $Domain.Trim()
	} Else {
        Return $Null
	}
}


# Domain-Join für MacOS
Function Join-MacOS-to-Domain {
    Param (
		[Parameter(Mandatory)]
        [String]$DomainName,
		[Parameter(Mandatory)]
        [String]$Username,
        [SecureString]$SecurePW
		[Parameter(Mandatory)]
    )

	Log 0 'Starte Domain-Join'

	## Init
	If (Is-Root -eq $False) {
		Log 4 'Domain-Join benötigt Root' -Fore Red
		Log 5 'Abbruch' -Fore Magenta
		Return $False
	}

	$DomainName = $DomainName.Trim()

	$oHostnamesInfo = Get-HostNames-Info
	$ComputerName = $oHostnamesInfo.LocalHostName

	
    try {
		$PW = [System.Runtime.InteropServices.Marshal]::PtrToStringAuto([System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($SecurePW))

        # Führe dsconfigad aus. 2>&1 leitet stderr nach stdout um, um Fehlermeldungen abzufangen.
        $result = & /usr/sbin/dsconfigad -add "$DomainName" -computer "$ComputerName" -username "$Username" -password "$PW" -force 2>&1

		# Fehlerauswertung
		$HasError = $False
		Switch -regex ($result) {
			"invalid username or password" {
				$HasError = $True
				Log 4 'Domain-Join failed:'
				Log 4 'Benutzername oder Passwort ungültig'
				Break
			}
			"failed to add" {
				$HasError = $True
				Log 4 'Domain-Join failed:'
				Log 5 '- Existiert der Computername bereits im AD?'
				Log 5 '- Fehlt eine Berechtigung?'
				break
			}
			"Authentication server cannot be contacted" {
				$HasError = $True
				Log 4 'Domain-Join failed:'
				Log 5 'Domänencontroller nicht erreichbar'
				break
			}
			default {
				Write-Error "Unbekannter Fehler beim Hinzufügen zur Domäne."
				Write-Error "Details: $result"
			}
		}
        
        # Überprüfe den Exit-Code des letzten Befehls und die Ausgabe auf Fehlermuster.
        if ($LASTEXITCODE -eq 0 -and $result -notmatch "failed to add" -and $result -notmatch "invalid username or password" -and $result -notmatch "Authentication server cannot be contacted") {
            Write-Host "Mac erfolgreich zur Domäne hinzugefügt." -ForegroundColor Green
            return $true
        } else {
            Write-Error "Fehler beim Hinzufügen des Macs zur Domäne. Mögliche Ursachen: falscher Benutzername/Passwort, Computername bereits im AD vorhanden oder keine Verbindung zum AD."
            Write-Error "Details: $result"
            return $false
        }
    } catch {
        Write-Error "Ein unerwarteter Fehler ist aufgetreten beim Hinzufügen zur Domäne: $($_.Exception.Message)"
        return $false
    } finally {
        # Sicheres Löschen des Passworts aus dem Speicher
        [System.Runtime.InteropServices.Marshal]::ZeroFreeBSTR([System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($SecurePW))
    }
}


# MacOS: Löscht den Domain-Join
Function Unjoin-MacOS-from-Domain {
    Param (
		[Parameter(Mandatory)]
        [String]$Username,
		[Parameter(Mandatory)]
        [SecureString]$SecurePW
    )

    Log 0 'Entferne Mac aus der Domäne'

	If (Is-Root -eq $False) {
		Log 4 'Domain-UnJoin benötigt Root' -Fore Red
		Log 5 'Abbruch' -Fore Magenta
		Return $False
	}
    
    try {
        $PW = [System.Runtime.InteropServices.Marshal]::PtrToStringAuto([System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($SecurePW))
        # Führe dsconfigad aus. 2>&1 leitet stderr nach stdout um, um Fehlermeldungen abzufangen.
        $Res = & /usr/sbin/dsconfigad -remove -force -u "$Username" -p "$PW" 2>&1
        
        # Überprüfe den Exit-Code des letzten Befehls und die Ausgabe auf Fehlermuster.
        if ($LASTEXITCODE -eq 0 -and $Res -notmatch "failed to remove" -and $Res -notmatch "invalid username or password" -and $Res -notmatch "Authentication server cannot be contacted") {
            Write-Host "Mac erfolgreich aus der Domäne entfernt." -ForegroundColor Green
            return $true
        } else {
            Write-Error "Fehler beim Entfernen des Macs aus der Domäne. Mögliche Ursachen: falscher Benutzername/Passwort oder keine Verbindung zum AD."
            Write-Error "Details: $Res"
            return $false
        }
    } catch {
        Write-Error "Ein unerwarteter Fehler ist aufgetreten beim Entfernen aus der Domäne: $($_.Exception.Message)"
        return $false
    } finally {
        # Sicheres Löschen des Passworts aus dem Speicher
        [System.Runtime.InteropServices.Marshal]::ZeroFreeBSTR([System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($SecurePW))
    }
}


#Endregion Domain-Join


#Region MacOS File Vault

# Liefert $True, wenn FileVault aktiviert ist
Function Is-FileVault-Enabled {
    try {
		# fdesetup status
		# 	FileVault is Off.
		# 	Deferred enablement appears to be active for user 'root'
		# 	> kann so wieder ausgeschaltet werden:
		# 		sudo fdesetup disable -user root
        $fileVaultStatus = (& fdesetup status) -join "`n"
        if ($fileVaultStatus -match "FileVault is On.") {
            return $true
        } else {
            return $false
        }
    } catch {
        Write-Error "Fehler beim Abrufen des FileVault-Status: $($_.Exception.Message)"
        return $false
    }
}


#Endregion MacOS File Vault


#Region URLs

# !Ex
# 	Join-URLs 'https://eu.ninjarmm.com/', 'oauth/token/'
# » 
#	https://eu.ninjarmm.com/oauth/token
Function Join-URLs() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[String[]]$URLs,
		[Switch]$AddSlash
	)
	
	$Res = $Null
	ForEach($URL in $URLs) {
		$URL = $URL.Trim()
		If ($Res -eq $Null) {
			$Res = $URL.TrimEnd('/')
		} Else {
			$Res = '{0}/{1}' -f $Res, $URL.TrimEnd('/')
		}
	}
	
	If ($AddSlash) {
		Return $Res + '/'
	} Else {
		Return $Res
	}
}

#Endegion URLs


#Region NinjaOne

# Authentifiziert sich am NinjaOne REST API
# Liefert den AuthHeader für folgeaufrufe zurück
# -Dbg zeigt Details zum Auth-Token an
# 
# !M
# https://app.NinjaOne.com/apidocs-beta/authorization/overview
# 250415
Function Auth-NinjaOne() {
	Param(
		[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]

		# Zugriffs-Scopes
		# Wenn nicht definiert, werden alle gewährt
		#  @('monitoring', 'management', 'control')
		# !Ex https://github.com/homotechsual/NinjaOne/blob/bfbc4832473fef510fe11b5a759f2d91f529b999/Source/Public/PSModule/Connect/Connect-NinjaOne.ps1#L142
		[ValidateSet('monitoring', 'management', 'control')]
		[String[]]$Scopes,
		[Switch]$Dbg
	)
	
	If ($Dbg) {
		Write-Host 'Auth-NinjaOne()' -Fore Gray
	}
	
	## Init
	# Allenfalls alle Scopes definieren
	If ($Scopes -eq $Null) {
		$Scopes = @('monitoring', 'management', 'control')
	}
	
	
	# Sie müssen alle kleingeschrieben sein
	$Scopes = $Scopes | % { $_.ToLower() }
	# Write-Host $Scopes | Out-String
	
	
	## Config
	$AuthURL = Join-URLs @($Script:NinjaOneBaseURL, 'oauth/token/')

	$Body = @{
		grant_type = 'client_credentials'
		client_id = $Script:NinjaOneClientID
		client_secret = $Script:NinjaOneClientSecret
		redirect_uri = 'https://localhost'
		scope = ($Scopes -join ' ')
	}
	
	
	## Prepare

	$AuthHeaders = New-Object 'System.Collections.Generic.Dictionary[[String],[String]]'
	# Muss kleingeschrieben sein!
	$AuthHeaders.Add('Accept', 'application/json')
	$AuthHeaders.Add('Content-Type', 'application/x-www-form-urlencoded')
	
	
	## Main

	$AuthToken = Invoke-RestMethod -Uri $AuthURL -Method POST `
												-Headers $AuthHeaders `
												-Body $body
	# Return $AuthToken

	If ($Dbg) {
		Write-Host ($AuthToken | Out-String)
	}

	$AccessToken = $AuthToken | Select-Object -ExpandProperty 'access_token' -EA 0

	$AuthHeader = New-Object 'System.Collections.Generic.Dictionary[[String],[String]]'
	$AuthHeader.Add('accept', 'application/json')
	$AuthHeader.Add('Authorization', "Bearer $AccessToken")

	Return $AuthHeader
}


# Sucht das NinjaOne-Objekt für ein Gerät
Function Get-NinjaOne-Device-by-SystemName() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[Object]$AuthHeader,
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=1)]
		[String]$SystemName,
		[Switch]$Dbg
	)
	
	# Config
	$Limit = 10
	
	# https://app.NinjaOne.com/apidocs-beta/core-resources/operations/search
	$SearchDevicesURL = 'v2/devices/search'

	$ApiURL = Join-URLs @($Script:NinjaOneBaseURLAPI, $SearchDevicesURL)
	
	$QryURL = Join-URLs @($ApiURL, `
								('?q={0}&limit={1}' -f [URI]::EscapeDataString($SystemName), $Limit))
	
	If ($Dbg) { Write-Host "QryURL: `n$QryURL" }
	
	
	# Die Geräte suchen
	$Devices = Invoke-RestMethod -Uri $QryURL -Method GET -Headers $AuthHeader
	# Return $Devices

	# Das Gerät filtern
	$MyDevice = $Devices.Devices | ? systemName -eq $SystemName
	# $ID = $MyDevice.id
	# Write-Host "ID: $ID"
	Return $MyDevice
}


# Sucht von einem NinjaOne Geräte-Objekt
# alle Felder, die es hat
Function Get-NinjaOne-Device-Fields() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[Object]$AuthHeader,
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=1)]
		[Int]$DeviceID,
		[Switch]$Dbg
	)
	
	## Config

	# https://app.NinjaOne.com/apidocs-beta/core-resources/operations/search
	$GetURL = "v2/device/$DeviceID/custom-fields"

	$ApiURL = Join-URLs @($Script:NinjaOneBaseURLAPI, $GetURL)
	If ($Dbg) { Write-Host "ApiURL: `n$ApiURL" }

	# Die Fields auslesen
	$Fields = Invoke-RestMethod -Uri $ApiURL -Method GET -Headers $AuthHeader
	Return $Fields
}


# Setzt den Wert eines Custom Field
Function Set-NinjaOne-Device-Field() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[Object]$AuthHeader,
	
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=1)]
		[Int]$DeviceID,

		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName)]
		[String]$FieldName = 'ssfmahandynr',

		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName)]
		[String]$FieldValue,
		[Switch]$Dbg
	)
	
	## Config

	# https://app.NinjaOne.com/apidocs/?links.active=core#/devices/getNodeCustomFields
	$GetURL = "v2/device/$DeviceID/custom-fields"

	$ApiURL = Join-URLs @($Script:NinjaOneBaseURLAPI, $GetURL)
	If ($Dbg) { Write-Host "ApiURL: `n$ApiURL" }
	
	$BodyProps = @{}
	$BodyProps.Add($FieldName, $FieldValue)
	# Return $BodyProps
	# Return (ConvertTo-Json -InputObject $BodyProps -Depth 100)

	# Die Fields Setzen
	$Res = Invoke-RestMethod -Uri $ApiURL -Method Patch `
									-Headers $AuthHeader `
									-ContentType 'application/json' `
									-Body (ConvertTo-Json -InputObject $BodyProps -Depth 100)

	Return $Res
}

#Endregion NinjaOne


#Region UI

# Liefert $True, wenn Ja gewählt wurde
Function Get-JaNein($Frage) {
	$Frage = $Frage.TrimEnd('?')
	
	do {
		$response = Read-Host "$($Frage)? (j/n)"
	} while ($response -notmatch '^(j|n)$')

	Return $response -eq 'j'
}

#Endregion UI


#Region Netzwerk

# Liefert die erste IPv4 Adresse für eine Domäne
# !Ex, liefert: [System.Net.IPAddress]
# $oIPv4 = Resolve-Domain-get-IPv4 -Domain 'akros.ch'
# $oIPv4.IPAddressToString
Function Resolve-Domain-get-IPv4($Domain) {
	# Punkt am Ende erzwingen
	$Domain = ('{0}.' -f $Domain.Trim().TrimEnd('.'))

	Try {
		$oHost = [System.Net.Dns]::GetHostEntry($Domain)
		# Erste IPv4
		$oFirstIp = $oHost.AddressList | ? { $_.AddressFamily -eq 'InterNetwork' } | Select-Object -First 1
		
		Return $oFirstIp
	} Catch {
		Return $Null
	}
}


# Prüft, ob die Domäne mit einer LAN IP-Adresse aufgelöst wird
Function Is-Computer-in-Domain-LAN($Domain) {
	$oIPv4 = Resolve-Domain-get-IPv4 -Domain $Domain
	If (-not $oIPv4) { Return $False }
	
	$FirstOctett = $oIPv4.IPAddressToString.Split('.')[0]
	# Sind wir im 10.x Netzwerk?
	Return ($FirstOctett -eq 10)
}

#Endregion Netzwerk