#!/usr/local/bin/pwsh
# !9^9 Nur UTF8 für MacOS, KEIN BOM

# Dient dem Schutz der NinjaOne Custom Fiels, 
# so dass Schädlinge diese nicht verändern oder lesen können
#
# Erhält von der NinjaOne Automatisierung die NinjaOne Rest API Hashes
#
# Liest dann via Rest API die Daten aus dem Custom Field ssfClientTasksData
# und überträgt die Daten in die richtigen Custom Fields


# 001, 250616, Tom


## Test-Daten
#  » In das Custom Field 'SSF Client Tasks Data' eines NinjaOne Computers einfügen
# 
# 
# 250616 1707 SaveFileVaultRecoveryKey
# V001
# RecoveryKey: x1
#
# 250617 170111 SaveFileVaultRecoveryKey
# V001
# RecoveryKey: x2
#
# 250617 1740 Tester
# V001
# Data




[CmdletBinding()]
Param (
	[Parameter(Mandatory)]
	# zB 'Gig.......................M'
	[String]$NinjaOneClientID,
	
	[Parameter(Mandatory)]
	# zB 'aB..................................................1w'	
	[String]$NinjaOneClientSecret,
	
	# Soll nur der $AuthHeader zurückgegeben werden?
	[Switch]$JustGetAuthHeader,
	# Debug?
	[Switch]$Dbg
)



### Init
$ScriptDir = [IO.Path]::GetDirectoryName($MyInvocation.MyCommand.Path)

### Config
$FieldName_ssfClientTasksData = 'ssfClientTasksData'

$NinjaOneBaseURL = 'https://eu.ninjarmm.com/'
$NinjaOneBaseURLAPI  = "$($NinjaOneBaseURL.TrimEnd('/'))/api/"

$SetNinjaOneDeviceFieldSecure_ps1 = 'Set-NinjaOne-Device-Field-Secure.ps1'



### Funcs

Function Print-HostNames {
    Write-Host "  ComputerName   : $(scutil --get ComputerName)"
    Write-Host "  HostName       : $(scutil --get HostName)"

	$LocalHostName = "$(scutil --get LocalHostName)"
    Write-Host "  LocalHostName  : $LocalHostName"
    Write-Host "  > Effektiv     : $($LocalHostName).local"
}


# Liefert den effektiven Hostname, also mit .local
Function Get-Hostname-Local() {
	$LocalHostName = "$(scutil --get LocalHostName)"
	Return "$($LocalHostName).local"
}


# Gibt die automatischen Variablen von NinjaOne aus
# NINJA_AGENT_NODE_ID ist die DeviceID
# 
# !Ex
# 	NINJA_AGENT_MACHINE_ID: V5;C69HWWJ1Q1;84:2F:57:3B:2B:41
# 	NINJA_AGENT_NODE_ID: 167
# 	NINJA_AGENT_VERSION_INSTALLED: 8.0.3261
# 	NINJA_COMPANY_NAME: Noser Engineering AG
# 	NINJA_DATA_PATH: /Applications/NinjaRMMAgent/programdata
# 	NINJA_EXECUTING_PATH: /Applications/NinjaRMMAgent/programfiles
# 	NINJA_LOCATION_ID: 20
# 	NINJA_LOCATION_NAME: (Kein Standort)
# 	NINJA_ORGANIZATION_ID: 2
# 	NINJA_ORGANIZATION_NAME: AKROS AG
# 	NINJA_PATCHER_VERSION_INSTALLED: 8.0.3261
# 	NINJA_SECURE_CHAIN_TOKEN: d61c462d-182b-4df7-9e4c-603e287c1f15
Function Print-NinjaOne-AutomaticVars {
	Write-Host "`nAutomatische Variablen von NinjaOne"
	# Alle automatischen NinjaOne Variablen ausgeben
	Get-ChildItem Env: | Where-Object { $_.Name -like 'NINJA*' } | ForEach-Object { Write-Host "$($_.Name): $($_.Value)" }	
}


# !Ex
# 	Join-URLs 'https://eu.ninjarmm.com/', 'oauth/token/'
# » 
#	https://eu.ninjarmm.com/oauth/token
Function Join-URLs() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[String[]]$URLs,
		[Switch]$AddSlash
	)
	
	$Res = $Null
	ForEach($URL in $URLs) {
		$URL = $URL.Trim()
		If ($Res -eq $Null) {
			$Res = $URL.TrimEnd('/')
		} Else {
			$Res = '{0}/{1}' -f $Res, $URL.TrimEnd('/')
		}
	}
	
	If ($AddSlash) {
		Return $Res + '/'
	} Else {
		Return $Res
	}
}


# Authentifiziert sich am NinjaOne REST API
# Liefert den AuthHeader für folgeaufrufe zurück
# -Dbg zeigt Details zum Auth-Token an
# 
# !M
# https://app.NinjaOne.com/apidocs-beta/authorization/overview
# 250415
Function Auth-NinjaOne() {
	Param(
		[Parameter(ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]

		# Zugriffs-Scopes
		# Wenn nicht definiert, werden alle gewährt
		#  @('monitoring', 'management', 'control')
		# !Ex https://github.com/homotechsual/NinjaOne/blob/bfbc4832473fef510fe11b5a759f2d91f529b999/Source/Public/PSModule/Connect/Connect-NinjaOne.ps1#L142
		[ValidateSet('monitoring', 'management', 'control')]
		[String[]]$Scopes,
		[Switch]$Dbg
	)
	
	
	## Init
	# Allenfalls alle Scopes definieren
	If ($Scopes -eq $Null) {
		$Scopes = @('monitoring', 'management', 'control')
	}
	
	
	# Sie müssen alle kleingeschrieben sein
	$Scopes = $Scopes | % { $_.ToLower() }
	# Write-Host $Scopes | Out-String
	
	
	## Config
	$AuthURL = Join-URLs @($Script:NinjaOneBaseURL, 'oauth/token/')

	$Body = @{
		grant_type = 'client_credentials'
		client_id = $Script:NinjaOneClientID
		client_secret = $Script:NinjaOneClientSecret
		redirect_uri = 'https://localhost'
		scope = ($Scopes -join ' ')
	}
	
	
	## Prepare

	$AuthHeaders = New-Object 'System.Collections.Generic.Dictionary[[String],[String]]'
	# Muss kleingeschrieben sein!
	$AuthHeaders.Add('Accept', 'application/json')
	$AuthHeaders.Add('Content-Type', 'application/x-www-form-urlencoded')
	
	
	## Main

	$AuthToken = Invoke-RestMethod -Uri $AuthURL -Method POST `
												-Headers $AuthHeaders `
												-Body $body
	# Return $AuthToken

	If ($Dbg) {
		Write-Host ($AuthToken | Out-String)
	}

	$AccessToken = $AuthToken | Select-Object -ExpandProperty 'access_token' -EA 0

	$AuthHeader = New-Object 'System.Collections.Generic.Dictionary[[String],[String]]'
	$AuthHeader.Add('accept', 'application/json')
	$AuthHeader.Add('Authorization', "Bearer $AccessToken")

	Return $AuthHeader
}


# Sucht das NinjaOne-Objekt für ein Gerät
Function Get-NinjaOne-Device-by-SystemName() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[Object]$AuthHeader,
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=1)]
		[String]$SystemName,
		[Switch]$Dbg
	)
	
	# Config
	$Limit = 10
	
	# https://app.NinjaOne.com/apidocs-beta/core-resources/operations/search
	$SearchDevicesURL = 'v2/devices/search'

	$ApiURL = Join-URLs @($Script:NinjaOneBaseURLAPI, $SearchDevicesURL)
	
	$QryURL = Join-URLs @($ApiURL, `
								('?q={0}&limit={1}' -f [URI]::EscapeDataString($SystemName), $Limit))
	
	If ($Dbg) { Write-Host "QryURL: `n$QryURL" }
	
	
	# Die Geräte suchen
	$Devices = Invoke-RestMethod -Uri $QryURL -Method GET -Headers $AuthHeader
	# Return $Devices

	# Das Gerät filtern
	$MyDevice = $Devices.Devices | ? systemName -eq $SystemName
	# $ID = $MyDevice.id
	# Write-Host "ID: $ID"
	Return $MyDevice
}


# Sucht von einem NinjaOne Geräte-Objekt
# alle Felder, die es hat
Function Get-NinjaOne-Device-Fields() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[Object]$AuthHeader,
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=1)]
		[Int]$DeviceID,
		[Switch]$Dbg
	)
	
	## Config

	# https://app.NinjaOne.com/apidocs-beta/core-resources/operations/search
	$GetURL = "v2/device/$DeviceID/custom-fields"

	$ApiURL = Join-URLs @($Script:NinjaOneBaseURLAPI, $GetURL)
	If ($Dbg) { Write-Host "ApiURL: `n$ApiURL" }

	# Die Fields auslesen
	$Fields = Invoke-RestMethod -Uri $ApiURL -Method GET -Headers $AuthHeader
	Return $Fields
}


# Setzt den Wert eines Custom Field
Function Set-NinjaOne-Device-Field() {
	Param(
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=0)]
		[Object]$AuthHeader,
	
		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName, Position=1)]
		[Int]$DeviceID,

		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName)]
		[String]$FieldName_ssfClientTasksData = 'ssfmahandynr',

		[Parameter(Mandatory, ValueFromPipeline, ValueFromPipelineByPropertyName)]
		[String]$FieldValue,
		[Switch]$Dbg
	)
	
	## Config

	# https://app.NinjaOne.com/apidocs/?links.active=core#/devices/getNodeCustomFields
	$GetURL = "v2/device/$DeviceID/custom-fields"

	$ApiURL = Join-URLs @($Script:NinjaOneBaseURLAPI, $GetURL)
	If ($Dbg) { Write-Host "ApiURL: `n$ApiURL" }
	
	$BodyProps = @{}
	$BodyProps.Add($FieldName_ssfClientTasksData, $FieldValue)
	# Return $BodyProps
	# Return (ConvertTo-Json -InputObject $BodyProps -Depth 100)

	# Die Fields Setzen
	$Res = Invoke-RestMethod -Uri $ApiURL -Method Patch `
									-Headers $AuthHeader `
									-ContentType 'application/json' `
									-Body (ConvertTo-Json -InputObject $BodyProps -Depth 100)

	Return $Res
}



## Parser für den Inhalt von ssfClientTasksData


# Parst von *einem* Block die Zeilen und erzeugt ein PSCustomObject
Function Parse-Block-Lines() {
    Param (
        [String[]]$BlockLines,
		[Switch]$Dbg
    )

	# Sicherstellen, dass wir wirklich Zeilen haben
	$BlockLines = $BlockLines -split "`n" | ? { -Not [String]::IsNullOrWhiteSpace( $_ ) }

	If ($Dbg) {
		Write-Host '2jbdjK Parse-Block-Lines(), $BlockLines:' -Fore Magenta
		Write-Host ($BlockLines | Out-String)
	}

    ## Header-Zeile analysieren
    $headerLine = $BlockLines[0]
	If ($Dbg) {
		Write-Host 'Parse-Block-Lines(), $headerLine:' -Fore Magenta
		Write-Host ($headerLine | Out-String)
	}

    $headerPattern = '^(?<Timestamp>\d{6}\s+\d{4,6})\s+(?<Key>\S+)\s*(?<KeyAddon>.*)$'

    if ($headerLine -notmatch $headerPattern) {
        Write-Host "Ungültige Headerzeile: $headerLine" -Fore Red
		Return $Null
    }

    $timestampRaw = $matches['Timestamp']
    $key          = $matches['Key']
    $KeyAddon      = $matches['KeyAddon']

    # Konvertierung in [datetime]
    $datetime = $null
    if ($timestampRaw -match '^(?<date>\d{6})\s+(?<time>\d{4,6})$') {
        $datePart = $matches['date']
        $timePart = $matches['time']

        $yy = [int]$datePart.Substring(0, 2)
        $mm = [int]$datePart.Substring(2, 2)
        $dd = [int]$datePart.Substring(4, 2)

        $hh = [int]$timePart.Substring(0, 2)
        $mi = [int]$timePart.Substring(2, 2)
        $ss = if ($timePart.Length -ge 6) { [int]$timePart.Substring(4, 2) } else { 0 }

        $year = if ($yy -lt 100) { 2000 + $yy } else { $yy }

        $datetime = Get-Date -Year $year -Month $mm -Day $dd -Hour $hh -Minute $mi -Second $ss
    }

    # Comment
	$Comment = ''
	# Vxxx
	$Version = $null
    # Key-Value Werte
	$KeyValueData = @{}
    $AdditionalLines = @()

    ForEach ($Line in $BlockLines[1..($BlockLines.Count - 1)]) {
		$Line = $Line.Trim()
        If ($Line -match '^V\d{3}$' -and -not $Version) {
            $Version = $Line
        }
        Elseif ($Line -match '^\s*(?<PKey>[^:]+)\s*:\s*(?<PValue>.+)$') {
            $keyName = $matches['PKey'].Trim()
            $value = $matches['PValue'].Trim()
            $KeyValueData[$keyName] = $value
        }
        Elseif ($Line -match '^#(?<Comment>.*)') {
			$Comment = $Matches['Comment'].Trim()
		}		
        Else {
            $AdditionalLines += $Line
        }
    }

    # Objekt erstellen und Root-Eigenschaften zusammenbauen
    $baseObject = [Ordered]@{
        Processed   = $False
        Key         = $key
        KeyAddon    = $KeyAddon
		Comment		= $Comment
        Version     = $Version
        Timestamp   = $datetime
    }

    # Properties hinzufügen
    ForEach ($kvp in $KeyValueData.GetEnumerator()) {
        $baseObject[$kvp.Key] = $kvp.Value
    }
	
	$baseObject['AdditionalLines'] = $AdditionalLines
	$baseObject['OriLines'] = $BlockLines

    return [PSCustomObject]$baseObject
}



# Parst alle Blöcke und erzeugt für jeden je ein PSCustomObject
# !Ex
# 	$Blöcke = Get-TextBlocks -Text BlockLines
# 	$oBlöcke = Parse-ClientTasksData -Blocks $Blöcke
Function Parse-ClientTasksData {
    Param (
        [String[]]$Text,
		[Switch]$Dbg
    )

	# String in Linien Splitten und leere Zeilen entfernen
	$Lines = $Text -split "`n" | ? { -Not [String]::IsNullOrWhiteSpace( $_ ) }
	
	If ($Dbg) {
		Write-Host 'Parse-ClientTasksData(), $Lines:' -Fore Magenta
		Write-Host ($Lines | Out-String)
	}

	$Blöcke = @(Get-TextBlocks -Lines $Lines -Dbg:$Dbg)

	If ($Dbg) {
		Write-Host 'Parse-ClientTasksData(), $Blöcke:' -Fore Magenta
		For($idx = 0; $idx -lt $Blöcke.Count; $idx++) {
			Write-Host "Block: $idx"
			Write-Host ($Blöcke[$idx] | Out-String)
		}
	}

	$Res = @()
    ForEach ($BlockLines in $Blöcke) {
		$Res += Parse-Block-Lines -BlockLines $BlockLines -Dbg:$Dbg
	}
	Return $Res
}



# Trennt die Zeilen in Blöcke, e.g.:
# 	$Text = @'
# 		250616 1246 LocalUserPWChanged lk flaksjf lkas df
# 		V002
# 		Prop1: x
# 		Prop2: y
# 		Prop3: y
# 		Testzeile 1
# 		Testzeile 2
#	'@
Function Get-TextBlocks {
    param (
        [String[]]$Lines,
		[Switch]$Dbg
    )
	
	## Init
	# Sicherstellen, dass wir wirklich Zeilen haben
	$Lines = $Lines -split "`n" | ? { -Not [String]::IsNullOrWhiteSpace( $_ ) }

	If ($Dbg) {
		Write-Host 'Get-TextBlocks(), $Lines:' -Fore Magenta
		Write-Host ($Lines | Out-String)
	}

	## Config
    # Regex für den Start eines Blocks
    $startPattern = '^\d{6}\s+\d{4,6}\s+\S.+$'

	## Main
    $blocks = @()
    $currentBlock = @()

    foreach ($Line in $Lines -split "`n") {
        if ($Line.Trim() -eq '') {
            continue # Leerzeilen vollständig ignorieren
        }

        if ($Line -match $startPattern) {
            if ($currentBlock.Count -gt 0) {
                $blocks += ,@($currentBlock)
                $currentBlock = @()
            }
        }

        $currentBlock += $Line
    }

    # Letzten Block hinzufügen, falls vorhanden
    if ($currentBlock.Count -gt 0) {
        $blocks += ,@($currentBlock)
    }

	If ($Dbg) {
		Write-Host 'Get-TextBlocks(), $blocks:' -Fore Magenta
		For($idx = 0; $idx -lt $blocks.Count; $idx++) {
			Write-Host "Block: $idx"
			Write-Host ($blocks[$idx] | Out-String)
		}
	}

    Return $blocks
}


# Konvertiert nicht verarbeitete Blöcke wieder zu Zeilen
# Um sie wieder ins ssfClientTasksData zu schreiben, wenn wir Fehler haben
# In -Comment kann der Fehler mitgegeben werden
Function Get-Block-unprocessed-AsLines {
    Param (
        [PSCustomObject[]]$Blocks,
        [String]$Comment
    )
	
	# Neue Liste erzeugen
	$Res = @()

	ForEach ($Block in $Blocks | ? Processed -eq $False) {
		$Idx = -1
		ForEach ($Line in $Block.OriLines) {
			$Idx++
			$Res += [String]$Line
			If ($Idx -eq 0) {
				$Res += ('# {0}' -f $Comment.TrimStart('#').Trim())
			}
		}
		# Leere Zeile als Trenner
		$Res += ''
	}
	Return $Res -join "`n"
}



## Prozessor für die einzelnen ssfClientTasksData Elemente

# Speichert den FileVault Recovery Key
# Block-Struktur:
# 	250616 094302 SaveFileVaultRecoveryKey
#	V001
#	RecoveryKey: x
# 	
Function Save-FileVaultRecoveryKey() {
	Param (
        [PSCustomObject]$oBlock,
		[Object]$AuthHeader,
		[String]$DeviceID,
		[Switch]$Dbg
    )
	
	Switch($oBlock.Version) {
		'V001' {
			If ($Dbg) { 
				Write-Host 'Save-FileVaultRecoveryKey(): V001' -Fore Magenta
				Write-Host ('FieldName : {0}' -f 'ssfMacOsFileVaultKey')
				Write-Host ('FieldValue: {0}' -f $oBlock.RecoveryKey)
			}
			# Zeile ergänzen
			
			$RecoveryData = ("{0}:`n{1}" -f (Get-Date).ToString('yyMMdd HHmm'), $oBlock.RecoveryKey)
			
			& $SetNinjaOneDeviceFieldSecure_ps1 -DeviceIDs $DeviceID `
			-FieldName 'ssfMacOsFileVaultKey' `
			-FieldValue $RecoveryData `
			-Action 'AddNewLine' `
			-AuthHeader $AuthHeader
		}
		
		Default {
			Write-Host "`n2jbb5z Fehler in: Save-FileVaultRecoveryKey()" -Fore Red
			Write-Host ('Version unbekannt: {0}' -f $oBlock.Version)
		}
	}
	
}


# Verarbeitet alle Elemente im Feld ssfClientTasksData
Function Transfer-ssfClientTasksData() {
	Param (
        [PSCustomObject[]]$oBlöcke,
		[Object]$AuthHeader,
		[String]$DeviceID,
		[Switch]$Dbg
    )

	ForEach($oBlock in $oBlöcke) {
		If ($Dbg) {
			Write-Host 'Block:' -Fore Magenta
			Write-Host $oBlock
		}
		Switch ($oBlock.Key) {
			# Den FileFault RecoveryKey
			'SaveFileVaultRecoveryKey' {
				If ($Dbg) {
					Write-Host 'Verarbeite: SaveFileVaultRecoveryKey' -Fore Magenta
				}
				Write-Host "`nSpeichere den FileVault Recovery Key"
				Save-FileVaultRecoveryKey -oBlock $oBlock `
										  -AuthHeader $AuthHeader `
										  -DeviceID $DeviceID `
										  -Dbg:$Dbg
				$oBlock.Processed = $True				
			}
			
			Default {
				If ($Dbg) {
					Write-Host "Switch-Fehler!: $($oBlock.Key)" -Fore Red
				}
				Write-Host "`n2jbb5y Fehler in: Transfer-ssfClientTasksData()" -Fore Red
				Write-Host ('Unbekannter Key: {0}' -f $oBlock.Key)
			}
		}
	}
}



### Prepare

## Debug
# Print-HostNames
# Print-NinjaOne-AutomaticVars

# Scripts suchen
$SetNinjaOneDeviceFieldSecure_ps1 = Get-ChildItem -LiteralPath $ScriptDir | ? Name -eq $SetNinjaOneDeviceFieldSecure_ps1 | Select -ExpandProperty FullName



## Authentifizieren
$AuthHeader = Auth-NinjaOne
If ($JustGetAuthHeader) { Return $AuthHeader }


## Die NiniaOneDeviceID bestimmen
$HasNinjaoneDeviceID = -not [String]::IsNullOrWhiteSpace($Env:NINJA_AGENT_NODE_ID)

If($HasNinjaoneDeviceID) {
	$NinjaoneDeviceID = $Env:NINJA_AGENT_NODE_ID
} Else {
	# Die Device-ID aufgrund des Hostnamens suchen
	$ThisDevice = Get-NinjaOne-Device-by-SystemName `
						-AuthHeader $AuthHeader `
						-SystemName (Get-Hostname-Local)
	
	# Debug
	# Write-Host "`n$ThisDevice Objekt:"
	# $ThisDevice | Out-String | % { Write-Host $_ }
	
	# $ThisDevice.ID
	# $ThisDevice.systemName	
	$NinjaoneDeviceID = $ThisDevice.ID
}


# Haben wir die DeviceID?
If ([String]::IsNullOrWhiteSpace($NinjaoneDeviceID)) {
	Write-Host "`n2jbb1V Fehler: Device-ID nicht gefunden!" -Fore Red
	Write-Host 'Abbruch'
	Break Script
}


## Vom Gerät Die Custom Fields lesen
$Fields = Get-NinjaOne-Device-Fields -AuthHeader $AuthHeader `
									-DeviceID $NinjaoneDeviceID `
									-Dbg:$Dbg

# Alle gefundenen Felder ausgeben?
If ($Dbg) {
	Write-Host ("`nDevice ID: {0}" -f $NinjaoneDeviceID) -Fore Green
	($Fields | FL | Out-String) -split "`n" | `
		? { -Not [String]::IsNullOrWhiteSpace($_) } | `
		% { Write-Host "  $_" }
}




### Main

# Haben wir Daten in ssfClientTasksData?
$ssfClientTasksData = $Fields.$FieldName_ssfClientTasksData

If ([String]::IsNullOrWhiteSpace($ssfClientTasksData)) {
	Write-Host "`nKeine Daten in: $FieldName_ssfClientTasksData"
	Write-Host 'Fertig'
	Break Script	
}
If ($Dbg) { 
	Write-Host '$ssfClientTasksData:' -Fore Magenta
	Write-Host $ssfClientTasksData
}


# Die Elemente in ssfClientTasksData Parsen
If ($Dbg) { Write-Host 'Parse Blöcke' -Fore Gray }
$oBlöcke = Parse-ClientTasksData -Text $ssfClientTasksData -Dbg:$Dbg
# Return $oBlöcke
If ($Dbg) { 
	Write-Host '$oBlöcke:' -Fore Magenta
	Write-Host ($oBlöcke | Out-String)
}

# Die Elemente verarbeiten
If ($Dbg) { Write-Host 'Starte Transfer' -Fore Gray }
Transfer-ssfClientTasksData -oBlöcke $oBlöcke `
							-AuthHeader $AuthHeader `
							-DeviceID $NinjaoneDeviceID
	

If ($Dbg) { 
	Write-Host '$oBlöcke nach der Verarbeitung:' -Fore Magenta
	Write-Host ($oBlöcke | Out-String)
}



## Die ssfClientTasksData Daten mit den unverarbeiteten Daten wieder füllen

$oBlöckeUnverarbeitetLines = Get-Block-unprocessed-AsLines `
							-Blocks $oBlöcke `
							-Comment ('{0} Verarbeitungsversuche, failed' -f (Get-Date).ToString('yyMMdd HHmm'))

# Haben wir nicht verarbeitete Elemente?
If ([String]::IsNullOrWhiteSpace($oBlöckeUnverarbeitetLines)) {
	Return
}


# Die nicht verarbeiteten Elemente wieder zurückschreiben
& $SetNinjaOneDeviceFieldSecure_ps1 -DeviceIDs $NinjaoneDeviceID `
	-FieldName $FieldName_ssfClientTasksData `
	-FieldValue $oBlöckeUnverarbeitetLines `
	-Action 'Set' `
	-AuthHeader $AuthHeader

Return 
# Return $oBlöckeUnverarbeitet
# Return $Fields

Break Script

# Haben wir einen Wert?
$WertOri = $Fields.$FieldName_ssfClientTasksData
$WertNeu = ''
Switch ($Action) {
	'Set' {
		$WertNeu = $FieldValue
	}
	'Add' {
		$WertNeu = $WertOri + $FieldValue
	}
	'AddNewLine' {
		$WertNeu = ("{0}`n{1}" -f $WertOri, $FieldValue)
	}
	Default {
		Write-Error "Ungültige Action: $Action"
	}
}

	
# Das Feld setzen
Set-NinjaOne-Device-Field -AuthHeader $AuthHeader -DeviceID $ThisDevice.ID `
						-FieldName $FieldName_ssfClientTasksData -FieldValue $WertNeu -Dbg:$Dbg
