#!/usr/local/bin/pwsh

# !9^9 Nur UTF8 für MacOS, KEIN BOM

# Aktiviert die MacOS Disk-Verschlüsselung (FileVault)
# und speichert das Passwort in NinjaOne
# Beim Aufruf ohne Param wird der FileVault Status angezeigt

# 001, 250611, Tom


# !Dbg:
#	Mit -TestAddNewRec_ssfClientTasksData
#	Wird nur ein neuer Dummy-Datensatz ins NinjaOne Custom Field geschrieben

[CmdletBinding()]
Param(
	[Switch]$TestAddNewRec_ssfClientTasksData
)



### Init
$ScriptDir = [IO.Path]::GetDirectoryName($MyInvocation.MyCommand.Path)
$ThisScriptPath = $MyInvocation.MyCommand.Definition


### Root erzwingen
# 250620 1850
If ((id -u) -ne 0) {
    Write-Host 'Starte das Script als root'

    # Argumente aus den BoundParameters zusammensetzen
    $paramList = @()
    foreach ($kvp in $PSBoundParameters.GetEnumerator()) {
        if ($kvp.Value -is [switch] -and $kvp.Value.IsPresent) {
            $paramList += "-$($kvp.Key)"
        } else {
            $paramList += "-$($kvp.Key) `"$($kvp.Value)`""
        }
    }

    $pwsh = (Get-Command pwsh).Source

    if ($paramList.Count -gt 0) {
        sudo $pwsh $ThisScriptPath @paramList
    } else {
        sudo $pwsh $ThisScriptPath
    }

    exit
}


# Ab hier läuft das Script als root
Write-Host "`nScript läuft jetzt als root`n`n" -Fore Gray



### Init

# Ermitteln des aktuell angemeldeten Benutzers auf macOS
# Die Ausgabe von ls -l /dev/console ist:
#	"crw-rw-rw- 1 loggedinuser wheel 3, 0 May 20 10:00 /dev/console"
# Zeigen root:
# 	$LoggedInUser = $(/bin/zsh -c "ls -l /dev/console | awk '{ print `$3 }'")
# 	$LoggedInUser = [System.Environment]::UserName
#
# Zeigen den User, der am Desktop angemeldet ist:
# echo $SUDO_USER
# logname
$LoggedInUser = $(logname)



### Config
$TomLibMacOS_ps1 = 'TomLib-MacOS.ps1'

# NinjaOne Custom Field Name
$FieldName_ssfClientTasksData = 'ssfClientTasksData'

# Pfad zur recovery.plist Datei
# WICHTIG
# - Der Recovery-Key FileVault Recovery Key gehört zum User, der ihn erzeugt hat!
# - FileVault kann nur mit Root-Rechten aktiviert werden
# Deshalb dieser Pfad:
$recoveryPlistPath = "/private/var/root/$loggedInUser-recovery.plist" 
# Write-Host $recoveryPlistPath

$SetLoginScript_sh = '/Library/Scripts/SSF/NinjaOne/Set-Login-Script.sh'
$NinjaCliPath = '/Applications/NinjaRMMAgent/programdata/ninjarmm-cli'



### Funcs


# Liefert den Record für das Custom Field:
#
# 250616 1707 SaveFileVaultRecoveryKey
# V001
# RecoveryKey: x1
Function New-ssfClientTasksData-Record-SaveFileVaultRecoveryKey() {
	Param (
		[Parameter(Mandatory)]
		[String]$FileVaultRecoveryKey
	)
	
	# Config
	$RecKey = 'SaveFileVaultRecoveryKey'

	$Rec = @"
$((Get-Date).ToString('yyMMdd HHmm')) $RecKey
V001
RecoveryKey: $FileVaultRecoveryKey
"@	
	
	Return $Rec
}


# Erzeugt im NinjaOne Custom Field ssfClientTasksData einen Record
Function Create-ssfClientTasksData-Rec() {
	Param(
		[Parameter(Mandatory)]
		[String]$FileVaultRecoveryKey
	)

	# Den Record erzeugen
	$RecData = New-ssfClientTasksData-Record-SaveFileVaultRecoveryKey `
				-FileVaultRecoveryKey $FileVaultRecoveryKey
	
	# Dbg
	# Write-Host "`nRecData:"
	# Write-Host ($RecData | Out-String)
	
	# Den aktuellen Wert lesen
	# !9 Liefert ein String[]
	$WertOri = & "$Script:NinjaCliPath" get "$Script:FieldName_ssfClientTasksData"
	$WertOri = $WertOri -Join "`n"
	
	
	# Den Rec ergänzen
	$WertNeu = ("{0}`n`n{1}" -f $WertOri, $RecData)
	# Dbg
	# Write-Host "`nWertNeu:"
	# Write-Host ($WertNeu | Out-String)
	

	# Pfad zum NinjaRMM-Agenten
	& "$Script:NinjaCliPath" set "$Script:FieldName_ssfClientTasksData" "$WertNeu"
	
}


# Prüft, dass das NinjaOne Custom Field den FileVaultRecoveryKey hat
# Liefert $True, wenn der Key vorhanden ist
Function Assert-ssfClientTasksData-has-FileVaultRecoveryKey() {
	Param(
		[Parameter(Mandatory)]
		# eg EDK4-JR4M-LRW2-GHQQ-47PL-39EQ
		[String]$FileVaultRecoveryKey,
		[Int]$RetryMs = 8000,
		[Int]$TimeoutSec = 120
	)

	Write-Host "`nPrüfe, ob NinjaOne den FileVault Recovery Key hat"

	$StartTime = Get-Date
	$FileVaultKeyFound = $False
	Do {
		
		# Den aktuellen Wert lesen
		# !9 Liefert ein String[]
		$WertOri = & "$Script:NinjaCliPath" get "$Script:FieldName_ssfClientTasksData"
		$WertOri = $WertOri -Join "`n"
		
		# Wenn wir einen Wert erhalten haben
		If (-not [string]::IsNullOrWhiteSpace($WertOri)) {
			$FileVaultKeyFound = $WertOri.IndexOf($FileVaultRecoveryKey, [StringComparison]::OrdinalIgnoreCase) -ge 0
		}

		# Wenn der VaultKey fehlt, warten
		If ($FileVaultKeyFound) {
			Write-Host 'OK, gefunden' -Fore Green
		} Else {
			Write-Host ('Noch nicht gefunden, warte {0}ms' -f $RetryMs) -Fore Magenta
			Start-Sleep -Milliseconds $RetryMs
		}
		
		$Elapsed = (Get-Date) - $StartTime
		
	} While (-not $FileVaultKeyFound -and $Elapsed.TotalSeconds -lt $TimeoutSec)
	
	Return $FileVaultKeyFound
}



### Prepare

# Lib laden
$TomLibMacOS_ps1 = Join-Path $ScriptDir $TomLibMacOS_ps1
. $TomLibMacOS_ps1 $PSBoundParameters

# Status bestimmen
$IsFileVaultEnabled = Is-FileVault-Enabled
$HasRecoveryPlistFile = Test-Path -LiteralPath $recoveryPlistPath -PathType Leaf
$IsRecoveryPlistFileMissing = -not $HasRecoveryPlistFile


$ResSummary = [Ordered]@{
	IsFileVaultEnabled =$IsFileVaultEnabled
}

If ($HasRecoveryPlistFile) {
	$ResSummary = @{ RecoveryPlistFile = $recoveryPlistPath }
} Else {
	$ResSummary = @{ RecoveryPlistFile = 'Nicht vorhanden' }
}


### Main


# Nur im Test-Modus?
If ($TestAddNewRec_ssfClientTasksData) {
	Write-Host "`nResultat:" -Fore Magenta
	Write-Host "`n-TestAddNewRec_ssfClientTasksData"
	Write-Host "Setze den Record im Custom Field: $($Script:FieldName_ssfClientTasksData)"
	Create-ssfClientTasksData-Rec -FileVaultRecoveryKey ((Get-Date).ToString('yyMMdd HHmmss'))
	Write-Host 'Done'
	Write-Host "`n"
	Return
}



# Ist FileVault bereits aktiv?
If ($IsFileVaultEnabled -and $IsRecoveryPlistFileMissing) {
	Write-Host "`nResultat:" -Fore Magenta
	Write-Host '- FileVault ist bereits aktiviert' -Fore DarkGreen
	Write-Host '- Der Recovery-Key ist bereits in NinjaOne gesichert' -Fore DarkGreen
	Break Script
}



## FileVault ist noch nicht aktiv

# Wenn noch keine RecoveryPlist erzeugt wurde, wurde FileVault noch nicht aktiviert
If ($IsRecoveryPlistFileMissing) {
	
	# FileVault wurde noch nicht aktiviert
    Write-Host "Aktiviere FileVault für: $LoggedInUser"

	try {
		# Verwende Start-Process mit -Wait, um sicherzustellen, dass der Befehl abgeschlossen ist
		Start-Process -FilePath 'fdesetup' -ArgumentList 'enable', '-user', $LoggedInUser, '-defer', $recoveryPlistPath, '-keychain', '-forceatlogin', '0' -Wait -NoNewWindow -PassThru | Out-Null

		Write-Host 'FileVault-Aktivierung eingeleitet'
		
		# Dieses Script als login Script definieren
		Write-Host "`nAktiviere Login-Script"
		# Parameter für das Bash-Skript
		$Username = $LoggedInUser
		$PwshScriptToExecute = "$ThisScriptPath"
		$Action = 'add'
		# $Mode = 'runonce'
		$Mode = 'persistent'
		
		try {
			# Option 1: Ohne optionalen Modus-Parameter
			& $SetLoginScript_sh $Username $PwshScriptToExecute $Action $Mode

			if ($LASTEXITCODE -eq 0) {
				Write-Host "Bash-Skript erfolgreich ausgeführt."
			} else {
				Write-Error "Bash-Skript wurde mit Exit-Code $LASTEXITCODE beendet."
			}
		} catch {
			Write-Error "Fehler beim Ausführen des Bash-Skripts: $($_.Exception.Message)"
		}		

		Write-Host "`nStarte in 5s das Logoff" -Fore Red
		Write-Host 'Bitte wieder anmelden und so den FileVault aktivieren!' -Fore Magenta
		
		Start-Sleep -Milliseconds 5500
		[void] (Read-Host "Drücke [Enter] zum Fortfahren")
		
		Logoff-MacOS-User -LoggedInUser $LoggedInUser
		
	} catch {
		Write-Error "Fehler bei der FileVault-Aktivierung: $($_.Exception.Message)"
	}
		
	
} Else {

	# FileVault wurde bereits aktiviert,
	# Aber der Recovery-key wurde noch nicht in NinjaOne gesichert
	
    Write-Host 'Übertrage den FileVault Recovery Key zu NinjaOne'

    try {
        # Den RecoveryKey aus der Plist-Datei lesen
        $FileVaultRecoveryKey = (& /usr/libexec/PlistBuddy -c "Print :RecoveryKey" $recoveryPlistPath)
		
		Write-Host $FileVaultRecoveryKey
		
		$ResSummary = @{ RecoveryKey = $FileVaultRecoveryKey }


        if ([string]::IsNullOrEmpty($FileVaultRecoveryKey)) {
            Write-Warning "Der Recovery Key konnte nicht aus $recoveryPlistPath gelesen werden."
        } else {
			Write-Host 'Hat NinjaOne den Recovery Key bereits?'
			$RecoveryKeyTransferOK = Assert-ssfClientTasksData-has-FileVaultRecoveryKey `
										-FileVaultRecoveryKey $FileVaultRecoveryKey `
										-TimeoutSec 10
			
			If ($RecoveryKeyTransferOK) {
				Write-Host 'NinjaOne hat den Recovery Key bereits'
			} Else {
				Write-Host 'Übergebe Recovery Key an NinjaOne'
				
				# Den Record erzeugen
				Create-ssfClientTasksData-Rec -FileVaultRecoveryKey $FileVaultRecoveryKey
				
				
				# Hat NinjaOne den VaultKey?
				$RecoveryKeyTransferOK = Assert-ssfClientTasksData-has-FileVaultRecoveryKey -FileVaultRecoveryKey $FileVaultRecoveryKey
				
				Write-Host "`nfileVaultKeyOK: $RecoveryKeyTransferOK"
			}
			
			
			## Wenn der Recovery-Key übertragen wurde,
			## die PlistBuddy Datei löschen
			$FileExists = Test-Path -LiteralPath $recoveryPlistPath -PathType Leaf
			
			If ($RecoveryKeyTransferOK -and $FileExists) {
				Write-Host ('Lösche Recovery Key Datei: {0}' -f $recoveryPlistPath)
				Remove-Item -Path $recoveryPlistPath -Force
				$ResSummary = @{ RecoveryPlistFileExists = $False }
			} Else {
				$ResSummary = @{ RecoveryPlistFileExists = $FileExists }
			}
			
			$ResSummary = @{ RecoveryKeyTransferOK = $RecoveryKeyTransferOK }
		
			Write-Host 'done.'
			
        }
    } catch {
        Write-Error "Fehler beim Lesen der recovery.plist oder beim Übergeben an NinjaRMM: $($_.Exception.Message)"
    }	
}



## Resultat ausgeben
Write-Host "`nResultat:" -Fore Magenta
Write-Host ([PSCustomObject]$ResSummary | Out-String) -Fore Magenta
Write-Host "`n"
Write-Host "`n"


