﻿# 001, 241227, Tom@jig.ch


If ($PSVersionTable.PSVersion.Major -ne 5) {
	Write-Host "`nDas Script benötigt PowerShell 5" -Fore Red
	Break Script
}


# Setzt alle 17 Varianten von Microsoft um,
# damit der Wechsel aufs neue Outlook hoffentlich nicht gemacht wird

# !Q https://learn.microsoft.com/de-de/microsoft-365-apps/outlook/get-started/control-install?branch=main


# 001, 241218, Tom@jig.ch



## Config
## Init

# Start Elevated
if (!([Security.Principal.WindowsPrincipal][Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")) { 
	Start-Process powershell.exe "-NoProfile -ExecutionPolicy Bypass -File `"$PSCommandPath`"" -Verb RunAs; exit 
}


## Funcs


Function Show-Countdown($TimeoutSec = 1, [Switch]$ShowText, [String]$Msg = 'Es geht automatisch weiter') {
	$Msg = $Msg.Trim()
	$Text = ('{0} in ' -f $Msg)
	$Cnt = 0
	While ($Cnt -lt $TimeoutSec) {
		If ($ShowText) {
			# Write-Host ("`rWeiter in $($TimeoutSec - $Cnt)s") -NoNewLine
			Write-Host ("`r$Text$($TimeoutSec - $Cnt)s   ") -NoNewLine
		}
		Else {
			# Pünktchen zeichen
			Write-Host ('.' * ($TimeoutSec - $Cnt) + ' ' * $TimeoutSec + "`r") -NoNewLine
			Write-Host ('.' * ($TimeoutSec - $Cnt) + ' ' * $TimeoutSec + "`r") -NoNewLine
		}
		Start-Sleep -Seconds 1
		$Cnt++
	}
	Write-Host ("`r" + ' ' * ($Text.Length + 5) + "`n")
}


# Stoppt das Script, allenfalls mit einer Benutzerbestätigung, mit einem Timeout oder einem sofortigen Abbruch
# $WaitOnEnd
# 0    Script sofort beenden
# 1    Script nach einer Benutzerbestätigung beenden
# >1   Script nach einem Timeout mit $WaitOnEnd Sekunden beenden
# 200807
Function Stop-Script($WaitOnEnd) {
	If ($WaitOnEnd -eq $Null) { $WaitOnEnd = $Script:WaitOnEnd }

	Switch ($WaitOnEnd) {
		0 {
			# Nichts zu tun, Script beenden
		}
		1 {
			# Pause
			Log 0 'Press any key to continue …'
			Try { $x = $host.UI.RawUI.ReadKey("NoEcho,IncludeKeyDown") } Catch {}
		}
		Default {
			# Start-Sleep -Seconds $WaitOnEnd
			Show-Countdown $WaitOnEnd -ShowText -Msg 'Script wird beendet'
		}
	}
	Break Script
}



# Beendet das Script, forciert aber, dass der User die Fehlermeldung sieht
Function Stop-Script-Wait($WaitOnEnd) {
	If ($WaitOnEnd -eq $null -or $WaitOnEnd -eq 0) {
		# 10s warten, wenn nichts definiert ist
		Stop-Script 10
	}
 Else {
		Stop-Script $WaitOnEnd
	}
}




function Set-RegistryKey {
	Param (
	[Parameter(Mandatory)]
	# Der vollständige Registry-Pfad (z. B. HKCU:\Software\...)
	[string]$RegistryPath,  

	[Parameter(Mandatory)]
	# Der Name der Eigenschaft, die gesetzt werden soll
	[string]$PropertyName,  

	[Parameter(Mandatory)]
	# Der Wert der Eigenschaft (z. B. "Test" oder 1)
	[object]$PropertyValue,  

	[Parameter()]
	# Der Typ der Eigenschaft (Standard: String)
	[ValidateSet("String", "DWord", "QWord", "Binary", "MultiString", "ExpandString")]
	[string]$PropertyType = "String"  
	)

	Try {
		# Prüfen, ob der Registry-Pfad existiert, und erstellen, falls er fehlt
		if (-not (Test-Path -Path $RegistryPath)) {
			Write-Verbose "Registry path '$RegistryPath' does not exist. Creating it..."
			New-Item -Path $RegistryPath -Force | Out-Null
		}

		# Prüfen, ob die Eigenschaft existiert
		$existingProperty = Get-ItemProperty -Path $RegistryPath -Name $PropertyName -ErrorAction SilentlyContinue

		if (-not $existingProperty) {
			# Eigenschaft erstellen
			Write-Verbose "Property '$PropertyName' does not exist. Creating it..."
			New-ItemProperty -Path $RegistryPath -Name $PropertyName -PropertyType $PropertyType -Value $PropertyValue | Out-Null
		} else {
			# Eigenschaft aktualisieren
			Write-Verbose "Property '$PropertyName' exists. Updating it..."
			Set-ItemProperty -Path $RegistryPath -Name $PropertyName -Value $PropertyValue
		}

		Write-Host "Successfully set registry key: $RegistryPath\$PropertyName = $PropertyValue"
	} catch {
		Write-Error "Failed to set registry key. Error: $_"
	}
}


function Remove-RegistryTarget {
	param (
		[Parameter(Mandatory = $true)]
		# Vollständiger Registry-Pfad inkl. mögliches Property (z. B. HKCU:\Software\...\PropertyName)
		[string]$RegistryPath 
	)

	try {
		# Zerlegen des Pfads in Key und Property (letztes Element könnte eine Property sein)
		$lastBackslashIndex = $RegistryPath.LastIndexOf('\')
		$path = $RegistryPath.Substring(0, $lastBackslashIndex) # Alles vor dem letzten Backslash
		$property = $RegistryPath.Substring($lastBackslashIndex + 1) # Das letzte Element nach dem Backslash

		# Prüfen, ob der Registry-Pfad existiert
		if (Test-Path -Path $path) {
			# Prüfen, ob das letzte Element eine Property ist
			$propertyExists = Get-ItemProperty -Path $path -Name $property -ErrorAction SilentlyContinue
			if ($propertyExists) {
				# Lösche die Property
				Write-Verbose "Deleting property '$property' from '$path'..."
				Remove-ItemProperty -Path $path -Name $property -Force
				Write-Host "Successfully deleted property '$property' from '$path'."
			} else {
				# Wenn keine Property existiert, prüfen, ob der gesamte Pfad gelöscht werden soll
				if (Test-Path -Path $RegistryPath) {
					Write-Verbose "Deleting registry path '$RegistryPath'..."
					Remove-Item -Path $RegistryPath -Recurse -Force
					Write-Host "Successfully deleted registry path '$RegistryPath'."
				} else {
					Write-Warning "The registry path '$RegistryPath' does not exist as a property or key. No action taken."
				}
			}
		} else {
			# Write-Warning "The registry path '$path' does not exist. No action taken."
		}
	} catch {
		Write-Error "Error: $_"
		Write-Error "Failed to remove registry target: '$path'"
	}
}




## Prepare

## Main

Write-Host "`nDeaktiviere die neue Outlook App" -Fore Yellow


Write-Host "`n  Verstecke den Outlook-Toggle Button" -Fore Green
# Ausblenden des Outlook-Toggle Buttons
# https://learn.microsoft.com/de-de/microsoft-365-apps/outlook/get-started/control-install?branch=main#prevent-users-from-switching-to-new-outlook
Set-RegistryKey -RegistryPath 'HKCU:\Software\Microsoft\Office\16.0\Outlook\Options\General' `
                -PropertyName 'HideNewOutlookToggle' `
                -PropertyValue 1 `
                -PropertyType DWord



Write-Host "`n  Stoppe die Vorinstallation der neuen Outlook-App" -Fore Green
# Blockieren der neuen Outlook-Vorinstallation unter Windows
# https://learn.microsoft.com/de-de/microsoft-365-apps/outlook/get-started/control-install?branch=main#block-new-outlook-preinstall-on-windows
Get-AppxPackage 'Microsoft.OutlookForWindows' | Remove-AppxProvisionedPackage -AllUsers -Online


# Diesen Windows Orchestrator-Registrierungswert löschen
Remove-RegistryTarget -RegistryPath 'HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\WindowsUpdate\Orchestrator\UScheduler_Oobe\OutlookUpdate'



Write-Host "`n  Die neuen Outlook-App für alle Benutzer entfernen" -Fore Green
Write-Host '  (Kann später jederzeit wieder installiert werden)'
# Das neue Outlook für Windows für alle Benutzer zu entfernen
Get-AppxPackage 'Microsoft.OutlookForWindows' | Remove-AppxPackage -AllUsers



Write-Host "`n  Blockiere der neuen Outlook-Installation als Teil von Mail und Kalender" -Fore Green
# Blockieren der neuen Outlook-Installation als Teil von Mail und Kalender
# !Q https://learn.microsoft.com/de-de/microsoft-365-apps/outlook/get-started/control-install?branch=main#block-new-outlook-installation-as-part-of-mail-and-calendar-deprecation

# Verhindern, dass Ihre Benutzer das neue Outlook aus Windows Mail- und Kalenderanwendungen erwerben, 
# Diese Apps vom Gerät deinstallieren
Get-AppxProvisionedPackage -Online | Where {$_.DisplayName -match 'microsoft.windowscommunicationsapps'} | Remove-AppxProvisionedPackage -Online -PackageName {$_.PackageName}

# Mail- und Kalender-Apps für die aktuellen Benutzer zu entfernen
Try {
	Get-AppxPackage 'microsoft.windowscommunicationsapps' | Remove-AppxPackage -AllUsers
} Catch {
	Write-Host '  » Konnte die Windows Communication Apps nicht entfernen' -Fore Magenta
}


Write-Host "`n  Deaktiviere der neuen Outlook-Migration" -Fore Green
# Deaktivieren der neuen Outlook-Migration
# !Q https://learn.microsoft.com/de-de/microsoft-365-apps/outlook/get-started/control-install?branch=main#opt-out-of-new-outlook-migration

# Outlook-Migration deaktivieren
Set-RegistryKey -RegistryPath 'HKCU:\Software\Policies\Microsoft\office\16.0\outlook\preferences' `
                -PropertyName 'NewOutlookMigrationUserSetting' `
                -PropertyValue 0 `
                -PropertyType DWord



Write-Host "`n  Deaktiviere die Migrationsrichtlinie" -Fore Green

# Deaktivieren der Migrationsrichtlinie
# !Q https://learn.microsoft.com/de-de/microsoft-365-apps/outlook/manage/admin-controlled-migration-policy#hide-the-toggle-in-new-outlook-for-windows

Set-RegistryKey -RegistryPath 'HKCU:\Software\Microsoft\Office\16.0\Outlook\Options\General' `
                -PropertyName 'DoNewOutlookAutoMigration' `
                -PropertyValue 0 `
                -PropertyType DWord



Write-Host "`n  Deaktiviere das Intervall der Migrationsrichtlinie" -Fore Green

# Richtlinie zum Definieren des Intervalls zwischen Migrationsversuchen
# !Q https://learn.microsoft.com/de-de/microsoft-365-apps/outlook/manage/admin-controlled-migration-policy#hide-the-toggle-in-new-outlook-for-windows


# Wenn die Intervallrichtlinie nicht oder auf 0 festgelegt ist, 
# wird die neue Outlook-Migration nicht erneut initiiert.
Set-RegistryKey -RegistryPath 'HKCU:\Software\Microsoft\Office\16.0\Outlook\Options\General' `
                -PropertyName 'NewOutlookAutoMigrationRetryIntervals' `
                -PropertyValue 0 `
                -PropertyType DWord



Write-Host "`n  Fertig`n`n" -Fore Green
Stop-Script-Wait -WaitOnEnd 10
